import { Query } from '../types/query';
import { featureFlags } from '../utils/feature_flags';
import { Source, SourceName } from '../types/source';
import { DisplayField } from '../types/display_field';
import { formatSdl } from 'format-graphql';
import { CompileOutput } from '../types/compile_output';
import { GraphQLFilters } from './graphql_filters';
import { Dimension } from '../types/dimension';
import { Value } from '../types/value';

const REQUIRED_FIELDS = ['id', 'iid', 'title', 'webUrl', 'reference', 'state'];

export function generateCode(query: Query): CompileOutput {
  const { context } = query;
  const filters = new GraphQLFilters(query);
  let sourceName = '';

  if (context.source.name === SourceName.Epics) {
    sourceName = 'epics';
  } else if (context.source.name === SourceName.MergeRequests) {
    sourceName = 'mergeRequests';
  } else if (context.source.name === SourceName.Issues || !context.source) {
    sourceName = featureFlags.glqlWorkItems ? 'workItems' : 'issues';
  }

  const requiredFields = !query.isSubquery ? REQUIRED_FIELDS.map(renderField).join('') : '';

  let outputFields = query.fields;
  let unscopedFragment = '';
  let groupByDimensions: Dimension[] = [];
  let aggregateMetrics: DisplayField[] = [];

  if (query.groupBy?.length && query.aggregate?.length) {
    const aggregatedFields = renderFields(query.aggregate, context.source);
    const segments = query.segment();

    if (segments) {
      const aggregatedFragment = segments
        .map((query) => {
          const filters = new GraphQLFilters(query.query);

          filters.constraints['first'] = new Value.Number(0);
          delete filters.constraints['before'];
          delete filters.constraints['after'];

          return `${query.key}: ${sourceName}${filters} {${aggregatedFields}}`;
        })
        .join('');

      if (aggregatedFragment) {
        groupByDimensions = query.groupBy;
        aggregateMetrics = query.aggregate;
        unscopedFragment = aggregatedFragment;
        outputFields = query.groupBy.map((d) => d.field).concat(query.aggregate);

        context.variables = context.variables.filter(
          (v) => !['before', 'after', 'limit'].includes(v.key),
        );
      }
    }
  }

  if (!unscopedFragment) {
    const fields = renderFields(outputFields, context.source);
    const pageInfo = !query.isSubquery
      ? 'pageInfo { startCursor endCursor hasNextPage hasPreviousPage } count'
      : '';
    unscopedFragment = `${sourceName}${filters} { nodes {${requiredFields} ${fields}} ${pageInfo}}`;
  }

  const variables = context.variables.length > 0 ? `(${context.variables.join(', ')})` : '';
  let scopedFragment = '';
  if (context.project) {
    scopedFragment = `project(fullPath: "${context.project}") {${unscopedFragment}}`;
  } else if (context.group) {
    scopedFragment = `group(fullPath: "${context.group}") {${unscopedFragment}}`;
  } else {
    scopedFragment = unscopedFragment;
  }
  const output = formatSdl(`query GLQL${variables} {${scopedFragment}}`);

  return CompileOutput.success(output)
    .withVariables(context.variables)
    .withFields(outputFields)
    .withAggregationContext(groupByDimensions, aggregateMetrics);
}

function renderFields(fields: DisplayField[], source?: Source) {
  return fields.map((field) => renderWorkItemWidget(field.name, source)).join(' ');
}

function renderWorkItemWidget(field: string, source?: Source) {
  if (!featureFlags.glqlWorkItems || source?.name !== SourceName.Issues) {
    return renderField(field);
  }

  const widgetName = (() => {
    switch (field.toLowerCase()) {
      case 'assignees':
        return 'WorkItemWidgetAssignees';
      case 'labels':
        return 'WorkItemWidgetLabels';
      case 'milestone':
        return 'WorkItemWidgetMilestone';
      case 'startdate':
      case 'duedate':
        return 'WorkItemWidgetStartAndDueDate';
      case 'timeestimate':
      case 'totaltimespent':
        return 'WorkItemWidgetTimeTracking';
      case 'healthstatus':
        return 'WorkItemWidgetHealthStatus';
      case 'iteration':
        return 'WorkItemWidgetIteration';
      case 'weight':
        return 'WorkItemWidgetWeight';
      case 'progress':
        return 'WorkItemWidgetProgress';
      case 'color':
        return 'WorkItemWidgetColor';
      case 'epic':
      case 'parent':
        return 'WorkItemWidgetHierarchy';
      case 'taskcompletionstatus':
        return 'WorkItemWidgetDescription';
      case 'lastcomment':
        return 'WorkItemWidgetNotes';
      case 'status':
        return 'WorkItemWidgetStatus';
      default:
        return '';
    }
  })();

  if (!widgetName) return renderField(field);

  return `widgets { ... on ${widgetName} { type ${renderField(field)} } }`;
}

function renderField(field: string) {
  switch (field.toLowerCase()) {
    case 'labels':
      return `${field} { nodes { id title color textColor } }`;
    case 'assignees':
    case 'approvedby':
    case 'reviewers':
      return `${field} { nodes { id avatarUrl username name webUrl } }`;
    case 'author':
      return `${field} { id avatarUrl username name webUrl }`;
    case 'mergeuser':
      return `${field} { id avatarUrl username name webUrl }`;
    case 'milestone':
      return `${field} { id iid dueDate startDate title webPath }`;
    case 'epic':
    case 'parent':
      return `${
        featureFlags.glqlWorkItems ? 'parent' : 'epic'
      } { id iid reference state title webUrl }`;
    case 'color':
      return 'color textColor';
    case 'progress':
      return 'currentValue endValue progress startValue updatedAt';
    case 'iteration':
      return `${field} { id iid startDate dueDate title webUrl iterationCadence { id title } }`;
    case 'taskcompletionstatus':
      return `${field} { completedCount count }`;
    case 'lastcomment':
      return `${field}:notes(filter: ONLY_COMMENTS last: 1) { nodes { bodyHtml } }`;
    case 'project':
    case 'sourceproject':
    case 'targetproject':
      return `${field} { fullPath webUrl nameWithNamespace }`;
    case 'group':
      return `${field} { fullPath webUrl fullName }`;
    case 'workitemtype':
      if (featureFlags.glqlWorkItems) {
        return 'workItemType { name iconName }';
      } else {
        throw new Error('workItemType is not supported in non-work item queries');
      }
    case 'status':
      return `${field} { category color description iconName name }`;
    case 'meantimetomerge':
      // will be derived later during the transform step
      return 'totalTimeToMerge count';
    default:
      return `${field}\n`;
  }
}
