import { SourceAnalyzer } from '.';
import { Field, FieldName } from '../../types/field';
import { FieldType } from '../../types/field_type';
import { Operator } from '../../types/operator';
import { ReferenceType } from '../../types/reference_type';
import { RelationshipType } from '../../types/relationship_type';
import { Source, SourceName } from '../../types/source';
import { typeFieldType } from './shared';

export class MergeRequestsSourceAnalyzer extends SourceAnalyzer {
  source = new Source(SourceName.MergeRequests);

  validSortFields: Field[] = [
    new Field(FieldName.Merged),
    new Field(FieldName.Closed),
    new Field(FieldName.Title),
    new Field(FieldName.Popularity),
    new Field(FieldName.Milestone),
    new Field(FieldName.Updated),
    new Field(FieldName.Created),
  ];

  isFieldValid(field: Field) {
    field = field.dealias();

    return [
      FieldName.Type,
      FieldName.Id,
      FieldName.Label,
      FieldName.Created,
      FieldName.Updated,
      FieldName.Assignee,
      FieldName.Author,
      FieldName.Milestone,
      FieldName.Group,
      FieldName.Project,
      FieldName.Reviewer,
      FieldName.Approver,
      FieldName.Merger,
      FieldName.State,
      FieldName.Merged,
      FieldName.Deployed,
      FieldName.Draft,
      FieldName.Environment,
      FieldName.IncludeSubgroups,
      FieldName.SourceBranch,
      FieldName.TargetBranch,
      FieldName.MyReaction,
      FieldName.Subscribed,
    ].includes(field.name);
  }

  fieldType(field: Field) {
    field = field.dealias();

    switch (field.name) {
      case FieldName.Type:
        return typeFieldType();

      case FieldName.Id:
        return new FieldType.NumberLike()
          .withOps(Operator.Equal)
          .or(
            new FieldType.ListLike(RelationshipType.HasOne, new FieldType.NumberLike()).withOps(
              Operator.In,
            ),
          );

      case FieldName.State:
        return new FieldType.EnumLike(['opened', 'closed', 'merged', 'all']).withOps(
          Operator.Equal,
        );

      case FieldName.Assignee:
      case FieldName.Reviewer:
        return new FieldType.StringLike()
          .or(new FieldType.ReferenceLike(ReferenceType.User))
          .or(new FieldType.Nullable());

      case FieldName.Approver:
        return new FieldType.StringLike()
          .or(new FieldType.ReferenceLike(ReferenceType.User))
          .or(new FieldType.Nullable())
          .or(
            new FieldType.ListLike(
              RelationshipType.HasMany,
              new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.User)),
            ).withOps(Operator.Equal, Operator.NotEqual),
          );

      case FieldName.Author:
        return new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.User));

      case FieldName.Merger:
        return new FieldType.StringLike()
          .withOps(Operator.Equal)
          .or(new FieldType.ReferenceLike(ReferenceType.User).withOps(Operator.Equal));

      case FieldName.Label:
        return new FieldType.StringLike()
          .or(
            new FieldType.ListLike(
              RelationshipType.HasMany,
              new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.Label)),
            ).withOps(Operator.Equal, Operator.NotEqual),
          )
          .or(new FieldType.Nullable())
          .or(new FieldType.ReferenceLike(ReferenceType.Label));

      case FieldName.Milestone:
        return new FieldType.StringLike()
          .or(new FieldType.Nullable())
          .or(new FieldType.EnumLike(['upcoming', 'started']))
          .or(new FieldType.ReferenceLike(ReferenceType.Milestone));

      case FieldName.SourceBranch:
      case FieldName.TargetBranch:
        return new FieldType.StringLike().or(
          new FieldType.ListLike(RelationshipType.HasOne, new FieldType.StringLike()),
        );

      case FieldName.Environment:
      case FieldName.Group:
      case FieldName.Project:
        return new FieldType.StringLike().withOps(Operator.Equal);

      case FieldName.Created:
      case FieldName.Updated:
      case FieldName.Merged:
      case FieldName.Deployed:
        return new FieldType.DateLike();

      case FieldName.Draft:
      case FieldName.Subscribed:
        return new FieldType.BooleanLike();

      case FieldName.IncludeSubgroups:
        return new FieldType.BooleanLike().pairedWith(new Field(FieldName.Group));

      case FieldName.MyReaction:
        return new FieldType.StringLike();

      default:
        return new FieldType.UnknownFieldType();
    }
  }
}
