import { SourceAnalyzer } from '.';
import { getFieldMapping } from '../../field_mapping';
import { Context } from '../../types/context';
import { Field, FieldName } from '../../types/field';
import { FieldType } from '../../types/field_type';
import { Operator } from '../../types/operator';
import { ReferenceType } from '../../types/reference_type';
import { RelationshipType } from '../../types/relationship_type';
import { Source, SourceName } from '../../types/source';
import { featureFlags } from '../../utils/feature_flags';
import { typeFieldType } from './shared';

export class IssuesSourceAnalyzer extends SourceAnalyzer {
  source = new Source(SourceName.Issues);

  validSortFields: Field[] = [
    new Field(FieldName.Due),
    new Field(FieldName.Title),
    new Field(FieldName.Popularity),
    new Field(FieldName.Closed),
    new Field(FieldName.Weight),
    new Field(FieldName.Health),
    new Field(FieldName.Milestone),
    new Field(FieldName.Updated),
    new Field(FieldName.Created),
  ];

  constructor() {
    super();

    if (featureFlags.glqlWorkItems) {
      this.validSortFields.push(new Field(FieldName.Start));
    }
  }

  isFieldValid(field: Field) {
    field = field.dealias();

    if (field.name === FieldName.Parent && featureFlags.glqlWorkItems) return true;

    return (
      [
        FieldName.Type,
        FieldName.Id,
        FieldName.State,
        FieldName.Status,
        FieldName.Label,
        FieldName.Epic,
        FieldName.Created,
        FieldName.Updated,
        FieldName.Assignee,
        FieldName.Author,
        FieldName.Milestone,
        FieldName.Weight,
        FieldName.Group,
        FieldName.Project,
        FieldName.Confidential,
        FieldName.Health,
        FieldName.Closed,
        FieldName.Iteration,
        FieldName.Cadence,
        FieldName.Due,
        FieldName.IncludeSubgroups,
        FieldName.MyReaction,
        FieldName.Subscribed,
      ].includes(field.name) || field instanceof Field.Custom
    );
  }

  fieldType(field: Field) {
    field = field.dealias();

    if (field instanceof Field.Custom) {
      return new FieldType.StringLike()
        .withOps(Operator.Equal)
        .or(
          new FieldType.ListLike(RelationshipType.HasMany, new FieldType.StringLike()).withOps(
            Operator.Equal,
          ),
        );
    }

    switch (field.name) {
      case FieldName.Type:
        return typeFieldType();

      case FieldName.Id:
        return new FieldType.NumberLike()
          .withOps(Operator.Equal)
          .or(
            new FieldType.ListLike(RelationshipType.HasOne, new FieldType.NumberLike()).withOps(
              Operator.In,
            ),
          );

      case FieldName.State:
        return new FieldType.EnumLike(['opened', 'closed', 'all']).withOps(Operator.Equal);

      case FieldName.Status:
        return new FieldType.StringLike().withOps(Operator.Equal);

      case FieldName.Assignee:
        return new FieldType.StringLike()
          .or(new FieldType.ReferenceLike(ReferenceType.User))
          .or(
            new FieldType.ListLike(
              RelationshipType.HasMany,
              new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.User)),
            ),
          )
          .or(new FieldType.NumberLike())
          .or(new FieldType.Nullable());

      case FieldName.Author:
        return (
          new FieldType.StringLike()
            .or(new FieldType.ReferenceLike(ReferenceType.User))
            // hack: Technically, `author` is a HasOne field, but the backend supports it
            // differently from other HasOne fields like milestone and iteration.
            .or(
              new FieldType.ListLike(
                RelationshipType.HasMany,
                new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.User)),
              ).withOps(Operator.In, Operator.NotEqual),
            )
            .or(new FieldType.Nullable())
        );

      case FieldName.Label:
        return new FieldType.StringLike()
          .or(
            new FieldType.ListLike(
              RelationshipType.HasMany,
              new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.Label)),
            ),
          )
          .or(new FieldType.Nullable())
          .or(new FieldType.ReferenceLike(ReferenceType.Label));

      case FieldName.Cadence:
        return new FieldType.StringLike()
          .withOps(Operator.Equal)
          .or(
            new FieldType.ListLike(RelationshipType.HasOne, new FieldType.NumberLike()).withOps(
              Operator.In,
            ),
          )
          .or(new FieldType.NumberLike().withOps(Operator.Equal))
          .or(new FieldType.Nullable());

      case FieldName.Milestone:
        return new FieldType.StringLike()
          .or(
            new FieldType.ListLike(
              RelationshipType.HasOne,
              new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.Milestone)),
            ),
          )
          .or(new FieldType.Nullable())
          .or(new FieldType.EnumLike(['upcoming', 'started']))
          .or(new FieldType.ReferenceLike(ReferenceType.Milestone));

      case FieldName.Iteration:
        return new FieldType.StringLike()
          .or(
            new FieldType.ListLike(
              RelationshipType.HasOne,
              new FieldType.NumberLike()
                .or(new FieldType.StringLike())
                .or(new FieldType.ReferenceLike(ReferenceType.Iteration)),
            ),
          )
          .or(new FieldType.NumberLike())
          .or(new FieldType.Nullable())
          .or(new FieldType.EnumLike(['current']))
          .or(new FieldType.ReferenceLike(ReferenceType.Iteration));

      case FieldName.Epic:
      case FieldName.Parent: {
        // Use the new field mapping abstraction for Epic field
        const context = new Context();
        context.source = new Source(SourceName.Issues);
        const mapping = getFieldMapping(field, context);
        if (mapping) {
          return mapping.fieldType;
        }
        return new FieldType.UnknownFieldType();
      }

      case FieldName.Group:
      case FieldName.Project:
        return new FieldType.StringLike().withOps(Operator.Equal);

      case FieldName.Closed:
      case FieldName.Created:
      case FieldName.Updated:
      case FieldName.Due:
        return new FieldType.DateLike();

      case FieldName.Weight:
        return new FieldType.NumberLike().or(new FieldType.Nullable());

      case FieldName.Subscribed:
      case FieldName.Confidential:
        return new FieldType.BooleanLike();

      case FieldName.IncludeSubgroups:
        return new FieldType.BooleanLike().pairedWith(new Field(FieldName.Group));

      case FieldName.Health:
        return new FieldType.StringEnumLike(['on track', 'needs attention', 'at risk']).or(
          new FieldType.Nullable(),
        );

      case FieldName.MyReaction:
        return new FieldType.StringLike();

      default:
        return new FieldType.UnknownFieldType();
    }
  }
}
