import { SourceAnalyzer } from '.';
import { Field, FieldName } from '../../types/field';
import { FieldType } from '../../types/field_type';
import { Operator } from '../../types/operator';
import { ReferenceType } from '../../types/reference_type';
import { RelationshipType } from '../../types/relationship_type';
import { Source, SourceName } from '../../types/source';
import { typeFieldType } from './shared';

export class EpicsSourceAnalyzer extends SourceAnalyzer {
  source = new Source(SourceName.Epics);

  validSortFields: Field[] = [
    new Field(FieldName.Start),
    new Field(FieldName.Due),
    new Field(FieldName.Title),
    new Field(FieldName.Created),
    new Field(FieldName.Updated),
  ];

  isFieldValid(field: Field) {
    field = field.dealias();

    return [
      FieldName.Type,
      FieldName.Id,
      FieldName.State,
      FieldName.Author,
      FieldName.Label,
      FieldName.Milestone,
      FieldName.Group,
      FieldName.Created,
      FieldName.Updated,
      FieldName.Confidential,
      FieldName.IncludeSubgroups,
    ].includes(field.name);
  }

  fieldType(field: Field) {
    field = field.dealias();

    switch (field.name) {
      case FieldName.Type:
        return typeFieldType();

      case FieldName.Id:
        return new FieldType.NumberLike()
          .withOps(Operator.Equal)
          .or(
            new FieldType.ListLike(
              RelationshipType.HasOne,
              new FieldType.NumberLike().withOps(Operator.In),
            ),
          );

      case FieldName.State:
        return new FieldType.EnumLike(['opened', 'closed', 'all']).withOps(Operator.Equal);

      case FieldName.Author:
        return (
          new FieldType.StringLike()
            .or(new FieldType.ReferenceLike(ReferenceType.User))
            // hack: Technically, `author` is a HasOne field, but the backend supports it
            // differently from other HasOne fields like milestone and iteration.
            .or(
              new FieldType.ListLike(
                RelationshipType.HasMany,
                new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.User)),
              ).withOps(Operator.In),
            )
        );

      case FieldName.Label:
        return new FieldType.StringLike()
          .or(
            new FieldType.ListLike(
              RelationshipType.HasMany,
              new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.Label)),
            ),
          )
          .or(new FieldType.ReferenceLike(ReferenceType.Label));

      case FieldName.Milestone:
        return new FieldType.StringLike()
          .or(
            new FieldType.ListLike(
              RelationshipType.HasOne,
              new FieldType.StringLike().or(new FieldType.ReferenceLike(ReferenceType.Milestone)),
            ),
          )
          .or(new FieldType.EnumLike(['upcoming', 'started']))
          .or(new FieldType.ReferenceLike(ReferenceType.Milestone));

      case FieldName.Group:
        return new FieldType.StringLike().withOps(Operator.Equal);

      case FieldName.Created:
      case FieldName.Updated:
        return new FieldType.DateLike();

      case FieldName.Confidential:
        return new FieldType.BooleanLike();

      case FieldName.IncludeSubgroups:
        return new FieldType.BooleanLike().pairedWith(new Field(FieldName.Group));

      default:
        return new FieldType.UnknownFieldType();
    }
  }
}
