import { Field } from '../types/field';
import { Query } from '../types/query';
import { Source } from '../types/source';
import { ExpressionAnalyzer } from './expression';
import { FieldAnalyzer } from './field';
import { OperatorType } from '../types/operator';
import { GlqlAnalyzeError } from '../errors';
import { SortAnalyzer } from './sort';

export class QueryAnalyzer {
  constructor(
    public readonly query: Query,
    public readonly source: Source,
  ) {}

  analyze() {
    this.analyzeExpressions();
    this.analyzeSort();
  }

  analyzeExpressions() {
    const operatorFieldMap: Partial<Record<OperatorType, Field[]>> = {};
    const fields = this.query.expressions.map((expr) => expr.field.dealias());

    for (const expr of this.query.expressions) {
      new ExpressionAnalyzer(expr, this.source).analyze();
      new FieldAnalyzer(expr.field, this.source).analyzePairedFields(fields);

      const fieldType = this.source.analyzer.fieldType(expr.field);

      if (fieldType.isListLike && expr.operator.type !== OperatorType.In) continue;

      if (operatorFieldMap[expr.operator.type]?.some((field) => field.eq(expr.field)))
        throw new GlqlAnalyzeError.DuplicateOperatorForField(expr.operator, expr.field);

      (operatorFieldMap[expr.operator.type] ||= []).push(expr.field);
    }
  }

  analyzeSort() {
    if (this.query.sort) new SortAnalyzer(this.query.sort, this.source).analyze();
  }
}
