import { Field } from '../types/field';
import { FieldType } from '../types/field_type';
import { Operator } from '../types/operator';
import { RelationshipType } from '../types/relationship_type';
import { Value } from '../types/value';

export class FieldTypeAnalyzer {
  constructor(public readonly fieldType: FieldType) {}

  analyze(value: Value): boolean {
    if (this.fieldType instanceof FieldType.Nullable) {
      return (
        (value instanceof Value.Token && ['none', 'any'].includes(value.value.toLowerCase())) ||
        value instanceof Value.Null ||
        (value instanceof Value.List && value.value.length === 0)
      );
    }

    if (this.fieldType instanceof FieldType.StringLike) {
      return value instanceof Value.Quoted;
    }

    if (this.fieldType instanceof FieldType.ReferenceLike) {
      return (
        value instanceof Value.Reference && value.referenceType.eq(this.fieldType.referenceType)
      );
    }

    if (this.fieldType instanceof FieldType.NumberLike) {
      return value instanceof Value.Number;
    }

    if (this.fieldType instanceof FieldType.DateLike) {
      if (value instanceof Value.Date || value instanceof Value.Quoted) {
        return !isNaN(new Date(value.value).getTime());
      }
      return false;
    }

    if (this.fieldType instanceof FieldType.BooleanLike) {
      return (
        value instanceof Value.Null ||
        value instanceof Value.Bool ||
        (value instanceof Value.Number && (value.value === 0 || value.value === 1)) ||
        (value instanceof Value.Quoted &&
          ['true', 'false', '0', '1'].includes(value.value.toLowerCase()))
      );
    }

    if (this.fieldType instanceof FieldType.ListLike) {
      return (
        value instanceof Value.List &&
        value.value.every((v) =>
          (this.fieldType as FieldType.ListLike).fieldType.analyzer.analyze(v),
        )
      );
    }

    if (
      this.fieldType instanceof FieldType.EnumLike ||
      this.fieldType instanceof FieldType.StringEnumLike
    ) {
      return (
        (value instanceof Value.Token || value instanceof Value.Quoted) &&
        this.fieldType.values.map((v) => v.toLowerCase()).includes(value.value.toLowerCase())
      );
    }

    if (
      this.fieldType instanceof FieldType.WithOperators ||
      this.fieldType instanceof FieldType.PairedWith
    ) {
      return this.fieldType.fieldType.analyzer.analyze(value);
    }

    if (this.fieldType instanceof FieldType.Multiple) {
      return this.fieldType.fieldTypes.some((f) => f.analyzer.analyze(value));
    }

    return false;
  }

  get pairableFields(): Field[] {
    if (this.fieldType instanceof FieldType.Multiple) {
      return this.fieldType.fieldTypes.flatMap((f) => f.analyzer.pairableFields);
    }

    if (this.fieldType instanceof FieldType.WithOperators) {
      return this.fieldType.fieldType.analyzer.pairableFields;
    }

    if (this.fieldType instanceof FieldType.PairedWith) {
      return this.fieldType.fields;
    }

    return [];
  }

  get operators(): Set<Operator> {
    if (
      this.fieldType instanceof FieldType.Nullable ||
      this.fieldType instanceof FieldType.StringLike ||
      this.fieldType instanceof FieldType.NumberLike ||
      this.fieldType instanceof FieldType.BooleanLike ||
      this.fieldType instanceof FieldType.EnumLike ||
      this.fieldType instanceof FieldType.StringEnumLike ||
      this.fieldType instanceof FieldType.ReferenceLike
    ) {
      return new Set([Operator.Equal, Operator.NotEqual]);
    }

    if (this.fieldType instanceof FieldType.WithOperators) {
      return new Set(this.fieldType.operators);
    }

    if (this.fieldType instanceof FieldType.DateLike) {
      return new Set([
        Operator.Equal,
        Operator.GreaterThan,
        Operator.GreaterThanEquals,
        Operator.LessThan,
        Operator.LessThanEquals,
      ]);
    }

    if (this.fieldType instanceof FieldType.ListLike) {
      if (this.fieldType.relationshipType === RelationshipType.HasOne) {
        return new Set([Operator.In, Operator.NotEqual]);
      }

      if (this.fieldType.relationshipType === RelationshipType.HasMany) {
        return new Set([Operator.In, Operator.Equal, Operator.NotEqual]);
      }
    }

    if (this.fieldType instanceof FieldType.PairedWith) {
      return this.fieldType.fieldType.analyzer.operators;
    }

    return new Set();
  }
}
