import { GlqlAnalyzeError } from '../errors';
import { Field } from '../types/field';
import { Operator } from '../types/operator';
import { Source } from '../types/source';
import { Value } from '../types/value';
import { ValueAnalyzer } from './value';

export class FieldAnalyzer {
  constructor(
    public readonly field: Field,
    public readonly source: Source,
  ) {}

  get operators(): Set<Operator> {
    return new Set(
      this.source.analyzer.allFieldTypes(this.field).flatMap((f) => [...f.analyzer.operators]),
    );
  }

  operatorsForValue(value: Value): Set<Operator> {
    return (
      new ValueAnalyzer(value, this.source).fieldType(this.field)?.analyzer.operators || new Set()
    );
  }

  analyzeField() {
    if (this.field instanceof Field.Unknown)
      throw new GlqlAnalyzeError.UnrecognizedField(this.field);
  }

  analyzeFieldForSource() {
    return this.source.analyzer.analyzeField(this.field);
  }

  analyzePairedFields(fields: Field[]) {
    const field = this.field.dealias();
    const { pairableFields } = this.source.analyzer.fieldType(field).analyzer;

    if (pairableFields.length > 0 && !fields.some((f) => pairableFields.some((p) => p.eq(f))))
      throw new GlqlAnalyzeError.IncorrectFieldPairing(this.field, pairableFields);
  }

  analyzeOperator(op: Operator) {
    const ops = [...this.operators];
    const supportedOperators = [...ops];
    if (!ops.some((o) => o.eq(op)))
      throw new GlqlAnalyzeError.UnsupportedOperatorForField(this.field, op, supportedOperators);
  }

  analyzeOperatorForValue(op: Operator, value: Value) {
    const ops = [...this.operatorsForValue(value)];

    if (!ops.some((o) => o.eq(op)))
      throw new GlqlAnalyzeError.UnsupportedOperatorValueForField(this.field, op, value, ops);
  }

  analyzeValue(value: Value) {
    const types = this.source.analyzer.allFieldTypes(this.field);

    if (!types.some((t) => t.analyzer.analyze(value)))
      throw new GlqlAnalyzeError.UnsupportedValueForField(this.field, value, types);
  }

  analyze(op: Operator, value: Value) {
    this.analyzeField();
    this.analyzeFieldForSource();
    this.analyzeOperator(op);
    this.analyzeValue(value);
    this.analyzeOperatorForValue(op, value);
  }
}
