# frozen_string_literal: true

require "json"

RSpec.describe Glql do # rubocop:disable RSpec/SpecFilePathFormat
  it "has a version number" do
    expect(Glql::VERSION).not_to be_nil
  end

  it "compiles GLQL" do
    query = 'type = Issue and label = "bug" and author = currentUser()'
    context = {
      username: "johndoe",
      fields: "title, labels, author",
      group: "gitlab-org"
    }
    result = described_class.compile(query, context)

    expect(result["success"]).to be(true)
    expect(result["fields"]).to eq([{ "key" => "title", "label" => "Title", "name" => "title" },
      { "key" => "labels", "label" => "Labels", "name" => "labels" },
      { "key" => "author", "label" => "Author", "name" => "author" }])

    expected_output = <<~GRAPHQL
      query GLQL($before: String, $after: String, $limit: Int) {
        group(fullPath: "gitlab-org") {
          issues(types: ISSUE, labelName: "bug", authorUsername: "johndoe", before: $before, after: $after, first: $limit, includeSubgroups: true) {
            nodes {
              id
              iid
              title
              webUrl
              reference
              state
              title
              labels {
                nodes {
                  id
                  title
                  color
                  textColor
                }
              }
              author {
                id
                avatarUrl
                username
                name
                webUrl
              }
            }
            pageInfo {
              startCursor
              endCursor
              hasNextPage
              hasPreviousPage
            }
            count
          }
        }
      }
    GRAPHQL

    expect(result["output"]).to eq(expected_output)
  end

  it "transforms GraphQL response data" do
    data = {
      group: {
        issues: {
          nodes: [
            {
              id: "gid://gitlab/Issue/650",
              iid: "42",
              title: "epics test",
              webUrl: "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/issues/42",
              reference: "#42",
              state: "opened",
              type: "ISSUE",
              updatedAt: "2025-08-04T09:32:02Z",
              epic: {
                id: "gid://gitlab/Epic/8",
                iid: "3",
                reference: "&3",
                state: "opened",
                title: "Consectetur nisi unde dolore cupiditate laborum et.",
                webUrl: "http://127.0.0.1:3000/groups/gitlab-org/-/epics/3",
                __typename: "Epic"
              },
              __typename: "Issue"
            }
          ],
          pageInfo: {
            startCursor: "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9",
            endCursor: "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9",
            hasNextPage: true,
            hasPreviousPage: false,
            __typename: "PageInfo"
          },
          count: 4,
          __typename: "IssueConnection"
        },
        __typename: "Group"
      }
    }

    context = { fields: "type, title, updated, epic" }
    result = described_class.transform(data, context)

    expect(result["fields"]).to eq([
      { "key" => "type", "label" => "Type", "name" => "type" },
      { "key" => "title", "label" => "Title", "name" => "title" },
      { "key" => "updated", "label" => "Updated", "name" => "updatedAt" },
      { "key" => "epic", "label" => "Epic", "name" => "epic" }
    ])

    expect(result["data"]).to eq({
      "__typename" => "IssueConnection",
      "count" => 4,
      "nodes" => [
        {
          "__typename" => "Issue",
          "epic" => {
            "__typename" => "Epic",
            "id" => "gid://gitlab/Epic/8",
            "iid" => "3",
            "reference" => "&3",
            "state" => "opened",
            "title" => "Consectetur nisi unde dolore cupiditate laborum et.",
            "webUrl" => "http://127.0.0.1:3000/groups/gitlab-org/-/epics/3"
          },
          "id" => "gid://gitlab/Issue/650",
          "iid" => "42",
          "reference" => "#42",
          "state" => "opened",
          "title" => "epics test",
          "type" => "ISSUE",
          "updated" => "2025-08-04T09:32:02Z",
          "updatedAt" => "2025-08-04T09:32:02Z",
          "webUrl" => "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/issues/42"
        }
      ],
      "pageInfo" => {
        "__typename" => "PageInfo",
        "endCursor" => "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9",
        "hasNextPage" => true,
        "hasPreviousPage" => false,
        "startCursor" => "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9"
      }
    })
  end
end
