# glql_rb

Ruby bindings for the GitLab Query Language (GLQL) compiler. This gem provides a native Ruby interface to compile GLQL queries into GraphQL, powered by Rust through native extensions.

## Installation

Add to your Gemfile:

```ruby
gem 'gitlab_query_language'
```

Then run:

```bash
bundle install
```

Or install directly:

```bash
gem install gitlab_query_language
```

## Usage

The gem provides a simple interface to compile GLQL queries into GraphQL:

```ruby
require 'gitlab_query_language'

# Compile a GLQL query
query = 'label != "backend" and author = currentUser() and weight = 1'
result = Glql.compile(query, {group: 'gitlab-org', username: 'johnhope', fields: 'title'})
puts result

{"fields" => [{"key" => "title", "label" => "Title", "name" => "title"}],
 "output" =>
  "query GLQL($before: String, $after: String, $limit: Int) {\n  group(fullPath: \"gitlab-org\") {\n    issues(authorUsername: \"johnhope\", weight: \"1\", not: {labelName: \"backend\"}, before: $before, after: $after, first: $limit, includeSubgroups: true) {\n      nodes {\n        id\n        iid\n        title\n        webUrl\n        reference\n        state\n        title\n      }\n      pageInfo {\n        startCursor\n        endCursor\n        hasNextPage\n        hasPreviousPage\n      }\n      count\n    }\n  }\n}\n",
 "success" => true,
 "variables" =>
  {"after" => {"type" => "String", "value" => nil},
   "before" => {"type" => "String", "value" => nil},
   "limit" => {"type" => "Int", "value" => nil}}}
```

## Development

### Setup

After checking out the repo, install dependencies:

```bash
mise install

cd glql_rb

bin/setup
```

### Building the Extension

The Rust extension is automatically compiled when you install the gem. To manually compile:

```bash
bundle exec rake compile
```

### Running Tests

Run the test suite:

```bash
bundle exec rake spec
```

### Interactive Console

Experiment with the gem in an interactive console:

```bash
bundle exec rake console
```

### Installing Locally

To install the gem onto your local machine:

```bash
bundle exec rake install
```

## Release Process

The gem is released automatically using GitLab's [gem-release component](https://gitlab.com/gitlab-org/components/gem-release). When a version bump is merged to the main branch:

1. The component builds both the regular gem and pre-compiled native gems for multiple platforms (Linux, macOS, with different architectures)
2. Runs smoke tests to verify the gem installs and functions correctly
3. Creates a GitLab release with changelog notes
4. Publishes all gem variants to RubyGems.org

The gem uses tag-based versioning to depend on specific GLQL core releases (e.g., `tag = "v0.21.0"` in `Cargo.toml`). This ensures explicit version coupling and reproducible builds. When a new GLQL version is released, the automated release script creates a merge request that:

- Updates the gem version in `glql_rb/lib/gitlab_query_language/version.rb`
- Updates the GLQL dependency tag in `glql_rb/ext/gitlab_query_language/Cargo.toml`
- Updates lock files (`Gemfile.lock`, `Cargo.lock`)

The release is triggered automatically when these changes are merged to main.

**Manual Testing:** To manually trigger the gem build jobs before merging (e.g., for testing), include the word "RELEASE" in the merge request title. This will run the build and smoke test jobs without publishing to RubyGems.

## Contributing

Bug reports and pull requests are welcome at https://gitlab.com/gitlab-org/glql.

To contribute:

1. Clone the repository
2. Create a feature branch (`git checkout -b feature/my-feature`)
3. Make your changes and ensure tests pass
4. Commit your changes (`git commit -am 'Add new feature'`)
5. Push to the branch (`git push origin feature/my-feature`)
6. Open a merge request

## Additional Resources

- [GLQL User Documentation](https://docs.gitlab.com/user/glql/)
- [GLQL Architecture Design Document](https://handbook.gitlab.com/handbook/engineering/architecture/design-documents/glql/)
- [Main GLQL Repository](https://gitlab.com/gitlab-org/glql)
