# GitLab Query Language

![Latest Release](https://img.shields.io/npm/v/@gitlab/query-language-rust) ![Rust Coverage](https://gitlab.com/gitlab-org/glql/badges/main/coverage.svg?job=coverage) ![TypeScript Coverage](https://gitlab.com/gitlab-org/glql/badges/main/coverage.svg?job=coverage-ts) ![Pipeline Status](https://gitlab.com/gitlab-org/glql/badges/main/pipeline.svg)

This is the Rust version of GLQL compiler library, [originally written in Haskell](https://gitlab.com/gitlab-org/gitlab-query-language/glql-haskell).

Goals of this library include:

- Transpiling GLQL queries, including the YAML frontmatter block to a GraphQL response. Currently, the queries are transpiled using a WASM build of the Haskell version and YAML is transformed on the frontend using JS-YAML.
- Transforming the result of a GraphQL call using transformer functions.

## Installation

GitLabQueryLanguage is an npm module that provides a simple interface for converting `glql`
queries into other formats.

It wraps the WebAssembly build of the `glql` compiler, written in Rust.

### Usage

Install the module:

```sh
npm install --save @gitlab/query-language-rust
```

Import and use the module:

```html
<script type="module">
  import { glql } from '@gitlab/query-language-rust';

  let query = 'label != "backend" and author = currentUser() and weight = 1 and updated > today()';

  let graphql = await glql.compile(query, {
    group: 'gitlab-org',
    username: 'johnhope',
  });

  console.log(graphql);
</script>
```

### Running the Example

Clone the repo, then install the dev dependencies, build the WASM target, and serve the `npm/` directory:

```sh
npm install
cargo install wasm-pack
rustup target add wasm32-unknown-unknown
wasm-pack build --target web
npx wds -r npm/
```

> [!tip] Tip
>
> You can switch between legacy issues query and work items query by setting the
> `glqlWorkItems` feature flag in `index.html` to `true` or `false`.

## Development

Ensure you have the following installed:

- [Rust](https://www.rust-lang.org/learn/get-started) (version 1.73.0)
- [Cargo](https://doc.rust-lang.org/cargo/getting-started/installation.html) (usually comes with Rust)
- [Lefthook](https://evilmartians.com/chronicles/introducing-lefthook) (for managing git hooks)
- [Nodejs](https://nodejs.org/en) (for running the GLQL -> GraphQL compiler)

To verify your Rust and Cargo installation, run:

```bash
rustc --version
cargo --version
```

If using VSCode, consider installing the [`rust-analyzer`](https://marketplace.visualstudio.com/items?itemName=rust-lang.rust-analyzer) extension.

### Setting Up Lefthook

Lefthook is used to automate certain checks before pushing code to the repository, such as running tests, ensuring code is formatted, and checking for lint warnings.

To install Lefthook globally on MacOS, run:

```bash
brew install lefthook
```

For other platforms or methods to install, see [installation docs](https://github.com/evilmartians/lefthook/blob/master/docs/install.md) for Lefthook.

Once `lefthook` is installed, enable it in your project by running the following command in the root of your repository:

```bash
lefthook install
```

This command will set up Lefthook to manage git hooks for pre-push checks as defined in the `lefthook.yml` file.

### Pre-push Hooks

The following checks will run automatically before you push your code:

- **Tests**: Ensures all tests pass using `cargo test`.
- **Clippy**: Runs `cargo clippy` to check for lint warnings and ensures no warnings are present.
- **Format**: Ensures code is formatted properly using `cargo fmt --check`.
- **Check**: Runs `cargo check` to verify that the code compiles without errors.

## Running Tests

### Rust Tests

To run the Rust tests manually, use the following command:

```bash
cargo test
```

This will build and run all tests defined in your Rust project.

### TypeScript Tests

To run the TypeScript tests, use:

```bash
npm test
```

For continuous integration testing:

```bash
npm run test:ci
```

### Code Coverage

Coverage reports are automatically generated in CI/CD and published to GitLab Pages at:

- Rust coverage: `https://gitlab-org.gitlab.io/glql/coverage/`
- TypeScript coverage: `https://gitlab-org.gitlab.io/glql/coverage-ts/`

### Formatting Code

To ensure your code is properly formatted using `rustfmt`, run:

```bash
cargo fmt -- --check
```

This command checks whether your code adheres to the formatting rules but does not modify any files. To automatically format your code, run:

```bash
cargo fmt
```

### Linting with Clippy

To lint your code and catch common mistakes or inefficiencies, run:

```bash
cargo clippy -- -D warnings
```

This will treat all warnings as errors and prevent the code from building if any warnings are found.

### End-to-End Testing

When making changes to the GLQL Rust codebase, you can follow these steps to test your modifications end-to-end in [GitLab GDK](https://gitlab.com/gitlab-org/gitlab-development-kit):

**In glql-rust directory:**

1. Build the Rust code to WASM `wasm-pack build --target web --out-dir pkg`
2. Build the npm package `npm run build`
3. Copy the full path to the build `echo "$(pwd)/npm/dist/main.js"`

**In Gitlab GDK directory:**

Update the import paths in both `app/assets/javascripts/glql/core/parser.js` and `app/assets/javascripts/glql/core/transformer.js`:

```js
// change from
import { glql } from '@gitlab/query-language-rust';

// to
import { glql } from 'file:///path/from/step/3/above';
```

Now you can test your changes in any markdown field embedding there your GLQL query, for example:

````markdown
```glql
display: table
fields: title, labels, author
query: group = "gitlab-org" and label = "group::knowlegde" and user = currentUser()
```
````

**Note:** Every time you make a change in `glql-rust`, you must rebuild both the WASM and npm packages for the changes to take effect.

## Contributing

1. If you don't already have access, start onboarding by [requesting access to the community forks](https://gitlab.com/gitlab-community/meta#request-access-to-community-forks).
1. Clone the [community fork repository](https://gitlab.com/gitlab-community/gitlab-org/glql).
1. Create a new feature branch (`git checkout -b feature/my-feature`).
1. Commit your changes (`git commit -am 'Add new feature'`).
1. Push to the branch (`git push origin feature/my-feature`).
1. Open an MR.

## Release process

> [!important] Pre-requisites
> You need Developer permissions in the project.

On the `main` branch, run any of the below pipeline jobs to create a
release:

- `release-patch`: Bump patch version and publish to NPM.
- `release-minor`: Bump minor version and publish to NPM.
- `release-major`: Bump major version and publish to NPM.

The release-\* job will additionally create a release MR to also bump the Ruby gem version and dependency. This currently requires manual approval to trigger the gem release.

See the gem [glql_rb/README.md](README) for more details on the release process.

## Additional Reading

### Learning Rust

- [The Rust book](https://doc.rust-lang.org/book/)
- [Rust by Example](https://doc.rust-lang.org/rust-by-example/)

## Learning more about GLQL

- [Architecture Design Document](https://handbook.gitlab.com/handbook/engineering/architecture/design-documents/glql/)
- [User facing docs](https://docs.gitlab.com/ee/user/glql/)
