# frozen_string_literal: true

require 'spec_helper'

RSpec.describe 'Merge request shortcuts', :js, feature_category: :code_review_workflow do
  let_it_be(:user) { create(:user) }
  let_it_be(:project) { create(:project, :public, :repository) }
  let(:merge_request) { create(:merge_request, source_project: project) }
  let(:note_text) { 'I got this!' }

  before_all do
    project.add_developer(user)
  end

  describe 'pressing "r"', quarantine: 'https://gitlab.com/gitlab-org/quality/test-failure-issues/-/issues/4085' do
    before do
      create(:note, noteable: merge_request, project: project, note: note_text)
      sign_in(user)
      visit project_merge_request_path(project, merge_request)
      wait_for_requests
    end

    it 'focuses main comment field by default' do
      find('body').native.send_key('r')

      expect(page).to have_selector('.js-main-target-form .js-gfm-input:focus')
    end

    it 'quotes the selected text in main comment form' do
      select_element('#notes-list .note-comment:first-child .note-text')
      find('body').native.send_key('r')

      page.within('.js-main-target-form') do
        expect(page).to have_field('Write a comment or drag your files here…', with: "> #{note_text}\n\n")
      end
    end

    it 'quotes the selected text in the discussion reply form' do
      find('#notes-list .note:first-child .js-reply-button').click
      select_element('#notes-list .note-comment:first-child .note-text')
      find('body').native.send_key('r')

      page.within('.notes .discussion-reply-holder') do
        expect(page).to have_field('Write a comment or drag your files here…', with: "> #{note_text}\n\n")
      end
    end
  end

  describe 'pressing "a"', quarantine: 'https://gitlab.com/gitlab-org/quality/test-failure-issues/-/issues/6978' do
    before do
      sign_in(user)
      visit project_merge_request_path(project, merge_request)
      wait_for_requests
    end

    it "opens assignee dropdown for editing",
      quarantine: 'https://gitlab.com/gitlab-org/quality/test-failure-issues/-/issues/6979' do
      find('body').native.send_key('a')

      expect(find('.block.assignee')).to have_selector('.dropdown-menu-user')
    end
  end

  describe 'pressing "m"', quarantine: 'https://gitlab.com/gitlab-org/quality/test-failure-issues/-/issues/3815' do
    before do
      sign_in(user)
      visit project_merge_request_path(project, merge_request)
      wait_for_requests
    end

    it "opens milestones dropdown for editing",
      quarantine: 'https://gitlab.com/gitlab-org/quality/test-failure-issues/-/issues/6980' do
      find('body').native.send_key('m')

      expect(find_by_testid('milestone-edit')).to have_selector('.gl-dropdown-inner')
    end
  end

  describe 'pressing "l"' do
    before do
      sign_in(user)
      visit project_merge_request_path(project, merge_request)
      wait_for_requests
    end

    it "opens labels dropdown for editing" do
      find('body').native.send_key('l')

      expect(find('.js-labels-block')).to have_selector('[data-testid="labels-select-dropdown-contents"]')
    end
  end
end
