# frozen_string_literal: true

require 'spec_helper'

RSpec.describe ResourceAccessTokens::CreateService, feature_category: :system_access do
  subject { described_class.new(user, resource, params).execute }

  let_it_be(:organization) { create(:organization) }
  let_it_be(:user) { create(:user, organization: organization) }
  let_it_be(:project) { create(:project, :private, organization: organization) }
  let_it_be(:group) { create(:group, :private, organization: organization) }
  let_it_be(:params) { { expires_at: Date.today + 1.month } }
  let_it_be(:max_pat_access_token_lifetime) do
    PersonalAccessToken::MAX_PERSONAL_ACCESS_TOKEN_LIFETIME_IN_DAYS.days.from_now.to_date.freeze
  end

  before do
    stub_config_setting(host: 'example.com')
  end

  describe '#execute' do
    shared_examples 'token creation fails' do
      let_it_be(:resource) { create(:project, organization: organization) }

      it 'does not add the project bot as a member' do
        expect { subject }.not_to change { resource.members.count }
      end

      it 'immediately destroys the bot user if one was created', :sidekiq_inline do
        expect { subject }.not_to change { User.bots.count }
      end
    end

    shared_examples 'deletes failed project bot' do
      it 'calls DeleteUserWorker for the project bot' do
        expect_next_instance_of(User) do |project_bot|
          project_bot.id = User.maximum(:id) + 1
          expect(DeleteUserWorker).to receive(:perform_async).with(
            user.id, project_bot.id,
            hard_delete: true, skip_authorization: true, reason_for_deletion: "Access token creation failed"
          ).and_call_original
        end

        subject
      end
    end

    shared_examples 'correct error message' do
      it 'returns correct error message' do
        expect(subject.error?).to be true
        expect(subject.errors).to include(error_message)
      end
    end

    shared_examples 'allows creation of bot with valid params' do
      it { expect { subject }.to change { User.count }.by(1) }

      shared_examples_for 'creates a user with the correct attributes' do
        it 'creates a user' do
          response = subject
          access_token = response.payload[:access_token]
          namespace = resource.is_a?(Group) ? resource : resource.project_namespace
          root = resource.root_ancestor
          group_id = root.is_a?(Group) ? root.id : nil
          access_token.user.reload

          expect(access_token.user.confirmed?).to eq(true)
          expect(access_token.group_id).to eq(group_id)
          expect(access_token.user_type).to eq("project_bot")
          expect(access_token.user.user_type).to eq("project_bot")
          expect(access_token.user.created_by_id).to eq(user.id)
          expect(access_token.user.namespace.organization.id).to eq(resource.organization.id)
          expect(access_token.organization.id).to eq(resource.organization.id)
          expect(access_token.user.bot_namespace).to eq(namespace)
        end
      end

      context 'when created by an admin' do
        let(:user) { create(:admin) }

        context 'when admin mode is enabled', :enable_admin_mode do
          it_behaves_like 'creates a user with the correct attributes'
        end

        context 'when admin mode is disabled' do
          it 'returns error' do
            response = subject

            expect(response.error?).to be true
          end
        end
      end

      context 'when created by a non-admin' do
        it_behaves_like 'creates a user with the correct attributes'
      end

      context 'bot name' do
        context 'when no name is passed' do
          it 'uses default name' do
            response = subject
            access_token = response.payload[:access_token]

            expect(access_token.user.name).to eq("#{resource.name.to_s.humanize} bot")
          end
        end

        context 'when user provides name' do
          let_it_be(:params) { { name: 'Random bot', expires_at: Date.today + 1.month } }

          it 'overrides the default name value' do
            response = subject
            access_token = response.payload[:access_token]

            expect(access_token.user.name).to eq(params[:name])
          end
        end
      end

      context 'description' do
        context 'when description is not passed' do
          it 'sets description as nil' do
            expect(subject.payload[:access_token].description).to be_nil
          end
        end

        context 'when description is passed' do
          let_it_be(:params) { { description: 'Test token', expires_at: Date.today + 1.month } }

          it 'set the description value' do
            expect(subject.payload[:access_token].description).to eq("Test token")
          end
        end
      end

      context 'bot username and email' do
        include_examples 'username and email pair is generated by Gitlab::Utils::UsernameAndEmailGenerator' do
          subject do
            response = described_class.new(user, resource, params).execute
            response.payload[:access_token].user
          end

          let(:username_prefix) do
            "#{resource.class.name.downcase}_#{resource.id}_bot"
          end

          let(:email_domain) do
            "noreply.#{Gitlab.config.gitlab.host}"
          end
        end
      end

      context 'access level' do
        context 'when user does not specify an access level' do
          it 'adds the bot user as a maintainer in the resource' do
            response = subject
            access_token = response.payload[:access_token]
            bot_user = access_token.user

            expect(resource.members.maintainers.map(&:user_id)).to include(bot_user.id)
          end
        end

        shared_examples 'bot with access level' do
          it 'adds the bot user with the specified access level in the resource' do
            response = subject
            access_token = response.payload[:access_token]
            bot_user = access_token.user

            expect(resource.members.developers.map(&:user_id)).to include(bot_user.id)
          end
        end

        context 'when user specifies an access level' do
          let_it_be(:params) { { access_level: Gitlab::Access::DEVELOPER, expires_at: Date.today + 1.month } }

          it_behaves_like 'bot with access level'
        end

        context 'with DEVELOPER access_level, in string format' do
          let_it_be(:params) { { access_level: Gitlab::Access::DEVELOPER.to_s, expires_at: Date.today + 1.month } }

          it_behaves_like 'bot with access level'
        end

        context 'when user is external' do
          before do
            user.update!(external: true)
          end

          it 'creates resource bot user with external status' do
            expect(subject.payload[:access_token].user.external).to eq true
          end
        end
      end

      context 'personal access token' do
        it { expect { subject }.to change { PersonalAccessToken.count }.by(1) }

        context 'when user does not provide scope' do
          it 'has default scopes' do
            response = subject
            access_token = response.payload[:access_token]

            expect(access_token.scopes).to eq(Gitlab::Auth.resource_bot_scopes)
          end
        end

        context 'when user provides scope explicitly' do
          let_it_be(:params) { { scopes: Gitlab::Auth::REPOSITORY_SCOPES, expires_at: Date.today + 1.month } }

          it 'overrides the default scope value' do
            response = subject
            access_token = response.payload[:access_token]

            expect(access_token.scopes).to eq(Gitlab::Auth::REPOSITORY_SCOPES)
          end
        end

        context 'expires_at' do
          context 'when no expiration value is passed' do
            let_it_be(:params) { { expires_at: nil } }

            context 'when allow_resource_access_token_creation_without_expiry_date is disabled' do
              before do
                stub_feature_flags(allow_resource_access_token_creation_without_expiry_date: false)
              end

              it 'defaults to PersonalAccessToken::MAX_PERSONAL_ACCESS_TOKEN_LIFETIME_IN_DAYS' do
                freeze_time do
                  response = subject
                  access_token = response.payload[:access_token]

                  expect(access_token.expires_at).to eq(
                    max_pat_access_token_lifetime.to_date
                  )
                end
              end

              it 'project bot membership does not expire when PAT expires' do
                response = subject
                access_token = response.payload[:access_token]
                project_bot = access_token.user

                expect(resource.members.find_by(user_id: project_bot.id).expires_at).to be_nil
              end

              context 'when require_personal_access_token_expiry is set to false' do
                before do
                  stub_application_setting(require_personal_access_token_expiry: false)
                end

                it 'returns a nil expiration date' do
                  response = subject
                  access_token = response.payload[:access_token]

                  expect(access_token.expires_at).to be_nil
                end
              end
            end

            context 'when allow_resource_access_token_creation_without_expiry_date is enabled' do
              before do
                stub_feature_flags(allow_resource_access_token_creation_without_expiry_date: true)
              end

              context 'when require_personal_access_token_expiry is true' do
                before do
                  stub_application_setting(require_personal_access_token_expiry: true)
                end

                it 'returns an error' do
                  response = subject
                  expect(response.error?).to be true
                  expect(response.message).to eq('expires_at is missing')
                end
              end

              context 'when require_personal_access_token_expiry is false' do
                before do
                  stub_application_setting(require_personal_access_token_expiry: false)
                end

                it 'returns a nil expiration date' do
                  response = subject
                  access_token = response.payload[:access_token]
                  expect(access_token.expires_at).to be_nil
                end
              end
            end
          end

          context 'when user provides expiration value' do
            let_it_be(:params) { { expires_at: Date.today + 1.month } }

            it 'overrides the default expiration value' do
              response = subject
              access_token = response.payload[:access_token]

              expect(access_token.expires_at).to eq(params[:expires_at])
            end

            it 'sets the project bot to not expire' do
              response = subject
              access_token = response.payload[:access_token]
              project_bot = access_token.user

              expect(resource.members.find_by(user_id: project_bot.id).expires_at).to be_nil
            end
          end

          context 'when expires_at key is missing' do
            let(:params) { { scopes: ['api'] } }

            context 'when flag is enabled and setting requires expiry' do
              before do
                stub_feature_flags(allow_resource_access_token_creation_without_expiry_date: resource)
                stub_application_setting(require_personal_access_token_expiry: true)
              end

              it 'returns an error' do
                response = subject
                expect(response.error?).to be true
                expect(response.message).to eq('expires_at is missing')
              end
            end
          end
        end

        context 'when invalid scope is passed' do
          let(:error_message) { 'Scopes can only contain available scopes' }
          let_it_be(:params) { { scopes: [:invalid_scope], expires_at: Date.today + 1.month } }

          it_behaves_like 'token creation fails'
          it_behaves_like 'correct error message'
          it_behaves_like 'deletes failed project bot'
        end
      end

      context "when access provisioning fails" do
        let(:unpersisted_member) { build(:project_member, source: resource) }
        let(:error_message) { 'Could not provision maintainer access to the access token. ERROR: error message' }

        before do
          allow_next_instance_of(ResourceAccessTokens::CreateService) do |service|
            allow(service).to receive(:create_membership).and_return(unpersisted_member)
          end

          allow(unpersisted_member).to receive_message_chain(:errors, :full_messages, :to_sentence)
            .and_return('error message')
        end

        context 'with MAINTAINER access_level, in integer format' do
          let_it_be(:params) { { access_level: Gitlab::Access::MAINTAINER, expires_at: Date.today + 1.month } }

          it_behaves_like 'token creation fails'
          it_behaves_like 'correct error message'
          it_behaves_like 'deletes failed project bot'
        end

        context 'with MAINTAINER access_level, in string format' do
          let_it_be(:params) { { access_level: Gitlab::Access::MAINTAINER.to_s, expires_at: Date.today + 1.month } }

          it_behaves_like 'token creation fails'
          it_behaves_like 'correct error message'
          it_behaves_like 'deletes failed project bot'
        end
      end

      it 'logs the event' do
        allow(Gitlab::AppLogger).to receive(:info)

        response = subject

        expect(Gitlab::AppLogger).to have_received(:info).with(/PROJECT ACCESS TOKEN CREATION: created_by: #{user.username}, project_id: #{resource.id}, token_user: #{response.payload[:access_token].user.name}, token_id: \d+/)
      end
    end

    shared_examples 'when user does not have permission to create a resource bot' do
      let(:error_message) { "User does not have permission to create #{resource_type} access token" }

      it_behaves_like 'token creation fails'
      it_behaves_like 'correct error message'
    end

    context 'when resource is a project' do
      let_it_be(:resource_type) { 'project' }
      let_it_be(:resource) { project }

      it_behaves_like 'when user does not have permission to create a resource bot'

      context 'user with valid permission' do
        before_all do
          resource.add_maintainer(user)
        end

        it_behaves_like 'allows creation of bot with valid params'

        it 'sets user_type but not group_id for projects in a UserNamespace' do
          expect(resource.namespace).to be_a(Namespaces::UserNamespace)

          response = subject
          access_token = response.payload[:access_token]
          expect(access_token.group_id).to be_nil
          expect(access_token.user_type).to eq('project_bot')
        end

        context 'when user specifies an access level of OWNER for the bot' do
          let_it_be(:params) { { access_level: Gitlab::Access::OWNER, expires_at: Date.today + 1.month } }

          context 'when the executor is a MAINTAINER' do
            let(:error_message) { 'Access level of the token contains permissions not held by the creating user' }

            context 'with OWNER access_level, in integer format' do
              it_behaves_like 'token creation fails'
              it_behaves_like 'correct error message'
            end

            context 'with OWNER access_level, in string format' do
              let(:error_message) { 'Access level of the token contains permissions not held by the creating user' }
              let_it_be(:params) { { access_level: Gitlab::Access::OWNER.to_s, expires_at: Date.today + 1.month } }

              it_behaves_like 'token creation fails'
              it_behaves_like 'correct error message'
            end
          end

          context 'when the executor is an OWNER' do
            let_it_be(:user) { project.first_owner }

            it 'adds the bot user with the specified access level in the resource' do
              response = subject

              access_token = response.payload[:access_token]
              bot_user = access_token.user

              expect(resource.members.owners.map(&:user_id)).to include(bot_user.id)
            end
          end
        end
      end
    end

    context 'when resource is a group' do
      let_it_be(:resource_type) { 'group' }
      let_it_be(:resource) { group }

      it_behaves_like 'when user does not have permission to create a resource bot'

      context 'user with valid permission' do
        before_all do
          resource.add_owner(user)
        end

        it_behaves_like 'allows creation of bot with valid params'

        context 'when user specifies an access level of OWNER for the bot' do
          let_it_be(:params) { { access_level: Gitlab::Access::OWNER, expires_at: Date.today + 1.month } }

          it 'adds the bot user with the specified access level in the resource' do
            response = subject
            access_token = response.payload[:access_token]
            bot_user = access_token.user

            expect(resource.members.owners.map(&:user_id)).to include(bot_user.id)
          end
        end
      end
    end

    context 'when resource is a sub-group' do
      let_it_be(:resource_type) { 'group' }
      let_it_be(:parent_group) { create(:group, :private, organization: organization) }
      let_it_be(:child_group) { create(:group, :private, organization: organization, parent: parent_group) }
      let_it_be(:resource) { child_group }

      it_behaves_like 'when user does not have permission to create a resource bot'

      context 'user with valid permission' do
        before_all do
          resource.add_owner(user)
        end

        it_behaves_like 'allows creation of bot with valid params'
      end
    end

    context 'when resource is a project inside a sub-group' do
      let_it_be(:resource_type) { 'project' }
      let_it_be(:parent_group) { create(:group, :private, organization: organization) }
      let_it_be(:child_group) { create(:group, :private, organization: organization, parent: parent_group) }
      let_it_be(:child_project) { create(:project, :private, organization: organization, namespace: child_group) }
      let_it_be(:resource) { child_project }

      it_behaves_like 'when user does not have permission to create a resource bot'

      context 'user with valid permission' do
        before_all do
          resource.add_owner(user)
        end

        it_behaves_like 'allows creation of bot with valid params'
      end
    end
  end
end
