# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::BackfillVulnerabilityDetectionTransitions,
  feature_category: :vulnerability_management do
  let(:vulnerability_occurrences) { table(:vulnerability_occurrences, database: :sec) }
  let(:vulnerability_detection_transitions) { table(:vulnerability_detection_transitions, database: :sec) }
  let(:vulnerabilities) { table(:vulnerabilities, database: :sec) }
  let(:scanners) { table(:vulnerability_scanners, database: :sec) }
  let(:identifiers) { table(:vulnerability_identifiers, database: :sec) }

  let(:organizations) { table(:organizations) }
  let(:namespaces) { table(:namespaces) }
  let(:projects) { table(:projects) }
  let(:users) { table(:users) }

  let!(:organization) { organizations.create!(name: 'org', path: 'org') }
  let!(:namespace) { namespaces.create!(name: 'ns', path: 'ns', organization_id: organization.id) }
  let!(:project_namespace) { namespaces.create!(name: 'proj', path: 'proj', organization_id: organization.id) }
  let!(:project) do
    projects.create!(
      name: 'proj', path: 'proj',
      namespace_id: namespace.id,
      project_namespace_id: project_namespace.id,
      organization_id: organization.id
    )
  end

  let!(:user) { users.create!(email: 'test@example.com', projects_limit: 10, organization_id: organization.id) }
  let!(:scanner) { scanners.create!(project_id: project.id, external_id: 'scanner', name: 'Scanner') }
  let!(:identifier) do
    identifiers.create!(
      project_id: project.id,
      fingerprint: SecureRandom.hex(20),
      external_type: 'cve',
      external_id: 'CVE-2021-1234',
      name: 'CVE-2021-1234'
    )
  end

  def create_finding_and_vulnerability(resolved_on_default_branch:, state: 1, updated_at: Time.current)
    occurrence = vulnerability_occurrences.create!(
      project_id: project.id,
      scanner_id: scanner.id,
      primary_identifier_id: identifier.id,
      name: 'Finding',
      severity: 1,
      report_type: 1,
      uuid: SecureRandom.uuid,
      location_fingerprint: SecureRandom.hex(20),
      metadata_version: '1.0',
      raw_metadata: '{}'
    )

    vulnerability = vulnerabilities.create!(
      project_id: project.id,
      author_id: user.id,
      finding_id: occurrence.id,
      title: 'Vuln',
      severity: 1,
      report_type: 1,
      state: state,
      resolved_on_default_branch: resolved_on_default_branch
    )

    vulnerability_occurrences.where(id: occurrence.id).update_all(
      vulnerability_id: vulnerability.id,
      updated_at: updated_at
    )

    vulnerability_occurrences.find(occurrence.id)
  end

  def args
    {
      start_id: vulnerability_occurrences.minimum(:id),
      end_id: vulnerability_occurrences.maximum(:id),
      batch_table: :vulnerability_occurrences,
      batch_column: :id,
      sub_batch_size: 100,
      pause_ms: 0,
      connection: SecApplicationRecord.connection
    }
  end

  subject(:perform_migration) { described_class.new(**args).perform }

  describe '#perform' do
    context 'when resolved_on_default_branch = true' do
      let!(:occurrence) { create_finding_and_vulnerability(resolved_on_default_branch: true) }

      it 'creates detection_transition with detected: false' do
        expect { perform_migration }.to change { vulnerability_detection_transitions.count }.by(1)

        transition = vulnerability_detection_transitions.last
        expect(transition.vulnerability_occurrence_id).to eq(occurrence.id)
        expect(transition.detected).to be(false)
      end
    end

    context 'when resolved_on_default_branch = false' do
      let!(:occurrence) { create_finding_and_vulnerability(resolved_on_default_branch: false) }

      it 'does not create detection_transition' do
        expect { perform_migration }.not_to change { vulnerability_detection_transitions.count }
      end
    end

    context 'when transition already exists (idempotency)' do
      let!(:occurrence) { create_finding_and_vulnerability(resolved_on_default_branch: true) }
      let!(:existing_transition) do
        vulnerability_detection_transitions.create!(
          vulnerability_occurrence_id: occurrence.id,
          project_id: project.id,
          detected: false,
          created_at: Time.current,
          updated_at: Time.current
        )
      end

      it 'does not create duplicate' do
        expect { perform_migration }.not_to change { vulnerability_detection_transitions.count }
      end
    end
  end
end
