import mermaid from 'mermaid';
import DOMPurify from 'dompurify';
import { getParameterByName } from '~/lib/utils/url_utility';
import { resetServiceWorkersPublicPath } from '~/lib/utils/webpack';

const resetWebpackPublicPath = () => {
  window.gon = { relative_url_root: getParameterByName('relativeRootPath') };
  resetServiceWorkersPublicPath();
};

resetWebpackPublicPath();
const setIframeRenderedSize = (h, w) => {
  const { origin } = window.location;
  window.parent.postMessage({ h, w }, origin);
};

const drawDiagram = async (source) => {
  const element = document.getElementById('app');
  const insertSvg = (svgCode) => {
    // eslint-disable-next-line no-unsanitized/property
    element.innerHTML = svgCode;

    element.firstElementChild.removeAttribute('height');
    const { height, width } = element.firstElementChild.getBoundingClientRect();

    setIframeRenderedSize(height, width);
  };

  const { svg } = await mermaid.mermaidAPI.render('mermaid', source);
  insertSvg(svg);
};

const darkModeEnabled = () => getParameterByName('darkMode') === 'true';

const initMermaid = () => {
  let theme = 'neutral';

  if (darkModeEnabled()) {
    theme = 'dark';
  }

  mermaid.initialize({
    // mermaid core options
    mermaid: {
      startOnLoad: false,
    },
    // mermaidAPI options
    theme,
    flowchart: {
      useMaxWidth: true,
      htmlLabels: true,
    },
    secure: ['secure', 'securityLevel', 'startOnLoad', 'maxTextSize', 'htmlLabels'],
    securityLevel: 'strict',
    dompurifyConfig: {
      ADD_TAGS: ['foreignObject'],
      HTML_INTEGRATION_POINTS: { foreignobject: true },
    },
  });
};

// The mapping of URLs to asset proxy URLs generated by the backend. Received in the
// message from the parent document, stored here so the DOMPurify hook can use it.
// If the asset proxy is not in use, this will be set to null.
// If the asset proxy is in use, this will be a mapping of URLs that *may* appear in
// the created DOM as image sources.  We *must* remove any image which doesn't have
// a corresponding asset proxy URL to avoid bypasses.
let proxiedUrls = null;

// DOMPurify runs afterSanitizeAttributes multiple times per node; we mark the asset
// proxy URLs we substitute in so we don't then remove them for not having a key in
// proxiedUrls!
const permittedUrls = new Set();

const configureDOMPurify = () => {
  DOMPurify.addHook('afterSanitizeAttributes', (node) => {
    if (proxiedUrls === null) return;

    if (node.tagName !== 'IMG') return;

    if (!node.hasAttribute('src')) return;
    if (permittedUrls.has(node.src)) return;

    const proxied = proxiedUrls[node.src];
    if (!proxied) {
      node.remove();
    } else {
      // eslint-disable-next-line no-param-reassign
      node.src = proxied;
      permittedUrls.add(proxied);
    }
  });
};

const addListener = () => {
  window.addEventListener(
    'message',
    (event) => {
      if (event.origin !== window.location.origin) {
        return;
      }
      const { source } = event.data;
      proxiedUrls = event.data.proxiedUrls;
      drawDiagram(source);
    },
    false,
  );
};

addListener();
configureDOMPurify();
initMermaid();
export default {};
