# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Integrations::BeyondIdentity, feature_category: :integrations do
  subject(:integration) { create(:beyond_identity_integration, :instance) }

  describe 'validations' do
    context 'when inactive' do
      before do
        integration.active = false
      end

      it { is_expected.not_to validate_presence_of(:token) }
    end

    context 'when active' do
      it { is_expected.to validate_presence_of(:token) }
    end
  end

  describe 'attributes' do
    it 'configures attributes' do
      expect(integration.supported_events).to be_blank
      expect(integration.to_param).to eq('beyond_identity')
      expect(integration.title).to eq('Beyond Identity')

      expect(integration.description).to eq(
        'Verify that GPG keys are authorized by Beyond Identity Authenticator.'
      )

      expect(integration.help).to include(
        'Verify that GPG keys are authorized by Beyond Identity Authenticator.'
      )
    end
  end

  describe '.api_arguments' do
    it 'returns api arguments' do
      expect(described_class.api_arguments).to eq([{
        required: true,
        name: :token,
        type: String,
        desc: 'API Token. User must have access to `git-commit-signing` endpoint.'
      }, {
        required: false,
        name: :exclude_service_accounts,
        type: Grape::API::Boolean,
        desc: "If enabled, Beyond Identity will not check commits from service accounts."
      }])
    end
  end

  describe '#execute' do
    it 'performs a request to beyond identity service' do
      params = { key_id: 'key-id', committer_email: 'email' }
      response = 'response'

      expect_next_instance_of(::Gitlab::BeyondIdentity::Client) do |instance|
        expect(instance).to receive(:execute).with(params).and_return(response)
      end

      expect(integration.execute(params)).to eq(response)
    end
  end

  describe '.activated_for_instance?' do
    let!(:integration) do
      create(:beyond_identity_integration, instance: instance, active: active, group: group, organization: organization)
    end

    let_it_be(:group_for_integration) { create(:group) }
    let_it_be(:organization_for_integration) { create(:organization) }

    subject(:activated) { described_class.activated_for_instance? }

    using RSpec::Parameterized::TableSyntax

    where(:instance, :group, :active, :expected, :organization) do
      true  | nil | true | true | lazy { organization_for_integration }
      false | lazy { group_for_integration } | true | false | nil
      true  | nil | false | false | lazy { organization_for_integration }
      false | lazy { group_for_integration } | false | false | nil
    end

    with_them do
      it 'returns true if integration is activated' do
        expect(activated).to eq(expected)
      end
    end
  end
end
