# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::UpdateOwaspTop10DefaultOfVulnerabilityReads,
  schema: 20241024183136,
  feature_category: :vulnerability_management do
  before(:all) do
    # This migration will not work if a sec database is configured. It should be finalized and removed prior to
    # sec db rollout.
    # Consult https://gitlab.com/gitlab-org/gitlab/-/merge_requests/171707 for more info.
    skip_if_multiple_databases_are_setup(:sec)
  end

  let(:organizations) { table(:organizations) }
  let(:namespaces) { table(:namespaces) }
  let(:projects) { table(:projects) }
  let(:users) { table(:users) }
  let(:scanners) { table(:vulnerability_scanners) }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:vulnerability_findings) { table(:vulnerability_occurrences) }
  let(:vulnerability_identifiers) { table(:vulnerability_identifiers) }

  let(:organization) { organizations.create!(name: 'organization', path: 'organization') }
  let(:namespace) { namespaces.create!(name: 'user', path: 'user', organization_id: organization.id) }
  let(:project) do
    projects.create!(namespace_id: namespace.id, project_namespace_id: namespace.id, organization_id: organization.id)
  end

  let(:user) { users.create!(username: 'john_doe', email: 'johndoe@gitlab.com', projects_limit: 10) }
  let(:scanner) { scanners.create!(project_id: project.id, external_id: 'external_id', name: 'Test Scanner') }

  let(:vulnerability_1) { create_vulnerability(title: 'vulnerability 1', finding_id: create_finding.id) }
  let(:vulnerability_2) { create_vulnerability(title: 'vulnerability 2', finding_id: create_finding.id) }
  let(:vulnerability_3) { create_vulnerability(title: 'vulnerability 3', finding_id: create_finding.id) }

  let!(:vulnerability_read_1) { create_vulnerability_read(vulnerability_id: vulnerability_1.id, owasp_top_10: nil) }
  let!(:vulnerability_read_2) { create_vulnerability_read(vulnerability_id: vulnerability_2.id, owasp_top_10: 1) }
  let!(:vulnerability_read_3) { create_vulnerability_read(vulnerability_id: vulnerability_3.id, owasp_top_10: nil) }

  describe '#perform' do
    subject(:perform_migration) do
      described_class.new(
        start_id: vulnerability_reads.first.id,
        end_id: vulnerability_reads.last.id,
        batch_table: :vulnerability_reads,
        batch_column: :id,
        sub_batch_size: vulnerability_reads.count,
        pause_ms: 0,
        connection: ActiveRecord::Base.connection
      ).perform
    end

    it 'updates vulnerability_reads with NULL owasp_top_10 to -1' do
      expect { perform_migration }
        .to change { vulnerability_reads.where(owasp_top_10: -1).count }.from(0).to(2)
    end

    it 'does not change non-NULL owasp_top_10 values' do
      expect { perform_migration }.not_to change { vulnerability_read_2.reload.owasp_top_10 }.from(1)
    end
  end

  private

  def create_vulnerability(overrides = {})
    vulnerabilities.create!({
      project_id: project.id,
      author_id: user.id,
      title: 'test',
      severity: 1,
      confidence: 1,
      report_type: 1
    }.merge(overrides))
  end

  def create_vulnerability_read(overrides = {})
    vulnerability_reads.create!({
      project_id: project.id,
      vulnerability_id: 1,
      scanner_id: scanner.id,
      severity: 1,
      report_type: 1,
      state: 1,
      uuid: SecureRandom.uuid
    }.merge(overrides))
  end

  def create_finding(overrides = {})
    vulnerability_findings.create!({
      project_id: project.id,
      scanner_id: scanner.id,
      severity: 5, # medium
      report_type: 99, # generic
      primary_identifier_id: create_identifier.id,
      project_fingerprint: SecureRandom.hex(20),
      location_fingerprint: SecureRandom.hex(20),
      uuid: SecureRandom.uuid,
      name: "CVE-2018-1234",
      raw_metadata: "{}",
      metadata_version: "test:1.0"
    }.merge(overrides))
  end

  def create_identifier(overrides = {})
    vulnerability_identifiers.create!({
      project_id: project.id,
      external_id: "CVE-2018-1234",
      external_type: "CVE",
      name: "CVE-2018-1234",
      fingerprint: SecureRandom.hex(20)
    }.merge(overrides))
  end
end
