use gitalisk_core::repository::testing::local::LocalGitRepository;
use std::fs;
use std::path::{Path, PathBuf};

pub struct TestRepository {
    pub dir: PathBuf,
    pub git_repo: LocalGitRepository,
}

// this is a helper to create a test repository with a git repo structure
// and optionally copy fixture files from the fixtures directory, which is
// located at the root of the project under fixtures/
// example usage:
// ```rust,ignore
// let fixture_dir_name = Some("test-repo");
// let test_repo = TestRepository::new(fixture_dir_name);
// let repo_path = &test_repo.dir;
// assert!(repo_path.exists());
// assert!(repo_path.join(".git").exists());
// ```
impl TestRepository {
    pub fn new(fixture_dir_name: Option<&str>) -> Self {
        let git_repo = LocalGitRepository::new(None);

        if let Some(fixture_dir_name) = fixture_dir_name {
            copy_fixtures_to_test_repo(&git_repo.path, fixture_dir_name);
        }

        Self {
            dir: git_repo.path.clone(),
            git_repo,
        }
    }

    /// Creates a minimal Git repository without adding or committing files.
    /// This is useful for large repositories where the initial commit would be slow.
    pub fn new_minimal() -> Self {
        let git_repo = LocalGitRepository::new(None);

        Self {
            dir: git_repo.path.clone(),
            git_repo,
        }
    }
}

fn copy_fixtures_to_test_repo(repo_path: &Path, fixture_dir_name: &str) {
    let fixtures_root = PathBuf::from(env!("CARGO_MANIFEST_DIR"))
        .parent()
        .unwrap()
        .parent()
        .unwrap()
        .join("fixtures");

    let fixture_dir = fixtures_root.join(fixture_dir_name);

    if !fixture_dir.exists() {
        panic!("Fixture directory does not exist: {fixture_dir:?}");
    }

    copy_dir_all(&fixture_dir, repo_path).expect("Failed to copy fixtures to test repository");
}

pub fn copy_dir_all(src: &Path, dst: &Path) -> std::io::Result<()> {
    if !dst.exists() {
        fs::create_dir_all(dst)?;
    }

    for entry in fs::read_dir(src)? {
        let entry = entry?;
        let src_path = entry.path();
        let dst_path = dst.join(entry.file_name());

        if src_path.is_dir() {
            copy_dir_all(&src_path, &dst_path)?;
        } else {
            fs::copy(&src_path, &dst_path)?;
        }
    }

    Ok(())
}
