#[global_allocator]
static GLOBAL: mimalloc::MiMalloc = mimalloc::MiMalloc;

mod stats;
use stats::Stats;

use clap::Parser;
use ignore::{DirEntry, WalkBuilder};
use parser_core::parser::{
    ParserType, SupportedLanguage, UnifiedParseResult, detect_language_from_extension,
    supported_language_from_str,
};
use parser_core::rules::{MatchWithNodes, RuleManager};
use std::collections::{HashMap, HashSet};
use std::path::{Path, PathBuf};
use std::sync::LazyLock;
use std::thread;
use tracing::{error, info};

static DEFAULT_THREADS: LazyLock<usize> = LazyLock::new(num_cpus::get);

#[derive(Parser, Debug)]
#[clap(author, version, about, long_about = None)]
struct Cli {
    #[clap(subcommand)]
    command: Commands,
}

#[derive(Parser, Debug)]
enum Commands {
    Analyze {
        /// The path to the file to parse.
        #[clap(long)]
        directory: PathBuf,

        #[clap(long, default_value_t = *DEFAULT_THREADS)]
        threads: usize,

        #[clap(long, default_value = "")]
        languages: String,

        #[clap(long, default_value_t = false)]
        exclude_suggested_extensions: bool,

        /// Optional path to output stats in JSON format
        #[clap(long)]
        stats_output: Option<PathBuf>,
    },
}

// Entrypoint for the parser CLI - A utility for performing project tasks.
// Usage: cargo run --package xtask -- <command>
// For help: cargo run --package xtask -- --help
fn main() -> anyhow::Result<()> {
    let cli = Cli::parse();
    tracing_subscriber::fmt::init();

    match cli.command {
        Commands::Analyze {
            directory,
            threads,
            languages,
            exclude_suggested_extensions,
            stats_output,
        } => {
            info!("Analyzing files in {:?}", directory);
            info!("Threads: {:?}", threads);

            let files = capture_files(&directory, &languages, exclude_suggested_extensions);
            let stats = analyze_files(&files, threads);

            if let Some(output_path) = stats_output {
                let json_output = stats.to_json()?;
                std::fs::write(&output_path, json_output)?;
                info!("Stats written to {:?}", output_path);
            }
        }
    }

    Ok(())
}

fn file_type_ext_to_language(entry: &DirEntry) -> Option<SupportedLanguage> {
    if let Some(ext) = entry.path().extension().and_then(|e| e.to_str()) {
        let language = detect_language_from_extension(ext);
        if let Ok(language) = language {
            return Some(language);
        } else {
            return None;
        }
    }
    None
}

fn capture_files(
    directory: &Path,
    languages: &str,
    exclude_suggested_extensions: bool,
) -> Vec<PathBuf> {
    let mut files = Vec::new();
    // Split comma-separated languages and filter out empty strings
    let languages_set: HashSet<SupportedLanguage> = languages
        .split(',')
        .map(|s| s.trim())
        .filter(|s| !s.is_empty())
        .map(|s| supported_language_from_str(s).unwrap())
        .collect();

    if exclude_suggested_extensions {
        let excluded_list = languages_set
            .iter()
            .flat_map(|l| l.exclude_extensions().iter().copied())
            .collect::<Vec<&str>>();
        info!("Excluding extensions: {:?}", excluded_list);
    }

    if !languages.is_empty() {
        info!("Languages: {:?}", languages_set);
    } else {
        info!("No languages provided, will analyze all files");
    }

    let walker = WalkBuilder::new(directory)
        .require_git(false)
        // Maximum exclusions - skip everything we don't need
        .filter_entry(move |entry| {
            // For files, only include those with extensions that could be parsed
            if entry.file_type().map(|ft| ft.is_file()).unwrap_or(false)
                && let Some(language) = file_type_ext_to_language(entry)
            {
                if !languages_set.is_empty() {
                    return languages_set.contains(&language);
                } else {
                    return true;
                }
            }
            true // allow directories to be traversed
        })
        .build();

    for result in walker {
        match result {
            Ok(entry) => {
                if entry.file_type().map(|ft| ft.is_file()).unwrap_or(false)
                    && let Some(language) = file_type_ext_to_language(&entry)
                {
                    if exclude_suggested_extensions
                        && language
                            .exclude_extensions()
                            .iter()
                            .any(|suffix| entry.path().ends_with(suffix))
                    {
                        continue;
                    }
                    files.push(entry.path().to_path_buf());
                }
            }
            Err(e) => {
                error!("Error walking directory: {}", e);
            }
        }
    }

    files
}

fn analyze_files(files: &[PathBuf], threads: usize) -> Stats {
    if files.is_empty() {
        info!("No files found to analyze");
        return Stats::default();
    }

    let analyze_time_start = std::time::Instant::now();

    let chunk_size = files.len().div_ceil(threads);
    let file_chunks: Vec<Vec<PathBuf>> = files
        .chunks(chunk_size)
        .map(|chunk| chunk.to_vec())
        .collect();

    let mut handles = vec![];
    for chunk in file_chunks.into_iter() {
        let handle = thread::spawn(move || analyze_file_chunk(chunk));
        handles.push(handle);
    }

    // Wait for all threads to complete and collect results
    let mut global_stats = Stats::default();
    for handle in handles {
        match handle.join() {
            Ok(Ok(chunk_stats)) => {
                global_stats += chunk_stats;
            }
            Ok(Err(e)) => {
                error!("Thread returned error: {:?}", e);
            }
            Err(e) => {
                error!("Thread panicked: {:?}", e);
            }
        }
    }

    let analyze_time_end = std::time::Instant::now();
    let analyze_time = analyze_time_end.duration_since(analyze_time_start);
    info!("Analyzed {} files in {:?}", files.len(), analyze_time);
    info!("Global stats: {:?}", global_stats);

    global_stats
}

fn analyze_file_chunk(files: Vec<PathBuf>) -> Result<Stats, anyhow::Error> {
    let mut parsers: HashMap<SupportedLanguage, ParserType> = HashMap::new();
    let mut chunk_stats = Stats::default();

    for file in files {
        let file_path = file.to_str().unwrap();
        let extension = file.extension().unwrap().to_str().unwrap();
        let language = detect_language_from_extension(extension).unwrap();

        parsers
            .entry(language)
            .or_insert_with(|| ParserType::for_language(language));
        let parser = parsers.get(&language).unwrap();

        let file_content = match std::fs::read_to_string(&file) {
            Ok(content) => content,
            Err(e) => {
                error!("Failed to read file {}: {}", file_path, e);
                continue;
            }
        };

        let parse_result = match parser.parse(&file_content, Some(file_path)) {
            Ok(result) => result,
            Err(e) => {
                error!("Failed to parse file {}: {}", file_path, e);
                continue;
            }
        };

        // Get the AST for rule execution - we need to handle both AST-grep and Ruby results
        let matches = match &parse_result {
            UnifiedParseResult::AstGrep(ast_result) => {
                let rule_manager = RuleManager::new(language);
                rule_manager
                    .execute_rules(&ast_result.ast, Some(file_path))
                    .unwrap()
            }
            UnifiedParseResult::Ruby(_) => {
                // For Ruby, we don't use ast-grep rules, so return empty matches
                Vec::new()
            }
            UnifiedParseResult::TypeScript(_) => {
                // For TypeScript, we don't use ast-grep rules, so return empty matches
                Vec::new()
            }
        };

        let file_stats = analyze_file(matches, &parse_result, language, &file_content)?;
        chunk_stats += file_stats;
    }

    Ok(chunk_stats)
}

fn analyze_file(
    matches: Vec<MatchWithNodes>,
    parse_result: &UnifiedParseResult,
    supported_language: SupportedLanguage,
    file_content: &str,
) -> Result<Stats, anyhow::Error> {
    match supported_language {
        SupportedLanguage::Ruby => {
            if let UnifiedParseResult::Ruby(ruby_result) = parse_result {
                let analyzer = parser_core::ruby::analyzer::RubyAnalyzer::new();
                match analyzer.analyze_with_prism(file_content, &ruby_result.ast) {
                    Ok(analysis_result) => Ok(Stats::from_analysis_result(&analysis_result)),
                    Err(e) => {
                        error!("Error analyzing Ruby file: {:?}", e);
                        Ok(Stats::default())
                    }
                }
            } else {
                Ok(Stats::default())
            }
        }
        SupportedLanguage::Python => {
            if let UnifiedParseResult::AstGrep(ast_result) = parse_result {
                let analyzer = parser_core::python::analyzer::PythonAnalyzer::new();
                match analyzer.analyze(&matches, ast_result) {
                    Ok(analysis_result) => Ok(Stats::from_analysis_result(&analysis_result)),
                    Err(e) => {
                        error!("Error analyzing file: {:?}", e);
                        Ok(Stats::default())
                    }
                }
            } else {
                Ok(Stats::default())
            }
        }
        SupportedLanguage::Java => {
            if let UnifiedParseResult::AstGrep(ast_result) = parse_result {
                let analyzer = parser_core::java::analyzer::JavaAnalyzer::new();
                match analyzer.analyze(ast_result) {
                    Ok(analysis_result) => Ok(Stats::from_destructured_analysis_result(
                        &analysis_result.definitions,
                        &analysis_result.imports,
                        &analysis_result.references,
                    )),
                    Err(e) => {
                        error!("Error analyzing file: {:?}", e);
                        Ok(Stats::default())
                    }
                }
            } else {
                Ok(Stats::default())
            }
        }
        SupportedLanguage::Kotlin => {
            if let UnifiedParseResult::AstGrep(ast_result) = parse_result {
                let analyzer = parser_core::kotlin::analyzer::KotlinAnalyzer::new();
                match analyzer.analyze(ast_result) {
                    Ok(analysis_result) => Ok(Stats::from_destructured_analysis_result(
                        &analysis_result.definitions,
                        &analysis_result.imports,
                        &analysis_result.references,
                    )),
                    Err(e) => {
                        error!("Error analyzing file: {:?}", e);
                        Ok(Stats::default())
                    }
                }
            } else {
                Ok(Stats::default())
            }
        }
        SupportedLanguage::CSharp => {
            if let UnifiedParseResult::AstGrep(ast_result) = parse_result {
                let analyzer = parser_core::csharp::analyzer::CSharpAnalyzer::new();
                match analyzer.analyze(ast_result) {
                    Ok(analysis_result) => Ok(Stats::from_analysis_result_simple(&analysis_result)),
                    Err(e) => {
                        error!("Error analyzing file: {:?}", e);
                        Ok(Stats::default())
                    }
                }
            } else {
                Ok(Stats::default())
            }
        }
        SupportedLanguage::Rust => {
            if let UnifiedParseResult::AstGrep(ast_result) = parse_result {
                let analyzer = parser_core::rust::analyzer::RustAnalyzer::new();
                match analyzer.analyze(&matches, ast_result) {
                    Ok(analysis_result) => Ok(Stats::from_analysis_result_simple(&analysis_result)),
                    Err(e) => {
                        error!("Error analyzing file: {:?}", e);
                        Ok(Stats::default())
                    }
                }
            } else {
                Ok(Stats::default())
            }
        }
        SupportedLanguage::TypeScript => {
            if let UnifiedParseResult::AstGrep(ast_result) = parse_result {
                let analyzer = parser_core::typescript::analyzer::TypeScriptAnalyzer::new();
                match analyzer.analyze(ast_result) {
                    Ok(analysis_result) => Ok(Stats::from_analysis_result(&analysis_result)),
                    Err(e) => {
                        error!("Error analyzing file: {:?}", e);
                        Ok(Stats::default())
                    }
                }
            } else {
                Ok(Stats::default())
            }
        }
    }
}
