#[derive(Debug, Clone)]
pub struct Chunk<'a> {
    pub file_path: &'a str,
    pub start_byte: usize,
    pub end_byte: usize,
    pub start_line: usize,
    pub content: &'a str,
    pub language: &'a str,
}

#[derive(Debug)]
pub enum ChunkError {
    ParseError(String),
    NoSuitableSplitPoints,
    UnsupportedLanguage(String),
    InvalidOptions(String),
    TokenizationError(String),
}

impl std::fmt::Display for ChunkError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            ChunkError::ParseError(e) => write!(f, "Parse error: {e}"),
            ChunkError::NoSuitableSplitPoints => {
                write!(
                    f,
                    "No suitable split points found for code chunk exceeding max size"
                )
            }
            ChunkError::UnsupportedLanguage(lang) => write!(f, "Unsupported language: {lang}"),
            ChunkError::InvalidOptions(msg) => write!(f, "Invalid options: {msg}"),
            ChunkError::TokenizationError(e) => write!(f, "Tokenization error: {e}"),
        }
    }
}

impl std::error::Error for ChunkError {}

/// Helper function to calculate line number from byte position in source code
/// Lines are 1-indexed (first line is line 1)
pub fn byte_to_line_number(source_code: &str, byte_position: usize) -> usize {
    if byte_position == 0 {
        return 1;
    }

    let clamped_position = byte_position.min(source_code.len());
    source_code[..clamped_position]
        .chars()
        .filter(|&c| c == '\n')
        .count()
        + 1
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_byte_to_line_number() {
        let source = "line 1\nline 2\nline 3\n";

        // Start of file is line 1
        assert_eq!(byte_to_line_number(source, 0), 1);

        // Before first newline is still line 1
        assert_eq!(byte_to_line_number(source, 5), 1);

        // Right after first newline is line 2
        assert_eq!(byte_to_line_number(source, 7), 2);

        // Right after second newline is line 3
        assert_eq!(byte_to_line_number(source, 14), 3);

        // End of string is line 4 (after final newline)
        assert_eq!(byte_to_line_number(source, source.len()), 4);

        // Test with position beyond string length (should clamp)
        assert_eq!(byte_to_line_number(source, source.len() + 10), 4);
    }

    #[test]
    fn test_byte_to_line_number_no_newlines() {
        let source = "single line content";
        assert_eq!(byte_to_line_number(source, 0), 1);
        assert_eq!(byte_to_line_number(source, 5), 1);
        assert_eq!(byte_to_line_number(source, source.len()), 1);
    }

    #[test]
    fn test_byte_to_line_number_empty_string() {
        assert_eq!(byte_to_line_number("", 0), 1);
    }
}

#[derive(Debug, Clone)]
pub struct File<'a> {
    pub file_path: &'a str,
    pub source_code: &'a str,
}

pub trait FileChunker {
    fn chunk_file<'a>(
        &self,
        target: &mut Vec<Chunk<'a>>,
        file: &File<'a>,
    ) -> Result<(), ChunkError>;
}

pub trait Sizer {
    fn find_end_byte(&self, content: &str, start_byte: usize) -> Result<(usize, bool), ChunkError>;
}
