package main

import (
	"encoding/json"
	"os"
	"path/filepath"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestShouldDownload(t *testing.T) {
	t.Run("library file does not exist", func(t *testing.T) {
		tempDir := t.TempDir()
		needsDownload, reason := shouldDownload(tempDir, "1.0.0", "linux-amd64")
		assert.True(t, needsDownload)
		assert.Equal(t, "library file does not exist", reason)
	})

	t.Run("metadata file does not exist (legacy install)", func(t *testing.T) {
		tempDir := t.TempDir()

		// Create lib directory and library file
		libDir := filepath.Join(tempDir, "lib")
		require.NoError(t, os.MkdirAll(libDir, 0755))
		require.NoError(t, os.WriteFile(filepath.Join(libDir, "libparser_c_bindings.a"), []byte("fake lib"), 0644))

		needsDownload, reason := shouldDownload(tempDir, "1.0.0", "linux-amd64")
		assert.True(t, needsDownload)
		assert.Equal(t, "metadata file does not exist (legacy install)", reason)
	})

	t.Run("version mismatch", func(t *testing.T) {
		tempDir := t.TempDir()

		// Create lib directory and library file
		libDir := filepath.Join(tempDir, "lib")
		require.NoError(t, os.MkdirAll(libDir, 0755))
		require.NoError(t, os.WriteFile(filepath.Join(libDir, "libparser_c_bindings.a"), []byte("fake lib"), 0644))

		// Create metadata file with different version
		metadata := Metadata{
			Version:  "0.19.0",
			Platform: "linux-amd64",
			Date:     "2023-01-01T00:00:00Z",
		}
		metadataBytes, err := json.Marshal(metadata)
		require.NoError(t, err)
		require.NoError(t, os.WriteFile(filepath.Join(tempDir, "metadata.json"), metadataBytes, 0644))

		needsDownload, reason := shouldDownload(tempDir, "1.0.0", "linux-amd64")
		assert.True(t, needsDownload)
		assert.Equal(t, "version mismatch: have 0.19.0, need 1.0.0", reason)
	})

	t.Run("platform mismatch", func(t *testing.T) {
		tempDir := t.TempDir()

		// Create lib directory and library file
		libDir := filepath.Join(tempDir, "lib")
		require.NoError(t, os.MkdirAll(libDir, 0755))
		require.NoError(t, os.WriteFile(filepath.Join(libDir, "libparser_c_bindings.a"), []byte("fake lib"), 0644))

		// Create metadata file with different platform
		metadata := Metadata{
			Version:  "1.0.0",
			Platform: "darwin-amd64",
			Date:     "2023-01-01T00:00:00Z",
		}
		metadataBytes, err := json.Marshal(metadata)
		require.NoError(t, err)
		require.NoError(t, os.WriteFile(filepath.Join(tempDir, "metadata.json"), metadataBytes, 0644))

		needsDownload, reason := shouldDownload(tempDir, "1.0.0", "linux-amd64")
		assert.True(t, needsDownload)
		assert.Equal(t, "platform mismatch: have darwin-amd64, need linux-amd64", reason)
	})

	t.Run("everything up to date", func(t *testing.T) {
		tempDir := t.TempDir()

		// Create lib directory and library file
		libDir := filepath.Join(tempDir, "lib")
		require.NoError(t, os.MkdirAll(libDir, 0755))
		require.NoError(t, os.WriteFile(filepath.Join(libDir, "libparser_c_bindings.a"), []byte("fake lib"), 0644))

		// Create metadata file with matching version and platform
		metadata := Metadata{
			Version:  "1.0.0",
			Platform: "linux-amd64",
			Date:     "2023-01-01T00:00:00Z",
		}
		metadataBytes, err := json.Marshal(metadata)
		require.NoError(t, err)
		require.NoError(t, os.WriteFile(filepath.Join(tempDir, "metadata.json"), metadataBytes, 0644))

		needsDownload, reason := shouldDownload(tempDir, "1.0.0", "linux-amd64")
		assert.False(t, needsDownload)
		assert.Empty(t, reason)
	})

	t.Run("corrupted metadata file", func(t *testing.T) {
		tempDir := t.TempDir()

		// Create lib directory and library file
		libDir := filepath.Join(tempDir, "lib")
		require.NoError(t, os.MkdirAll(libDir, 0755))
		require.NoError(t, os.WriteFile(filepath.Join(libDir, "libparser_c_bindings.a"), []byte("fake lib"), 0644))

		// Create corrupted metadata file
		require.NoError(t, os.WriteFile(filepath.Join(tempDir, "metadata.json"), []byte("invalid json"), 0644))

		needsDownload, reason := shouldDownload(tempDir, "1.0.0", "linux-amd64")
		assert.True(t, needsDownload)
		assert.Contains(t, reason, "failed to parse metadata file")
	})
}

func TestWriteMetadata(t *testing.T) {
	tempDir := t.TempDir()

	err := writeMetadata(tempDir, "1.0.0", "linux-amd64")
	require.NoError(t, err)

	// Read back the metadata
	metadataPath := filepath.Join(tempDir, "metadata.json")
	metadataBytes, err := os.ReadFile(metadataPath)
	require.NoError(t, err)

	var metadata Metadata
	err = json.Unmarshal(metadataBytes, &metadata)
	require.NoError(t, err)

	assert.Equal(t, "1.0.0", metadata.Version)
	assert.Equal(t, "linux-amd64", metadata.Platform)
	assert.NotEmpty(t, metadata.Date)
}
