package main

import (
	"archive/tar"
	"compress/gzip"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/fs"
	"log/slog"
	"net/http"
	"os"
	"os/signal"
	"path/filepath"
	"runtime"
	"time"
)

// version is auto-updated by scripts/semantic-release-prepare.sh
const Version = "0.20.2"

// Metadata tracks the downloaded library information
type Metadata struct {
	Version  string `json:"version"`
	Platform string `json:"platform"`
	Date     string `json:"download_date"`
}

func main() {
	// Initialize structured logger
	logger := slog.New(slog.NewTextHandler(os.Stdout, &slog.HandlerOptions{
		Level: slog.LevelInfo,
	}))
	slog.SetDefault(logger)

	ctx, stop := signal.NotifyContext(context.Background(), os.Interrupt)
	defer stop()

	if err := run(ctx); err != nil {
		slog.Error("Application failed", "error", err)
		os.Exit(1)
	}
}

func run(ctx context.Context) error {
	workDir, err := os.Getwd()
	if err != nil {
		return fmt.Errorf("getting working directory: %w", err)
	}

	targetDir := filepath.Join(workDir, "libparser")
	if len(os.Args) > 1 {
		targetDir = os.Args[1]
	}

	if err := os.MkdirAll(targetDir, 0755); err != nil {
		return fmt.Errorf("mkdirall: %w", err)
	}

	root, err := filepath.Abs(targetDir)
	if err != nil {
		return fmt.Errorf("open root: %w", err)
	}

	platform := runtime.GOOS + "-" + runtime.GOARCH
	needsDownload, reason := shouldDownload(root, Version, platform)

	if !needsDownload {
		slog.Info("Library is up to date, skipping download", "version", Version, "platform", platform)
		return nil
	}

	slog.Info("Download required", "reason", reason, "version", Version, "platform", platform)

	projectId := "69095259" // https://gitlab.com/gitlab-org/rust/gitlab-code-parser
	url := fmt.Sprintf("https://gitlab.com/api/v4/projects/%s/packages/generic/release/%s/libparser_c_bindings-%s.tar.gz", projectId, Version, platform)

	slog.Info("Starting download",
		"version", Version,
		"platform", platform,
		"target", targetDir)

	if err := downloadAndExtract(ctx, url, root); err != nil {
		return fmt.Errorf("downloading and extracting library: %w", err)
	}

	// Write metadata file after successful download
	if err := writeMetadata(root, Version, platform); err != nil {
		slog.Warn("Failed to write metadata file", "error", err)
		// Don't fail the operation just because metadata writing failed
	}

	slog.Info("Successfully downloaded and extracted library", "path", targetDir)
	return nil
}

// shouldDownload determines if we need to download the library and provides a reason
func shouldDownload(root, expectedVersion, expectedPlatform string) (bool, string) {
	libraryPath := filepath.Join(root, "lib", "libparser_c_bindings.a")
	metadataPath := filepath.Join(root, "metadata.json")

	// Check if library file exists
	if _, err := os.Stat(libraryPath); errors.Is(err, fs.ErrNotExist) {
		return true, "library file does not exist"
	}

	// Check if metadata file exists
	metadataBytes, err := os.ReadFile(metadataPath)
	if err != nil {
		if errors.Is(err, fs.ErrNotExist) {
			return true, "metadata file does not exist (legacy install)"
		}
		return true, fmt.Sprintf("failed to read metadata file: %v", err)
	}

	// Parse metadata
	var metadata Metadata
	if err := json.Unmarshal(metadataBytes, &metadata); err != nil {
		return true, fmt.Sprintf("failed to parse metadata file: %v", err)
	}

	// Check version mismatch
	if metadata.Version != expectedVersion {
		return true, fmt.Sprintf("version mismatch: have %s, need %s", metadata.Version, expectedVersion)
	}

	// Check platform mismatch
	if metadata.Platform != expectedPlatform {
		return true, fmt.Sprintf("platform mismatch: have %s, need %s", metadata.Platform, expectedPlatform)
	}

	return false, ""
}

// writeMetadata creates a metadata file with download information
func writeMetadata(root, version, platform string) error {
	metadata := Metadata{
		Version:  version,
		Platform: platform,
		Date:     time.Now().UTC().Format(time.RFC3339),
	}

	metadataBytes, err := json.MarshalIndent(metadata, "", "  ")
	if err != nil {
		return fmt.Errorf("marshaling metadata: %w", err)
	}

	metadataPath := filepath.Join(root, "metadata.json")
	if err := os.WriteFile(metadataPath, metadataBytes, 0644); err != nil {
		return fmt.Errorf("writing metadata file: %w", err)
	}

	return nil
}

func downloadAndExtract(ctx context.Context, url string, root string) (retErr error) {
	ctx, cancel := context.WithTimeout(ctx, 10*time.Minute)
	defer cancel()

	client := &http.Client{
		Timeout: 5 * time.Minute,
	}

	req, err := http.NewRequestWithContext(ctx, "GET", url, nil)
	if err != nil {
		return fmt.Errorf("creating request: %w", err)
	}

	slog.Info("Downloading file", "url", url)
	resp, err := client.Do(req)
	if err != nil {
		return fmt.Errorf("downloading file: %w", err)
	}
	defer func() {
		err := resp.Body.Close()
		if retErr == nil && err != nil {
			retErr = fmt.Errorf("closing file: %w", err)
		}
	}()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("HTTP %d: %s", resp.StatusCode, resp.Status)
	}

	// Log content length if available
	if resp.ContentLength > 0 {
		slog.Info("Download started", "size_bytes", resp.ContentLength)
	}

	gzr, err := gzip.NewReader(resp.Body)
	if err != nil {
		return fmt.Errorf("creating gzip reader: %w", err)
	}
	defer func() {
		err := gzr.Close()
		if err != nil && retErr == nil {
			retErr = fmt.Errorf("close gzip: %w", err)
		}
	}()

	tarReader := tar.NewReader(gzr)
	slog.Info("Extracting file")
	for {
		header, err := tarReader.Next()
		if err == io.EOF {
			break
		}
		if err != nil {
			return fmt.Errorf("reading tar: %w", err)
		}

		if err := extractTarHeader(tarReader, header, root); err != nil {
			return err
		}
	}
	return nil
}

func extractTarHeader(tarReader *tar.Reader, header *tar.Header, root string) (retErr error) {
	targetPath := filepath.Join(root, header.Name)
	switch header.Typeflag {
	case tar.TypeDir:
		err := os.Mkdir(targetPath, 0755)
		if err != nil && !errors.Is(err, fs.ErrExist) {
			return fmt.Errorf("mkdirall: %q: %w", targetPath, err)
		}
	case tar.TypeReg:
		outFile, err := os.Create(targetPath)
		if err != nil {
			return fmt.Errorf("create file: %w", err)
		}
		defer func() {
			err := outFile.Close()
			if retErr == nil && err != nil {
				retErr = fmt.Errorf("close file: %w", err)
			}
		}()

		if _, err := io.Copy(outFile, tarReader); err != nil {
			return fmt.Errorf("copy: %w", err)
		}
	default:
		return fmt.Errorf("unknown type: %c in %s", header.Typeflag, header.Name)
	}

	return nil
}
