/*******************************************************************************
 * Copyright (c) 2000, 2024 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.search.tests.filesearch;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.HashSet;
import java.util.Iterator;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.IAnnotationModel;

import org.eclipse.ui.PlatformUI;

import org.eclipse.ui.texteditor.AnnotationTypeLookup;
import org.eclipse.ui.texteditor.ITextEditor;

import org.eclipse.ui.editors.text.EditorsUI;

import org.eclipse.search.internal.ui.SearchPlugin;
import org.eclipse.search.internal.ui.text.FileMatch;
import org.eclipse.search.internal.ui.text.FileSearchQuery;
import org.eclipse.search.internal.ui.text.FileSearchResult;
import org.eclipse.search.tests.SearchTestUtil;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.search.ui.text.AbstractTextSearchResult;
import org.eclipse.search.ui.text.FileTextSearchScope;
import org.eclipse.search.ui.text.Match;

import org.eclipse.search2.internal.ui.InternalSearchUI;
import org.eclipse.search2.internal.ui.SearchView;
import org.eclipse.search2.internal.ui.text.EditorAnnotationManager;

public class AnnotationManagerTest {
	private FileSearchQuery fQuery1;
	private FileSearchQuery fQuery2;

	private final AnnotationTypeLookup fAnnotationTypeLookup= EditorsUI.getAnnotationTypeLookup();

	@RegisterExtension
	static JUnitSourceSetup fgJUnitSource= new JUnitSourceSetup();

	@BeforeEach
	public void setUp() {
		SearchTestUtil.ensureWelcomePageClosed();
		EditorAnnotationManager.debugSetHighlighterType(EditorAnnotationManager.HIGHLIGHTER_ANNOTATION);
		String[] fileNamePattern= { "*.java" };
		FileTextSearchScope scope= FileTextSearchScope.newWorkspaceScope(fileNamePattern, false);
		fQuery1= new FileSearchQuery("Test", false, true, scope);  //$NON-NLS-1$
		fQuery2= new FileSearchQuery("Test", false, true, scope); //$NON-NLS-1$
	}

	@AfterEach
	public void tearDown() {
		InternalSearchUI.getInstance().removeAllQueries();
		fQuery1= null;
		fQuery2= null;

		EditorAnnotationManager.debugSetHighlighterType(EditorAnnotationManager.HIGHLLIGHTER_ANY);

	}

	@Test
	public void testAddAnnotation() throws Exception {
		NewSearchUI.runQueryInForeground(null, fQuery1);
		AbstractTextSearchResult result= (AbstractTextSearchResult) fQuery1.getSearchResult();
		Object[] files= result.getElements();
		try {
			for (Object f : files) {
				IFile file = (IFile) f;
				ITextEditor editor= (ITextEditor)SearchTestUtil.openTextEditor(SearchPlugin.getActivePage(), file);
				IAnnotationModel annotationModel= editor.getDocumentProvider().getAnnotationModel(editor.getEditorInput());
				annotationModel.getAnnotationIterator();
				HashSet<Position> positions= new HashSet<>();
				for (Iterator<Annotation> iter= annotationModel.getAnnotationIterator(); iter.hasNext();) {
					Annotation annotation= iter.next();
					if (annotation.getType().equals(fAnnotationTypeLookup.getAnnotationType(NewSearchUI.SEARCH_MARKER, IMarker.SEVERITY_INFO))) {
						positions.add(annotationModel.getPosition(annotation));
					}
				}

				Match[] matches= result.getMatches(file);
				for (int j= 0; j < matches.length; j++) {
					Position position= new Position(matches[j].getOffset(), matches[j].getLength());
					assertTrue(positions.remove(position), "position not found at: "+j); //$NON-NLS-1$
				}
				assertEquals(0, positions.size());

			}
		} finally {
			SearchPlugin.getActivePage().closeAllEditors(false);
		}
	}

	@Test
	public void testBogusAnnotation() throws Exception {
		NewSearchUI.runQueryInForeground(null, fQuery1);
		FileSearchResult result= (FileSearchResult) fQuery1.getSearchResult();
		IFile file= (IFile) result.getElements()[0];
		SearchTestUtil.openTextEditor(PlatformUI.getWorkbench().getWorkbenchWindows()[0].getPages()[0], file);
		result.addMatch(new FileMatch(file));
	}

	@Test
	public void testRemoveQuery() throws Exception {
		NewSearchUI.runQueryInForeground(null, fQuery1);
		AbstractTextSearchResult result= (AbstractTextSearchResult) fQuery1.getSearchResult();
		Object[] files= result.getElements();
		InternalSearchUI.getInstance().removeQuery(fQuery1);

		try {
			for (Object f : files) {
				IFile file = (IFile) f;
				ITextEditor editor= (ITextEditor)SearchTestUtil.openTextEditor(SearchPlugin.getActivePage(), file);
				IAnnotationModel annotationModel= editor.getDocumentProvider().getAnnotationModel(editor.getEditorInput());
				int annotationCount= 0;
				for (Iterator<Annotation> annotations= annotationModel.getAnnotationIterator(); annotations.hasNext();) {
					Annotation annotation= annotations.next();
					if (annotation.getType().equals(fAnnotationTypeLookup.getAnnotationType(NewSearchUI.SEARCH_MARKER, IMarker.SEVERITY_INFO))) {
						annotationCount++;
					}
				}
				assertEquals(0, annotationCount);
			}
		} finally {
			SearchPlugin.getActivePage().closeAllEditors(false);
		}
	}

	@Test
	public void testReplaceQuery() throws Exception {
		NewSearchUI.runQueryInForeground(null, fQuery1);
		AbstractTextSearchResult result= (AbstractTextSearchResult) fQuery1.getSearchResult();
		Object[] files= result.getElements();
		NewSearchUI.runQueryInForeground(null, fQuery2);
		try {
			for (Object f : files) {
				IFile file = (IFile) f;
				ITextEditor editor= (ITextEditor)SearchTestUtil.openTextEditor(SearchPlugin.getActivePage(), file);
				IAnnotationModel annotationModel= editor.getDocumentProvider().getAnnotationModel(editor.getEditorInput());
				IDocument document= editor.getDocumentProvider().getDocument(editor.getEditorInput());
				for (Iterator<Annotation> annotations= annotationModel.getAnnotationIterator(); annotations.hasNext();) {
					Annotation annotation= annotations.next();
					if (annotation.getType().equals(fAnnotationTypeLookup.getAnnotationType(NewSearchUI.SEARCH_MARKER, IMarker.SEVERITY_INFO))) {
						Position p= annotationModel.getPosition(annotation);
						String text= document.get(p.getOffset(), p.getLength());
						assertTrue(text.equalsIgnoreCase(fQuery2.getSearchString()));
					}
				}
			}
		} finally {
			SearchPlugin.getActivePage().closeAllEditors(false);
		}
	}

	@Test
	public void testSwitchQuery() throws Exception {
		NewSearchUI.runQueryInForeground(null, fQuery1);
		AbstractTextSearchResult result= (AbstractTextSearchResult) fQuery1.getSearchResult();
		Object[] files= result.getElements();
		NewSearchUI.runQueryInForeground(null, fQuery2);

		SearchView activateSearchResultView= (SearchView) NewSearchUI.activateSearchResultView();
		activateSearchResultView.showSearchResult(result);
		try {
			for (Object f : files) {
				IFile file = (IFile) f;
				ITextEditor editor= (ITextEditor)SearchTestUtil.openTextEditor(SearchPlugin.getActivePage(), file);
				IAnnotationModel annotationModel= editor.getDocumentProvider().getAnnotationModel(editor.getEditorInput());
				IDocument document= editor.getDocumentProvider().getDocument(editor.getEditorInput());
				for (Iterator<Annotation> annotations= annotationModel.getAnnotationIterator(); annotations.hasNext();) {
					Annotation annotation= annotations.next();
					if (annotation.getType().equals(fAnnotationTypeLookup.getAnnotationType(NewSearchUI.SEARCH_MARKER, IMarker.SEVERITY_INFO))) {
						Position p= annotationModel.getPosition(annotation);
						String text= document.get(p.getOffset(), p.getLength());
						assertTrue(text.equalsIgnoreCase(fQuery1.getSearchString()));
					}
				}
			}
		} finally {
			SearchPlugin.getActivePage().closeAllEditors(false);
		}
	}

}
