/*******************************************************************************
 * Copyright (c) 2007, 2018 Wind River Systems, Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 * Michael Scharf (Wind River) - initial API and implementation
 *******************************************************************************/
package org.eclipse.terminal.internal.test.terminalcanvas;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.ScrollBar;

public class TerminalTextCanvas extends Canvas {
	Image image;
	Point origin = new Point(0, 0);

	public TerminalTextCanvas(Composite parent, int style) {
		super(parent, SWT.NO_BACKGROUND | SWT.NO_REDRAW_RESIZE | SWT.V_SCROLL | SWT.H_SCROLL | style);
		loadImage(parent);

		final ScrollBar hBar = getHorizontalBar();
		hBar.addListener(SWT.Selection, e -> {
			int hSelection = hBar.getSelection();
			int destX = -hSelection - origin.x;
			Rectangle rect = image.getBounds();
			scroll(destX, 0, 0, 0, rect.width, rect.height, false);
			origin.x = -hSelection;
		});
		final ScrollBar vBar = getVerticalBar();
		vBar.addListener(SWT.Selection, e -> {
			int vSelection = vBar.getSelection();
			int destY = -vSelection - origin.y;
			Rectangle rect = image.getBounds();
			scroll(0, destY, 0, 0, rect.width, rect.height, false);
			origin.y = -vSelection;
		});
		addListener(SWT.Resize, e -> {
			Rectangle rect = image.getBounds();
			Rectangle client = getClientArea();
			hBar.setMaximum(rect.width);
			vBar.setMaximum(rect.height);
			hBar.setThumb(Math.min(rect.width, client.width));
			vBar.setThumb(Math.min(rect.height, client.height));
			int hPage = rect.width - client.width;
			int vPage = rect.height - client.height;
			int hSelection = hBar.getSelection();
			int vSelection = vBar.getSelection();
			if (hSelection >= hPage) {
				if (hPage <= 0)
					hSelection = 0;
				origin.x = -hSelection;
			}
			if (vSelection >= vPage) {
				if (vPage <= 0)
					vSelection = 0;
				origin.y = -vSelection;
			}
			redraw();
		});
		addListener(SWT.Paint, e -> {
			GC gc = e.gc;
			System.out.println(gc.getClipping() + " " + origin);
			gc.drawImage(image, origin.x, origin.y);
			Rectangle rect = image.getBounds();
			Rectangle client = getClientArea();
			int marginWidth = client.width - rect.width;
			if (marginWidth > 0) {
				gc.fillRectangle(rect.width, 0, marginWidth, client.height);
			}
			int marginHeight = client.height - rect.height;
			if (marginHeight > 0) {
				gc.fillRectangle(0, rect.height, client.width, marginHeight);
			}
		});
	}

	private void loadImage(Composite parent) {
		FileDialog dialog = new FileDialog(parent.getShell(), SWT.OPEN);
		dialog.setText("Open an image file or cancel");
		String string = dialog.open();
		if (string != null) {
			image = new Image(getDisplay(), string);
		}
	}

}
