/*******************************************************************************
 *  Copyright (c) 2010, 2015 Broadcom Corporation and others.
 *
 *  This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License 2.0
 *  which accompanies this distribution, and is available at
 *  https://www.eclipse.org/legal/epl-2.0/
 *
 *  SPDX-License-Identifier: EPL-2.0
 *
 *  Contributors:
 *     James Blackburn (Broadcom Corp.) - initial API and implementation
 *     Alexander Kurtakov <akurtako@redhat.com> - Bug 459343
 *******************************************************************************/
package org.eclipse.core.tests.internal.builders;

import static org.assertj.core.api.Assertions.assertThat;
import static org.eclipse.core.resources.ResourcesPlugin.getWorkspace;
import static org.eclipse.core.tests.resources.ResourceTestUtil.createInWorkspace;
import static org.eclipse.core.tests.resources.ResourceTestUtil.createTestMonitor;
import static org.eclipse.core.tests.resources.ResourceTestUtil.setAutoBuilding;
import static org.eclipse.core.tests.resources.ResourceTestUtil.updateProjectDescription;
import static org.eclipse.core.tests.resources.ResourceTestUtil.waitForEncodingRelatedJobs;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.lang.Thread.State;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.eclipse.core.internal.events.ResourceDelta;
import org.eclipse.core.internal.resources.ContentDescriptionManager;
import org.eclipse.core.internal.resources.Workspace;
import org.eclipse.core.resources.IBuildConfiguration;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.ILogListener;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.MultiRule;
import org.eclipse.core.tests.harness.TestBarrier2;
import org.eclipse.core.tests.internal.builders.TestBuilder.BuilderRuleCallback;
import org.eclipse.core.tests.resources.TestUtil;
import org.eclipse.core.tests.resources.util.WorkspaceResetExtension;
import org.junit.function.ThrowingRunnable;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.junit.jupiter.api.extension.ExtendWith;

/**
 * This class tests extended functionality (since 3.6) which allows
 * builders to be run with reduced (non-Workspace) scheduling rules.
 *
 * When one of these builders runs, other threads may modify the workspace
 * depending on the builder's scheduling rule
 */
@ExtendWith(WorkspaceResetExtension.class)
public class RelaxedSchedRuleBuilderTest {
	private ErrorLogging errorLogging = new ErrorLogging();

	@BeforeEach
	public void setUp() throws Exception {
		errorLogging.enable();
	}

	@AfterEach
	public void tearDown() throws Exception {
		errorLogging.disable();
		TestBuilder builder = DeltaVerifierBuilder.getInstance();
		if (builder != null) {
			builder.reset();
		}
		builder = EmptyDeltaBuilder.getInstance();
		if (builder != null) {
			builder.reset();
		}
		builder = EmptyDeltaBuilder2.getInstance();
		if (builder != null) {
			builder.reset();
		}
	}

	/**
	 * Test a simple builder with a relaxed scheduling rule
	 */
	@Test
	public void testBasicRelaxedSchedulingRules() throws Throwable {
		String projectName = "TestRelaxed";
		setAutoBuilding(false);
		final IProject project = getWorkspace().getRoot().getProject(projectName);
		createInWorkspace(project);
		updateProjectDescription(project).addingCommand(EmptyDeltaBuilder.BUILDER_NAME).withTestBuilderId("testbuild")
				.apply();

		// Ensure the builder is instantiated
		project.build(IncrementalProjectBuilder.CLEAN_BUILD, createTestMonitor());

		final TestBarrier2 tb = new TestBarrier2(TestBarrier2.STATUS_WAIT_FOR_START);

		// Create a builder set a null scheduling rule
		EmptyDeltaBuilder builder = EmptyDeltaBuilder.getInstance();
		builder.setRuleCallback(new BuilderRuleCallback() {
			@Override
			public ISchedulingRule getRule(String name, IncrementalProjectBuilder projectBuilder, int trigger,
					Map<String, String> args) {
				return null;
			}

			@Override
			public IProject[] build(int kind, Map<String, String> args, IProgressMonitor monitor) throws CoreException {
				assertTrue(Job.getJobManager().currentRule() == null);
				tb.setStatus(TestBarrier2.STATUS_START);
				while (!monitor.isCanceled()) {
					try {
						Thread.sleep(10);
					} catch (InterruptedException e) {
						// Don't care
					}
				}
				tb.setStatus(TestBarrier2.STATUS_DONE);
				return super.build(kind, args, monitor);
			}
		});

		final AtomicReference<ThrowingRunnable> exceptionInMainThreadCallback = new AtomicReference<>(Function::identity);
		Job j = new Job("build job") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					project.build(IncrementalProjectBuilder.FULL_BUILD, monitor);
				} catch (CoreException e) {
					exceptionInMainThreadCallback.set(() -> {
						throw e;
					});
				}
				return Status.OK_STATUS;
			}
		};
		j.schedule();

		// wait for build to be called
		tb.waitForStatus(TestBarrier2.STATUS_START);

		// Should be able to write a file in the project
		project.getFile("foo.c").create(null, true, createTestMonitor());
		assertTrue(project.getFile("foo.c").exists());

		// Cancel the builder
		j.cancel();
		tb.waitForStatus(TestBarrier2.STATUS_DONE);

		exceptionInMainThreadCallback.get().run();
	}

	/**
	 * Test two builders, each with relaxed scheduling rules, run in the same build operation
	 * Each should be run with their own scheduling rule.
	 * Tests:
	 *     Bug 306824 - null scheduling rule and non-null scheduling rule don't work together
	 *     Builders should have separate scheduling rules
	 */
	@Test
	public void testTwoBuildersRunInOneBuild() throws Throwable {
		String projectName = "testTwoBuildersRunInOneBuild";
		setAutoBuilding(false);
		final IProject project = getWorkspace().getRoot().getProject(projectName);
		createInWorkspace(project);

		updateProjectDescription(project) //
				.addingCommand(EmptyDeltaBuilder.BUILDER_NAME).withTestBuilderId("Project1Build1") //
				.andCommand(EmptyDeltaBuilder2.BUILDER_NAME).withTestBuilderId("Project1Build2").apply();

		// Ensure the builder is instantiated
		project.build(IncrementalProjectBuilder.CLEAN_BUILD, createTestMonitor());

		final TestBarrier2 tb1 = new TestBarrier2(TestBarrier2.STATUS_WAIT_FOR_START);
		final TestBarrier2 tb2 = new TestBarrier2(TestBarrier2.STATUS_WAIT_FOR_START);

		// Create a builder set a null scheduling rule
		EmptyDeltaBuilder builder = EmptyDeltaBuilder.getInstance();
		EmptyDeltaBuilder2 builder2 = EmptyDeltaBuilder2.getInstance();

		// Set the rule call-back
		builder.setRuleCallback(new BuilderRuleCallback() {
			@Override
			public ISchedulingRule getRule(String name, IncrementalProjectBuilder projectBuilder, int trigger,
					Map<String, String> args) {
				tb1.setStatus(TestBarrier2.STATUS_START);
				return project;
			}

			@Override
			public IProject[] build(int kind, Map<String, String> args, IProgressMonitor monitor) throws CoreException {
				// shared scheduling rule
				assertTrue(Job.getJobManager().currentRule().contains(project));
				tb1.setStatus(TestBarrier2.STATUS_RUNNING);
				tb1.waitForStatus(TestBarrier2.STATUS_WAIT_FOR_DONE);
				tb1.setStatus(TestBarrier2.STATUS_DONE);
				return super.build(kind, args, monitor);
			}
		});
		// Set the rule call-back
		builder2.setRuleCallback(new BuilderRuleCallback() {
			@Override
			public ISchedulingRule getRule(String name, IncrementalProjectBuilder projectBuilder, int trigger,
					Map<String, String> args) {
				// get rule is called before starting
				tb2.setStatus(TestBarrier2.STATUS_START);
				return null;
			}

			@Override
			public IProject[] build(int kind, Map<String, String> args, IProgressMonitor monitor) throws CoreException {
				// shared scheduling rule
				assertTrue(Job.getJobManager().currentRule() == null || Job.getJobManager().currentRule().contains(getWorkspace().getRoot()));
				tb2.setStatus(TestBarrier2.STATUS_RUNNING);
				tb2.waitForStatus(TestBarrier2.STATUS_WAIT_FOR_DONE);
				tb2.setStatus(TestBarrier2.STATUS_DONE);
				return super.build(kind, args, monitor);
			}
		});

		final AtomicReference<ThrowingRunnable> exceptionInMainThreadCallback = new AtomicReference<>(
				Function::identity);
		// Run the build
		Job j = new Job("build job1") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					project.build(IncrementalProjectBuilder.FULL_BUILD, monitor);
				} catch (CoreException e) {
					exceptionInMainThreadCallback.set(() -> {
						throw e;
					});
				}
				return Status.OK_STATUS;
			}
		};
		j.schedule();

		// Wait for the build to transition

		tb1.waitForStatus(TestBarrier2.STATUS_RUNNING);
		tb1.setStatus(TestBarrier2.STATUS_WAIT_FOR_DONE);
		tb1.waitForStatus(TestBarrier2.STATUS_DONE);

		tb2.waitForStatus(TestBarrier2.STATUS_RUNNING);
		tb2.setStatus(TestBarrier2.STATUS_WAIT_FOR_DONE);
		tb2.waitForStatus(TestBarrier2.STATUS_DONE);

		exceptionInMainThreadCallback.get().run();
	}

	HashSet<ISchedulingRule> getRulesAsSet(ISchedulingRule rule) {
		HashSet<ISchedulingRule> rules = new HashSet<>();
		if (rule == null) {
			return rules;
		}
		if (rule instanceof MultiRule mRule) {
			rules.addAll(Arrays.asList(mRule.getChildren()));
		} else {
			rules.add(rule);
		}
		return rules;
	}

	/**
	 * As the builder is run with a relaxed scheduling rule, we ensure that any changes made before
	 * the build is actually run are present in the delta.
	 * Acquiring the scheduling rule must be done outside of the WS lock, so this tests that
	 * a change which sneaks in during the window or the build thread acquiring its scheduling
	 * rule, is correctly present in the builder's delta.
	 */
	@Test
	public void testBuilderDeltaUsingRelaxedRuleBug343256(TestInfo testInfo) throws Throwable {
		final int timeout = 10000;
		String projectName = "testBuildDeltaUsingRelaxedRuleBug343256";
		setAutoBuilding(false);
		final IProject project = getWorkspace().getRoot().getProject(projectName);
		final IFile foo = project.getFile("foo");
		createInWorkspace(project);

		String testMethodName = testInfo.getTestMethod().get().getName();
		waitForEncodingRelatedJobs(testMethodName);
		waitForContentDescriptionUpdate(testMethodName);
		// wait for noBuildJob so POST_BUILD will fire
		((Workspace) getWorkspace()).getBuildManager().waitForAutoBuildOff();

		updateProjectDescription(project).addingCommand(EmptyDeltaBuilder.BUILDER_NAME)
				.withTestBuilderId("Project1Build1").apply();

		// Ensure the builder is instantiated
		project.build(IncrementalProjectBuilder.FULL_BUILD, createTestMonitor());

		final TestBarrier2 tb = new TestBarrier2(TestBarrier2.STATUS_WAIT_FOR_START);
		AtomicReference<Throwable> errorInBuildTriggeringJob = new AtomicReference<>();
		AtomicReference<Throwable> errorInWorkspaceChangingJob = new AtomicReference<>();

		Job workspaceChangingJob = new Job("Workspace Changing Job") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				tb.setStatus(TestBarrier2.STATUS_WAIT_FOR_RUN);
				try {
					createInWorkspace(foo);
				} catch (CoreException e) {
					errorInWorkspaceChangingJob.set(e);
				}
				return Status.OK_STATUS;
			}
		};

		Job buildTriggeringJob = new Job("Build Triggering Job") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					getWorkspace().build(new IBuildConfiguration[] { project.getActiveBuildConfig() },
							IncrementalProjectBuilder.INCREMENTAL_BUILD, true, monitor);
				} catch (CoreException e) {
					IStatus status = e.getStatus();
					IStatus[] children = status.getChildren();
					if (children.length > 0) {
						errorInBuildTriggeringJob.set(children[0].getException());
					} else {
						errorInBuildTriggeringJob.set(e);
					}
				}
				return Status.OK_STATUS;
			}
		};

		// Create a builder set a null scheduling rule
		EmptyDeltaBuilder builder = EmptyDeltaBuilder.getInstance();

		// Set the rule call-back
		builder.setRuleCallback(new BuilderRuleCallback() {

			boolean called = false;

			@Override
			public ISchedulingRule getRule(String name, IncrementalProjectBuilder projectBuilder, int trigger,
					Map<String, String> args) {
				// Remove once Bug 331187 is fixed.
				// Currently #getRule is called twice when building a specific build configuration (so as to minimized change in
				// 3.7 end-game.  As this test is trying to provoke a bug in the window between fetching a rule and applying it
				// to the build, we don't want to run the first time #getRule is called (in Workspace#build)
				if (!called) {
					called = true;
					return project;
				}
				tb.setStatus(TestBarrier2.STATUS_START);
				tb.waitForStatus(TestBarrier2.STATUS_WAIT_FOR_RUN);
				// Wait for workspace changing job acquiring lock to ensure that it performs
				// workspace change after getRule released workspace lock and before build
				// re-acquired it
				boolean workspaceChangingJobAcquiringLock = waitForThreadStateWaiting(workspaceChangingJob.getThread());
				assertTrue(workspaceChangingJobAcquiringLock,
						"timed out waiting for workspace changing job to wait for workspace lock");
				return project;
			}

			private boolean waitForThreadStateWaiting(Thread threadToWaitFor) {
				long startWaitingTime = System.currentTimeMillis();
				while (!(threadToWaitFor.getState() == State.TIMED_WAITING
						|| threadToWaitFor.getState() == State.WAITING)) {
					long elapsedWaitingTime = System.currentTimeMillis() - startWaitingTime;
					if (elapsedWaitingTime > timeout) {
						return false;
					}
				}
				return true;
			}

			@Override
			public IProject[] build(int kind, Map<String, String> args, IProgressMonitor monitor) throws CoreException {
				// shared scheduling rule
				assertTrue(Job.getJobManager().currentRule().equals(project));
				// assert that the delta contains the file foo
				IResourceDelta delta = getDelta(project);
				assertNotNull(delta, "no workspace change occurred between getRule and build");
				assertEquals(1, delta.getAffectedChildren().length,
						"unexpected number of changes occurred: " + ((ResourceDelta) delta).toDeepDebugString());
				assertEquals(foo, delta.getAffectedChildren()[0].getResource(), "unexpected resource was changed");
				tb.setStatus(TestBarrier2.STATUS_DONE);
				return super.build(kind, args, monitor);
			}
		});

		// buildTriggeringJob runs, builds, calls BuilderRuleCallback, that waits
		// for workspaceChangingJob to wait for the workspace lock, such that it
		// performs a workspace change between getRule and build
		buildTriggeringJob.schedule();
		// Wait for the build to transition to getRule
		tb.waitForStatus(TestBarrier2.STATUS_START);
		// Modify a file in the project
		workspaceChangingJob.schedule();

		workspaceChangingJob.join(timeout, null);
		buildTriggeringJob.join(timeout, null);
		if (errorInBuildTriggeringJob.get() != null) {
			throw errorInBuildTriggeringJob.get();
		}
		if (errorInWorkspaceChangingJob.get() != null) {
			throw errorInWorkspaceChangingJob.get();
		}
		tb.waitForStatus(TestBarrier2.STATUS_DONE);
		errorLogging.assertNoErrorsLogged();
	}

	private void waitForContentDescriptionUpdate(String testMethodName) {
		TestUtil.waitForJobs(testMethodName, 10, 5_000,
				ContentDescriptionManager.FAMILY_DESCRIPTION_CACHE_FLUSH);
	}

	/**
	 * Tests for regression in running the build with reduced scheduling rules.
	 */
	@Test
	public void testBug343256() throws Throwable {
		String projectName = "testBug343256";
		setAutoBuilding(false);
		final IProject project = getWorkspace().getRoot().getProject(projectName);
		createInWorkspace(project);
		updateProjectDescription(project) //
				.addingCommand(EmptyDeltaBuilder.BUILDER_NAME).withTestBuilderId("Project1Build1")
				.andCommand(EmptyDeltaBuilder2.BUILDER_NAME).withTestBuilderId("Project1Build2").apply();

		// Ensure the builder is instantiated
		project.build(IncrementalProjectBuilder.CLEAN_BUILD, createTestMonitor());

		final TestBarrier2 tb1 = new TestBarrier2(TestBarrier2.STATUS_WAIT_FOR_START);
		final TestBarrier2 tb2 = new TestBarrier2(TestBarrier2.STATUS_WAIT_FOR_START);

		// Scheduling rules to returng from #getRule
		final ISchedulingRule[] getRules = new ISchedulingRule[2];
		final ISchedulingRule[] buildRules = new ISchedulingRule[2];

		// Create a builder set a null scheduling rule
		EmptyDeltaBuilder builder = EmptyDeltaBuilder.getInstance();
		EmptyDeltaBuilder2 builder2 = EmptyDeltaBuilder2.getInstance();

		// Set the rule call-back
		builder.setRuleCallback(new BuilderRuleCallback() {
			@Override
			public ISchedulingRule getRule(String name, IncrementalProjectBuilder projectBuilder, int trigger,
					Map<String, String> args) {
				tb1.waitForStatus(TestBarrier2.STATUS_START);
				return getRules[0];
			}

			@Override
			public IProject[] build(int kind, Map<String, String> args, IProgressMonitor monitor) throws CoreException {
				HashSet<ISchedulingRule> h1 = getRulesAsSet(Job.getJobManager().currentRule());
				HashSet<ISchedulingRule> h2 = getRulesAsSet(buildRules[0]);
				// shared scheduling rule
				assertTrue(h1.equals(h2));
				tb1.setStatus(TestBarrier2.STATUS_DONE);
				return super.build(kind, args, monitor);
			}
		});
		// Set the rule call-back
		builder2.setRuleCallback(new BuilderRuleCallback() {
			@Override
			public ISchedulingRule getRule(String name, IncrementalProjectBuilder projectBuilder, int trigger,
					Map<String, String> args) {
				tb2.waitForStatus(TestBarrier2.STATUS_START);
				return getRules[1];
			}

			@Override
			public IProject[] build(int kind, Map<String, String> args, IProgressMonitor monitor) throws CoreException {
				HashSet<ISchedulingRule> h1 = getRulesAsSet(Job.getJobManager().currentRule());
				HashSet<ISchedulingRule> h2 = getRulesAsSet(buildRules[1]);
				assertTrue(h1.equals(h2));
				tb2.setStatus(TestBarrier2.STATUS_DONE);
				return super.build(kind, args, monitor);
			}
		});

		Job j;

		// Enable for Bug 331187
		//		// IProject.build()
		//		j = new Job("IProject.build()") {
		//			protected IStatus run(IProgressMonitor monitor) {
		//				try {
		//					project.build(IncrementalProjectBuilder.FULL_BUILD, monitor);
		//				} catch (CoreException e) {
		//					fail(e.toString());
		//				}
		//				return Status.OK_STATUS;
		//			}
		//		};
		//		invokeTestBug343256(project, getRules, buildRules, tb1, tb2, j);

		//		// IWorkspace.build()
		//		j = new Job("IWorkspace.build()") {
		//			protected IStatus run(IProgressMonitor monitor) {
		//				try {
		//					getWorkspace().build(IncrementalProjectBuilder.FULL_BUILD, monitor);
		//				} catch (CoreException e) {
		//					fail(e.toString());
		//				}
		//				return Status.OK_STATUS;
		//			}
		//		};
		//		invokeTestBug343256(project, getRules, buildRules, tb1, tb2, j);

		final AtomicReference<ThrowingRunnable> exceptionInMainThreadCallback = new AtomicReference<>(
				Function::identity);
		// IWorkspace.build(IBuildConfiguration[],...)
		j = new Job("IWorkspace.build(IBuildConfiguration[],...)") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					getWorkspace().build(new IBuildConfiguration[] {project.getActiveBuildConfig()}, IncrementalProjectBuilder.FULL_BUILD, true, monitor);
				} catch (CoreException e) {
					exceptionInMainThreadCallback.set(() -> {
						throw e;
					});
				}
				return Status.OK_STATUS;
			}
		};
		invokeTestBug343256(project, getRules, buildRules, tb1, tb2, j);

		exceptionInMainThreadCallback.get().run();
		// Test Auto-build
		//		j = new Job("Auto-build") {
		//			protected IStatus run(IProgressMonitor monitor) {
		//				try {
		//					getWorkspace().build(IncrementalProjectBuilder.CLEAN_BUILD, getMonitor());
		//				} catch (CoreException e) {
		//					fail(e.toString());
		//				}
		//				return Status.OK_STATUS;
		//			}
		//		};
		//		// Auto-build
		//		setAutoBuilding(true);
		//		// Wait for the build to transition
		//		invokeTestBug343256(project, getRules, buildRules, tb1, tb2, j);
	}

	/**
	 * Helper method do invoke a set of tests on Bug343256 using the different sets of builder API
	 */
	private void invokeTestBug343256(IProject project, ISchedulingRule[] getRules, ISchedulingRule[] buildRules, TestBarrier2 tb1, TestBarrier2 tb2, Job j) {
		// Test 1 - build project sched rule
		getRules[0] = getRules[1] = project;
		buildRules[0] = buildRules[1] = new MultiRule(new ISchedulingRule[] {getRules[0]});
		tb1.setStatus(TestBarrier2.STATUS_START);
		tb2.setStatus(TestBarrier2.STATUS_START);
		j.schedule();
		tb1.waitForStatus(TestBarrier2.STATUS_DONE);
		tb2.waitForStatus(TestBarrier2.STATUS_DONE);

		// Test 2 - build null rule
		getRules[0] = getRules[1] = null;
		buildRules[0] = buildRules[1] = null;
		tb1.setStatus(TestBarrier2.STATUS_START);
		tb2.setStatus(TestBarrier2.STATUS_START);
		j.schedule();
		tb1.waitForStatus(TestBarrier2.STATUS_DONE);
		tb2.waitForStatus(TestBarrier2.STATUS_DONE);

		// Test 3 - build mixed projects
		getRules[0] = buildRules[0] = project;
		getRules[1] = buildRules[1] = getWorkspace().getRoot().getProject("other");
		tb1.setStatus(TestBarrier2.STATUS_START);
		tb2.setStatus(TestBarrier2.STATUS_START);
		j.schedule();
		tb1.waitForStatus(TestBarrier2.STATUS_DONE);
		tb2.waitForStatus(TestBarrier2.STATUS_DONE);

		// Test 4 - build project + null
		getRules[0] = buildRules[0] = project;
		getRules[1] = buildRules[1] = null;
		// TODO: Fixed in Bug 331187 ; BuildManager#getRule is pessimistic when there's a mixed resource and null rule
		buildRules[0] = buildRules[1] = getWorkspace().getRoot();
		tb1.setStatus(TestBarrier2.STATUS_START);
		tb2.setStatus(TestBarrier2.STATUS_START);
		j.schedule();
		tb1.waitForStatus(TestBarrier2.STATUS_DONE);
		tb2.waitForStatus(TestBarrier2.STATUS_DONE);
	}

	private static class ErrorLogging {
		private final Queue<IStatus> loggedErrors = new ConcurrentLinkedQueue<>();

		private final ILogListener errorLogListener = (IStatus status, String plugin) -> {
			if (status.matches(IStatus.ERROR)) {
				loggedErrors.add(status);
			}
		};

		public void enable() {
			Platform.addLogListener(errorLogListener);
		}

		public void disable() {
			Platform.removeLogListener(errorLogListener);
		}

		public void assertNoErrorsLogged() {
			List<IStatus> errors = new ArrayList<>();
			loggedErrors.removeIf(errors::add);
			List<Throwable> thrownExceptions = errors.stream().map(IStatus::getException).filter(Objects::nonNull)
					.collect(Collectors.toList());
			assertThat(thrownExceptions).as("logged exceptions").isEmpty();
			assertThat(errors).as("logged errors").isEmpty();
		}
	}

}
