/*
 * Decompiled with CFR 0.152.
 */
package org.junit.vintage.engine.execution;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apiguardian.api.API;
import org.junit.platform.commons.logging.Logger;
import org.junit.platform.commons.logging.LoggerFactory;
import org.junit.platform.commons.util.ExceptionUtils;
import org.junit.platform.engine.CancellationToken;
import org.junit.platform.engine.ConfigurationParameters;
import org.junit.platform.engine.EngineExecutionListener;
import org.junit.platform.engine.TestDescriptor;
import org.junit.vintage.engine.descriptor.RunnerTestDescriptor;
import org.junit.vintage.engine.descriptor.VintageEngineDescriptor;
import org.junit.vintage.engine.execution.RunnerExecutor;

@API(status=API.Status.INTERNAL, since="5.12")
public class VintageExecutor {
    private static final Logger logger = LoggerFactory.getLogger(VintageExecutor.class);
    private static final int DEFAULT_THREAD_POOL_SIZE = Runtime.getRuntime().availableProcessors();
    private static final int SHUTDOWN_TIMEOUT_SECONDS = 30;
    private final VintageEngineDescriptor engineDescriptor;
    private final EngineExecutionListener engineExecutionListener;
    private final ConfigurationParameters configurationParameters;
    private final boolean parallelExecutionEnabled;
    private final boolean classes;
    private final boolean methods;

    public VintageExecutor(VintageEngineDescriptor engineDescriptor, EngineExecutionListener engineExecutionListener, ConfigurationParameters configurationParameters) {
        this.engineDescriptor = engineDescriptor;
        this.engineExecutionListener = engineExecutionListener;
        this.configurationParameters = configurationParameters;
        this.parallelExecutionEnabled = configurationParameters.getBoolean("junit.vintage.execution.parallel.enabled").orElse(false);
        this.classes = configurationParameters.getBoolean("junit.vintage.execution.parallel.classes").orElse(false);
        this.methods = configurationParameters.getBoolean("junit.vintage.execution.parallel.methods").orElse(false);
    }

    public void executeAllChildren(CancellationToken cancellationToken) {
        if (!this.parallelExecutionEnabled) {
            this.executeClassesAndMethodsSequentially(cancellationToken);
            return;
        }
        if (!this.classes && !this.methods) {
            logger.warn(() -> "Parallel execution is enabled but no scope is defined. Falling back to sequential execution.");
            this.executeClassesAndMethodsSequentially(cancellationToken);
            return;
        }
        boolean wasInterrupted = this.executeInParallel(cancellationToken);
        if (wasInterrupted) {
            Thread.currentThread().interrupt();
        }
    }

    private void executeClassesAndMethodsSequentially(CancellationToken cancellationToken) {
        RunnerExecutor runnerExecutor = new RunnerExecutor(this.engineExecutionListener, cancellationToken);
        Iterator<TestDescriptor> iterator = this.engineDescriptor.getModifiableChildren().iterator();
        while (iterator.hasNext()) {
            runnerExecutor.execute((RunnerTestDescriptor)iterator.next());
            iterator.remove();
        }
    }

    private boolean executeInParallel(CancellationToken cancellationToken) {
        ExecutorService executorService = Executors.newWorkStealingPool(this.getThreadPoolSize());
        RunnerExecutor runnerExecutor = new RunnerExecutor(this.engineExecutionListener, cancellationToken);
        List<RunnerTestDescriptor> runnerTestDescriptors = this.collectRunnerTestDescriptors(executorService);
        if (!this.classes) {
            this.executeClassesSequentially(runnerTestDescriptors, runnerExecutor);
            return false;
        }
        return this.executeClassesInParallel(runnerTestDescriptors, runnerExecutor, executorService);
    }

    private int getThreadPoolSize() {
        Optional optionalPoolSize = this.configurationParameters.get("junit.vintage.execution.parallel.pool-size");
        if (optionalPoolSize.isPresent()) {
            try {
                int poolSize = Integer.parseInt((String)optionalPoolSize.get());
                if (poolSize > 0) {
                    return poolSize;
                }
                logger.warn(() -> "Invalid value for parallel pool size: " + poolSize);
            }
            catch (NumberFormatException e) {
                logger.warn(() -> "Invalid value for parallel pool size: " + (String)optionalPoolSize.get());
            }
        }
        return DEFAULT_THREAD_POOL_SIZE;
    }

    private List<RunnerTestDescriptor> collectRunnerTestDescriptors(ExecutorService executorService) {
        return this.engineDescriptor.getModifiableChildren().stream().map(RunnerTestDescriptor.class::cast).map(it -> this.methods ? this.parallelMethodExecutor((RunnerTestDescriptor)((Object)it), executorService) : it).toList();
    }

    private RunnerTestDescriptor parallelMethodExecutor(RunnerTestDescriptor runnerTestDescriptor, ExecutorService executorService) {
        runnerTestDescriptor.setExecutorService(executorService);
        return runnerTestDescriptor;
    }

    private void executeClassesSequentially(List<RunnerTestDescriptor> runnerTestDescriptors, RunnerExecutor runnerExecutor) {
        for (RunnerTestDescriptor runnerTestDescriptor : runnerTestDescriptors) {
            runnerExecutor.execute(runnerTestDescriptor);
        }
    }

    private boolean executeClassesInParallel(List<RunnerTestDescriptor> runnerTestDescriptors, RunnerExecutor runnerExecutor, ExecutorService executorService) {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        for (RunnerTestDescriptor runnerTestDescriptor : runnerTestDescriptors) {
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> runnerExecutor.execute(runnerTestDescriptor), executorService);
            futures.add(future);
        }
        CompletableFuture<Void> allOf = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
        boolean wasInterrupted = false;
        try {
            allOf.get();
        }
        catch (InterruptedException e) {
            logger.warn((Throwable)e, () -> "Interruption while waiting for parallel test execution to finish");
            wasInterrupted = true;
        }
        catch (ExecutionException e) {
            throw ExceptionUtils.throwAsUncheckedException((Throwable)Objects.requireNonNullElse(e.getCause(), e));
        }
        finally {
            this.shutdownExecutorService(executorService);
        }
        return wasInterrupted;
    }

    private void shutdownExecutorService(ExecutorService executorService) {
        try {
            executorService.shutdown();
            if (!executorService.awaitTermination(30L, TimeUnit.SECONDS)) {
                logger.warn(() -> "Executor service did not terminate within the specified timeout");
                executorService.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            logger.warn((Throwable)e, () -> "Interruption while waiting for executor service to shut down");
            Thread.currentThread().interrupt();
        }
    }
}

