///////////////////////////////////////////////////////////////////////////////
//                                                         
// Dragon.cc
// ---------
// Dragon implementation module
//                                               
// Design and Implementation by Bjoern Lemke
//
// (C)opyright 2000-2007 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// IMPLEMENTATION MODULE
//
// Class: Dragon 
// 
// Description: body module for the dragon parser generater 
//
///////////////////////////////////////////////////////////////////////////////

// BASE INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/Chain.h>
#include <lfcbase/File.h>
#include <lfcbase/Tokenizer.h>
#include <lfcbase/ListT.h>
#include <lfcbase/SetT.h>

// DRAGON INCLUDES
#include "Production.h"
#include "Terminal.h"
#include "LR1Element.h"
#include "Dragon.h"
#include "Worm.h"
#include "LR1Analyser.h"
#include "LALRAnalyser.h"


Dragon::Dragon(const Chain& parserName, ParserMode mode, bool dynamicTable)
{
    _parserName = parserName;
    _dynamicTable = dynamicTable;
    _mode = mode;
}

Dragon::~Dragon()
{
}

void Dragon::generate(bool dumpIt)
{
    
    cout << "Reading Header ..." << endl;
    readHeader();
    
    cout << "Reading Terminals ..." << endl;
    readTerminals();

    cout << "Reading Productions ..." << endl;
    readProductions();
    
    cout << "Checking Productions ..." << endl;
    checkProductions();

    cout << "Analysing ..." << endl;
    createParseTable();

    if ( dumpIt )
    {
	cout << "Productions" << endl;

	Production* pP =  _productionSet.First();
	while (pP)
	{
	    cout << pP->getId() <<  " : " << pP->asChain() << endl;
	    pP =  _productionSet.Next();
	}

	cout << "Parse Table" << endl;
	cout << "-----------" << endl;
	printParseTable();

    } 
    else
    {
	generateCode();
    }
}

void Dragon::printParseTable()
{

    
    ParseTableEntry *pPTE = _parseTable.First();
    while (pPTE)
    {
	switch (pPTE->getAction())
	{
	case ParseTableEntry::SHIFT:
	    cout << pPTE->getState() << " SHIFT " << pPTE->getToken() << " " << pPTE->getArg() << endl;
	    break;
	case ParseTableEntry::REDUCE:
	    cout << pPTE->getState() << " REDUCE " << pPTE->getToken() << " " << pPTE->getArg() << endl;
	    break;
	case ParseTableEntry::JUMP:
	    cout << pPTE->getState() << " JUMP " << pPTE->getToken() << " " << pPTE->getArg() << endl;
	    break;
	case ParseTableEntry::ACCEPT:
	    cout << pPTE->getState() << " ACCEPT" << pPTE->getToken() << " " << pPTE->getArg() << endl;
	    break;
	}
	pPTE = _parseTable.Next();
    }

    
}

void Dragon::readHeader()
{
    
    Chain defFileName = _parserName + ".def";
    
    // cout << "open file " << defFileName << endl;

    File fin(defFileName);

    fin.open(File::READ);

    Chain line;
    bool sectionRead = false;
    while (fin.readLine(line) && sectionRead == false)
    {
	line=line.cutTrailing(Chain(" \t"));
	if ( line == Chain("HEADER"))
	{
	    while (fin.readLine(line) && sectionRead == false) 
	    {
		line=line.cutTrailing(Chain(" \t"));
		if (line && line[0] != '#')
		{
		    if (line != Chain("END"))
		    {
			Tokenizer tok(line, " \t");
			Chain name, value;
			if (tok.nextToken(name))
			{		      
			    tok.getTail(value);

			    value=value.cutTrailing(Chain(" \t"));			
    
			    if (name == Chain("SEPSIGN"))
			    {
				_sepsignList.Insert(value);
			    }
			    if (name == Chain("SEPIGNORE"))
			    {
				_sepignoreList.Insert(value);
			    }
			    if ( name == Chain("IGNORETOKEN") || name == Chain("IGNORE") )
			    {
				Terminal t(value, value, 0);
				_terminalSet.Insert(t);
			    }
			}		      
			else
			{
			    Chain msg = "scanning of line <" + line + "> failed.";
			    throw Exception(EXLOC, msg);
			}
		    }
		    else
		    {
			sectionRead = true;
		    }
		}
	    }
	}
    }
}

void Dragon::readTerminals()
{
    
    Chain defFileName = _parserName + ".def";
    
    File fin(defFileName);

    fin.open(File::READ);

    Chain line;

    unsigned num=1;

    bool sectionRead = false;

    while (fin.readLine(line) && sectionRead == false)
    {
	line=line.cutTrailing(Chain(" \t"));
	if ( line == Chain("TOKENSET"))
	{
	    while (fin.readLine(line) && sectionRead == false) 
	    {
		line=line.cutTrailing(Chain(" \t"));
		if (line && line[0] != '#')
		{
		    if (line != Chain("END"))
		    {

			Tokenizer *pTok;

			if (line[0] == '!')
			{
			    Tokenizer tok(line, " ");
			    Chain septoken;
			    tok.nextToken(septoken);
			    Chain sep = septoken.cutTrailing('!');
			   
			    pTok = new Tokenizer(line, sep);
			
			    Chain escToken;
			    pTok->nextToken(escToken);

			}
			else
			{
			    pTok = new Tokenizer(line, ":");
			}

			Chain regexp, token;
			if (pTok->nextToken(regexp))
			{
			    if (pTok->nextToken(token))
			    {			      
				token = token.cutTrailing(Chain(" \t"));
				regexp = regexp.cutTrailing(Chain(" \t")); 
				Terminal t(token, regexp, num);
				_terminalSet.Insert(t);
				num++;
			    }
			    else 
			    {
				Chain msg = "scanning of line <" + line + "> failed.";
				throw Exception(EXLOC, msg);
				
			    }
			} 
			else
			{
			    Chain msg = "scanning of line <" + line + "> failed.";
			    throw Exception(EXLOC, msg);				
			}

			delete pTok;
			
		    }
		    else
		    {
			sectionRead = true;
		    }
		}
	    }    
	}
    }
    fin.close();
}

void Dragon::readProductions()
{

    Chain defFileName = _parserName + ".def";
    
    File fin(defFileName);

    fin.open(File::READ);

    Chain line;

    bool sectionRead = false;

    unsigned long pid = 0;

    while (fin.readLine(line) && sectionRead == false)
    {
	line=line.cutTrailing(Chain(" \t"));
	if ( line == Chain("PRODUCTIONSET"))
	{
	    
	    while (fin.readLine(line) && sectionRead == false) 
	    {
		line=line.cutTrailing(Chain(" \t"));
		if (line && line[0] != '#')
		{
		    if (line != Chain("END"))
		    {

			Tokenizer tok(line, " \t");
			Chain token;
			
			if (tok.nextToken(token))
			{
			    Production prod;
			    
			    prod.setName(token);
			    prod.setId(pid);

			    pid++;

			    if (tok.nextToken(token))
			    {
				if (token != Chain(":"))
				{
				    Chain msg = "Invalid grammer token " + token;
				    throw Exception(EXLOC, msg);				    
				}
				while (tok.nextToken(token))
				{
				    if (token != Chain(";"))
				    {
					if ( _terminalSet.Find(Terminal(token)) )
					{
					    prod.addSymbol(token, true);
					}
					else
					{
					    prod.addSymbol(token, false);	       
					}
				    }
				    else 
				    {
					if (tok.nextToken(token))
					{				   
					    prod.setAction(token);	       
					}
					else
					{
					    // no action given
					}
				    }	
				}
				if ( !  _productionSet.Insert(prod) )
				{			       
				    Chain msg = "Duplicate production <" + prod.getName() + "> detected.";
				    throw Exception(EXLOC, msg);
				}
			    }
			}			
		    }
		    else
		    {
			sectionRead = true;
		    }
		}
	    }    
	}
    }
    fin.close();
}

void Dragon::checkProductions()
{

    ListT<Chain> prodNameList;

    Chain startProd;

    Production* pP =  _productionSet.First();
    while (pP)
    {
	if ( pP->getId() == 0 )
	    startProd = pP->getName();
	prodNameList.Insert(pP->getName());
	pP =  _productionSet.Next();
    }

    // check for unknown productions
    pP =  _productionSet.First();
    while (pP)
    {
	int pos = pP->getMaxPos() - 1;
	
	while ( pos >= 0 )
	{   
	    if ( ! pP->isTermAtPos(pos) )
	    {
		Chain symbol;
		pP->getSymbolAtPos(symbol, pos);
		if ( ! prodNameList.Find(symbol) )
		{
		    Chain msg = "Unknown symbol " + symbol;
		    throw Exception(EXLOC, msg);
		}
	    }
	    pos--;
	}
	pP =  _productionSet.Next();
    }

    // check for useless productions

    Chain *pS  =  prodNameList.First();
    while (pS)
    {

	bool notFound = true;
	pP =  _productionSet.First();
	while (pP && notFound )
	{
	
	    if ( pP->getName() != *pS )
	    {
		int pos = pP->getMaxPos() - 1;
		
		while ( pos >= 0 && notFound)
		{   
		    if ( ! pP->isTermAtPos(pos) )
		    {
			Chain symbol;
			pP->getSymbolAtPos(symbol, pos);
			if ( symbol == *pS )
			{
			    notFound = false;
			}
		    }
		    pos--;
		}
	    }
	    pP =  _productionSet.Next();
	}
	
	if ( notFound && *pS != startProd)
	{
	    Chain msg = "Useless production " + *pS;
	    throw Exception(EXLOC, msg);  
	}

	// cout << "Production " << *pS << " ok" << endl;
	pS  =  prodNameList.Next();
    }    
}

void Dragon::createParseTable()
{

    hashFirstSet();
    
    if ( _mode == LR1 )
    {
	LR1Analyser lr1( &_terminalSet, &_productionSet, &_firstHashSet);
	_numStates = lr1.analyse(_LR1HashSet, _LR1TransSet);
    }
    else
    {
	LALRAnalyser lalr( &_terminalSet, &_productionSet, &_firstHashSet);
	_numStates = lalr.analyse(_LR1HashSet, _LR1TransSet);	
    }
    
    createTable();
    
}

void Dragon::hashFirstSet()
{
    // create first set 
    
    SetT<Chain> prodSet;
    
    Production *pP = _productionSet.First();
    while (pP)
    {
	prodSet.Insert(pP->getName());
	pP = _productionSet.Next();
    }
    Chain *pS = prodSet.First();
    while (pS)
    {
	SetT<Chain> firstSet;
	getFirstSet(firstSet, *pS);
	FirstHash h(*pS, firstSet);
	_firstHashSet.Insert(h);
	pS = prodSet.Next();
    }
}


void Dragon::getFirstSet(SetT<Chain>& firstSet, const Chain& s)
{
    Terminal* pT = _terminalSet.First();
    while (pT)
    {
	if ((Chain)pT->getName() == s)
	{
	    firstSet.Insert(s);
	    return;
	}
	pT = _terminalSet.Next();
    }
    
    SetT<Chain> todoSet;    
    SetT<Chain> doneSet;

    todoSet.Insert(s);

    while ( ! todoSet.isEmpty() )
    {
	 
	Chain* pS = todoSet.First();

	SetT<Production> privateProdSet = _productionSet;

	Production* pP = privateProdSet.First();
	while (pP)
	{
	    if ( (Chain)pP->getName() == *pS ) 
	    {
		Chain s;
		if ( pP->getSymbolAtPos(s, 0) )
		{ 
		    if (pP->isTermAtPos(0))
		    {
			firstSet.Insert(s);
		    }
		    else
		    {			
			if ( ! doneSet.Find(s) )
			{
			    todoSet.Insert(s);
			}
		    }
		}
		else
		{
		    if ( _epsSet.Find(*pS) == 0 )
		    {
			_epsSet.Insert(*pS);
			getEpsilonJump(firstSet, *pS);
			_epsSet.Remove(*pS);
		    }
		    
		}	       
	    }
	    
	    pP = privateProdSet.Next();

	}
	
	doneSet.Insert(*pS);
	todoSet.Remove(*pS);       

    }

}

void Dragon::getEpsilonJump(SetT<Chain>& firstSet, const Chain& prod)
{

    SetT<Production> privateProdSet = _productionSet;

    SetT<Chain> todoSet;
    SetT<Chain> doneSet;

    todoSet.Insert(prod);

    while ( ! todoSet.isEmpty() )
    {
	Chain* pS = todoSet.First();

	Production *pP = privateProdSet.First();
	
	while (pP)
	{
	    unsigned i=0;
	    Chain sym;
	    while (pP->getSymbolAtPos(sym, i))
	    {		
		if (sym == *pS)
		{    
		    if ( pP->getSymbolAtPos(sym, i+1))
		    {
			if (pP->isTermAtPos(i+1))
			{
			    // cout << "Inserting " << sym << " into first set" << endl;
			    firstSet.Insert(sym);
			}
			else
			{
			
			    // cout << "Getting first set for " << sym << endl;
			    
                            if ( (Chain)sym == (Chain)prod)
			    {
				// cout << "Detecting loop ..." << endl;
				// calling getFirstSet for the same production 
				// leads to an infinite recursive loop		 
			    }
			    else
			    { 
				getFirstSet(firstSet, sym);
			    }			
			}
		    }
		    else
		    {
			
			// cout << "Inserting backtrack prod " << pP->getName() << " into todo set" << endl;
			if (doneSet.Find(pP->getName()) == 0 && pP->getId() != 0 )
			    todoSet.Insert(pP->getName());
			else
			    firstSet.Insert(Chain("$"));
		    }
		}
		i++;
	    }
	    pP = privateProdSet.Next();
	}
	
	doneSet.Insert(*pS);
	todoSet.Remove(*pS);

    }
}


void Dragon::createTable()
{
    cout << "Creating syntax table ..." << endl;
    
    LR1Hash* pH = _LR1HashSet.First();
    
    while (pH)
    {
	rotate();

	LR1Element* pE = pH->getSet().First();
	while (pE)
	{

	    Chain token;
	    if (pE->getSymbolAtPos(token))
	    {
		if ( ! pE->isProdAtPos() )
		{
		    LR1Trans* pT = _LR1TransSet.First();
		    while (pT)
		    {			
			if ( (Chain)pT->getToken() == (Chain)token 
			     && pT->getSid() == pH->getId() )
			{
			    ParseTableEntry e(pH->getId(), token, ParseTableEntry::SHIFT, pT->getTid());
			    
			    ParseTableEntry* pPTE = _parseTable.Find(e);
			    if ( pPTE)
			    {
				if ( pPTE->getAction() == ParseTableEntry::SHIFT && pPTE->getArg() != pT->getTid() )
				{
				 
				    cout << "------ Set A ------" << pPTE->getArg() << endl;
				    LR1Hash *pH = _LR1HashSet.First();
				    while (pH)
				    {
					if (pH->getId() == pPTE->getArg())
					{
					    LR1Element *pE = pH->getSet().First();
					    while (pE)
					    {
						cout << pE->asChain() << endl;
						pE = pH->getSet().Next();
					    }
					}

					pH = _LR1HashSet.Next();
				    }
				    cout << "------ Set B ------" << pT->getTid() << endl;
				    pH = _LR1HashSet.First();
				    while (pH)
				    {
					if (pH->getId() == pT->getTid())
					{
					    LR1Element *pE = pH->getSet().First();
					    while (pE)
					    {
						cout << pE->asChain() << endl;
						pE = pH->getSet().Next();
					    }
					}

					pH = _LR1HashSet.Next();
				    }

				    Chain msg = "Detected shift/shift conflict with production " + pE->asChain();
				    throw Exception(EXLOC, msg);
				    
				}
				if ( pPTE->getAction() == ParseTableEntry::REDUCE )
				{
				    Chain msg = "Detected shift/reduce conflict with production " + pE->asChain();
				    throw Exception(EXLOC, msg);
				}
			    }
			    else
			    {
				_parseTable.Insert(e);
			    }
			}
			pT = _LR1TransSet.Next();
		    }		   		   
		}		
	    }
	    else
	    {
		if ( pE->getProdId() != 0 ) // not start production
		{
		    ParseTableEntry e(pH->getId(), pE->getFirst(), ParseTableEntry::REDUCE, pE->getProdId());
		    
		    ParseTableEntry* pPTE = _parseTable.Find(e);
		    if ( pPTE)
		    {		     
			if ( pPTE->getAction() == ParseTableEntry::SHIFT )
			{
			    Chain msg = "Detected shift/reduce conflict with production " + pE->asChain();
			    throw Exception(EXLOC, msg);
			}
			else if ( pPTE->getAction() == ParseTableEntry::REDUCE )
			{
			    Chain msg = "Detected reduce/reduce conflict with production " + pE->asChain();
			    throw Exception(EXLOC, msg);

			}			
		    }
		    else
		    {
			_parseTable.Insert(e);
		    }
		}
		else 
		{
		    if ( (Chain)pE->getFirst() == Chain("$"))
		    {
			ParseTableEntry e(pH->getId(), pE->getFirst(), ParseTableEntry::ACCEPT);
			_parseTable.Insert(e);
		    }
		}
	    }
	    pE = pH->getSet().Next();
	}

	// create jump table

	LR1Trans* pT = _LR1TransSet.First();
	while (pT)
	{			
	    if ( pT->getSid() == pH->getId() && 
		 ! _terminalSet.Find(Terminal(pT->getToken())))		
	    {
		ParseTableEntry e(pH->getId(), pT->getToken(), ParseTableEntry::JUMP, pT->getTid());
		_parseTable.Insert(e);
	    }
	    pT = _LR1TransSet.Next();
	}
	pH = _LR1HashSet.Next();

    }
}

void Dragon::rotate()
{
    if ( _rot % 8 == 0)
	cout << "|\r";
    else if ( _rot % 8 == 1)    
	cout << "/\r";
    else if ( _rot % 8 == 2)    
	cout << "-\r";
    else if ( _rot % 8 == 3)    
	cout << "\\\r";
    else if ( _rot % 8 == 4)    
	cout << "|\r";
    else if ( _rot % 8 == 5)    
	cout << "/\r";
    else if ( _rot % 8 == 6)    
	cout << "-\r";
    else if ( _rot % 8 == 7)    
	cout << "\\\r";
    cout.flush();
    _rot++;

}



