#include <algorithm>
#include <string_view>

namespace ada::idna {

enum direction : uint8_t {
  NONE,
  BN,
  CS,
  ES,
  ON,
  EN,
  L,
  R,
  NSM,
  AL,
  AN,
  ET,
  WS,
  RLO,
  LRO,
  PDF,
  RLE,
  RLI,
  FSI,
  PDI,
  LRI,
  B,
  S,
  LRE
};

struct directions {
  uint32_t start_code;
  uint32_t final_code;
  direction direct;
};

static directions dir_table[] = {
    {0x0, 0x8, direction::BN},          {0x9, 0x9, direction::S},
    {0xa, 0xa, direction::B},           {0xb, 0xb, direction::S},
    {0xc, 0xc, direction::WS},          {0xd, 0xd, direction::B},
    {0xe, 0x1b, direction::BN},         {0x1c, 0x1e, direction::B},
    {0x1f, 0x1f, direction::S},         {0x20, 0x20, direction::WS},
    {0x21, 0x22, direction::ON},        {0x23, 0x25, direction::ET},
    {0x26, 0x2a, direction::ON},        {0x2b, 0x2b, direction::ES},
    {0x2c, 0x2c, direction::CS},        {0x2d, 0x2d, direction::ES},
    {0x2e, 0x2f, direction::CS},        {0x30, 0x39, direction::EN},
    {0x3a, 0x3a, direction::CS},        {0x3b, 0x40, direction::ON},
    {0x41, 0x5a, direction::L},         {0x5b, 0x60, direction::ON},
    {0x61, 0x7a, direction::L},         {0x7b, 0x7e, direction::ON},
    {0x7f, 0x84, direction::BN},        {0x85, 0x85, direction::B},
    {0x86, 0x9f, direction::BN},        {0xa0, 0xa0, direction::CS},
    {0xa1, 0xa1, direction::ON},        {0xa2, 0xa5, direction::ET},
    {0xa6, 0xa9, direction::ON},        {0xaa, 0xaa, direction::L},
    {0xab, 0xac, direction::ON},        {0xad, 0xad, direction::BN},
    {0xae, 0xaf, direction::ON},        {0xb0, 0xb1, direction::ET},
    {0xb2, 0xb3, direction::EN},        {0xb4, 0xb4, direction::ON},
    {0xb5, 0xb5, direction::L},         {0xb6, 0xb8, direction::ON},
    {0xb9, 0xb9, direction::EN},        {0xba, 0xba, direction::L},
    {0xbb, 0xbf, direction::ON},        {0xc0, 0xd6, direction::L},
    {0xd7, 0xd7, direction::ON},        {0xd8, 0xf6, direction::L},
    {0xf7, 0xf7, direction::ON},        {0xf8, 0x2b8, direction::L},
    {0x2b9, 0x2ba, direction::ON},      {0x2bb, 0x2c1, direction::L},
    {0x2c2, 0x2cf, direction::ON},      {0x2d0, 0x2d1, direction::L},
    {0x2d2, 0x2df, direction::ON},      {0x2e0, 0x2e4, direction::L},
    {0x2e5, 0x2ed, direction::ON},      {0x2ee, 0x2ee, direction::L},
    {0x2ef, 0x2ff, direction::ON},      {0x300, 0x36f, direction::NSM},
    {0x370, 0x373, direction::L},       {0x374, 0x375, direction::ON},
    {0x376, 0x377, direction::L},       {0x37a, 0x37d, direction::L},
    {0x37e, 0x37e, direction::ON},      {0x37f, 0x37f, direction::L},
    {0x384, 0x385, direction::ON},      {0x386, 0x386, direction::L},
    {0x387, 0x387, direction::ON},      {0x388, 0x38a, direction::L},
    {0x38c, 0x38c, direction::L},       {0x38e, 0x3a1, direction::L},
    {0x3a3, 0x3f5, direction::L},       {0x3f6, 0x3f6, direction::ON},
    {0x3f7, 0x482, direction::L},       {0x483, 0x489, direction::NSM},
    {0x48a, 0x52f, direction::L},       {0x531, 0x556, direction::L},
    {0x559, 0x589, direction::L},       {0x58a, 0x58a, direction::ON},
    {0x58d, 0x58e, direction::ON},      {0x58f, 0x58f, direction::ET},
    {0x591, 0x5bd, direction::NSM},     {0x5be, 0x5be, direction::R},
    {0x5bf, 0x5bf, direction::NSM},     {0x5c0, 0x5c0, direction::R},
    {0x5c1, 0x5c2, direction::NSM},     {0x5c3, 0x5c3, direction::R},
    {0x5c4, 0x5c5, direction::NSM},     {0x5c6, 0x5c6, direction::R},
    {0x5c7, 0x5c7, direction::NSM},     {0x5d0, 0x5ea, direction::R},
    {0x5ef, 0x5f4, direction::R},       {0x600, 0x605, direction::AN},
    {0x606, 0x607, direction::ON},      {0x608, 0x608, direction::AL},
    {0x609, 0x60a, direction::ET},      {0x60b, 0x60b, direction::AL},
    {0x60c, 0x60c, direction::CS},      {0x60d, 0x60d, direction::AL},
    {0x60e, 0x60f, direction::ON},      {0x610, 0x61a, direction::NSM},
    {0x61b, 0x61c, direction::AL},      {0x61e, 0x64a, direction::AL},
    {0x64b, 0x65f, direction::NSM},     {0x660, 0x669, direction::AN},
    {0x66a, 0x66a, direction::ET},      {0x66b, 0x66c, direction::AN},
    {0x66d, 0x66f, direction::AL},      {0x670, 0x670, direction::NSM},
    {0x671, 0x6d5, direction::AL},      {0x6d6, 0x6dc, direction::NSM},
    {0x6dd, 0x6dd, direction::AN},      {0x6de, 0x6de, direction::ON},
    {0x6df, 0x6e4, direction::NSM},     {0x6e5, 0x6e6, direction::AL},
    {0x6e7, 0x6e8, direction::NSM},     {0x6e9, 0x6e9, direction::ON},
    {0x6ea, 0x6ed, direction::NSM},     {0x6ee, 0x6ef, direction::AL},
    {0x6f0, 0x6f9, direction::EN},      {0x6fa, 0x70d, direction::AL},
    {0x70f, 0x710, direction::AL},      {0x711, 0x711, direction::NSM},
    {0x712, 0x72f, direction::AL},      {0x730, 0x74a, direction::NSM},
    {0x74d, 0x7a5, direction::AL},      {0x7a6, 0x7b0, direction::NSM},
    {0x7b1, 0x7b1, direction::AL},      {0x7c0, 0x7ea, direction::R},
    {0x7eb, 0x7f3, direction::NSM},     {0x7f4, 0x7f5, direction::R},
    {0x7f6, 0x7f9, direction::ON},      {0x7fa, 0x7fa, direction::R},
    {0x7fd, 0x7fd, direction::NSM},     {0x7fe, 0x815, direction::R},
    {0x816, 0x819, direction::NSM},     {0x81a, 0x81a, direction::R},
    {0x81b, 0x823, direction::NSM},     {0x824, 0x824, direction::R},
    {0x825, 0x827, direction::NSM},     {0x828, 0x828, direction::R},
    {0x829, 0x82d, direction::NSM},     {0x830, 0x83e, direction::R},
    {0x840, 0x858, direction::R},       {0x859, 0x85b, direction::NSM},
    {0x85e, 0x85e, direction::R},       {0x860, 0x86a, direction::AL},
    {0x8a0, 0x8b4, direction::AL},      {0x8b6, 0x8c7, direction::AL},
    {0x8d3, 0x8e1, direction::NSM},     {0x8e2, 0x8e2, direction::AN},
    {0x8e3, 0x902, direction::NSM},     {0x903, 0x939, direction::L},
    {0x93a, 0x93a, direction::NSM},     {0x93b, 0x93b, direction::L},
    {0x93c, 0x93c, direction::NSM},     {0x93d, 0x940, direction::L},
    {0x941, 0x948, direction::NSM},     {0x949, 0x94c, direction::L},
    {0x94d, 0x94d, direction::NSM},     {0x94e, 0x950, direction::L},
    {0x951, 0x957, direction::NSM},     {0x958, 0x961, direction::L},
    {0x962, 0x963, direction::NSM},     {0x964, 0x980, direction::L},
    {0x981, 0x981, direction::NSM},     {0x982, 0x983, direction::L},
    {0x985, 0x98c, direction::L},       {0x98f, 0x990, direction::L},
    {0x993, 0x9a8, direction::L},       {0x9aa, 0x9b0, direction::L},
    {0x9b2, 0x9b2, direction::L},       {0x9b6, 0x9b9, direction::L},
    {0x9bc, 0x9bc, direction::NSM},     {0x9bd, 0x9c0, direction::L},
    {0x9c1, 0x9c4, direction::NSM},     {0x9c7, 0x9c8, direction::L},
    {0x9cb, 0x9cc, direction::L},       {0x9cd, 0x9cd, direction::NSM},
    {0x9ce, 0x9ce, direction::L},       {0x9d7, 0x9d7, direction::L},
    {0x9dc, 0x9dd, direction::L},       {0x9df, 0x9e1, direction::L},
    {0x9e2, 0x9e3, direction::NSM},     {0x9e6, 0x9f1, direction::L},
    {0x9f2, 0x9f3, direction::ET},      {0x9f4, 0x9fa, direction::L},
    {0x9fb, 0x9fb, direction::ET},      {0x9fc, 0x9fd, direction::L},
    {0x9fe, 0x9fe, direction::NSM},     {0xa01, 0xa02, direction::NSM},
    {0xa03, 0xa03, direction::L},       {0xa05, 0xa0a, direction::L},
    {0xa0f, 0xa10, direction::L},       {0xa13, 0xa28, direction::L},
    {0xa2a, 0xa30, direction::L},       {0xa32, 0xa33, direction::L},
    {0xa35, 0xa36, direction::L},       {0xa38, 0xa39, direction::L},
    {0xa3c, 0xa3c, direction::NSM},     {0xa3e, 0xa40, direction::L},
    {0xa41, 0xa42, direction::NSM},     {0xa47, 0xa48, direction::NSM},
    {0xa4b, 0xa4d, direction::NSM},     {0xa51, 0xa51, direction::NSM},
    {0xa59, 0xa5c, direction::L},       {0xa5e, 0xa5e, direction::L},
    {0xa66, 0xa6f, direction::L},       {0xa70, 0xa71, direction::NSM},
    {0xa72, 0xa74, direction::L},       {0xa75, 0xa75, direction::NSM},
    {0xa76, 0xa76, direction::L},       {0xa81, 0xa82, direction::NSM},
    {0xa83, 0xa83, direction::L},       {0xa85, 0xa8d, direction::L},
    {0xa8f, 0xa91, direction::L},       {0xa93, 0xaa8, direction::L},
    {0xaaa, 0xab0, direction::L},       {0xab2, 0xab3, direction::L},
    {0xab5, 0xab9, direction::L},       {0xabc, 0xabc, direction::NSM},
    {0xabd, 0xac0, direction::L},       {0xac1, 0xac5, direction::NSM},
    {0xac7, 0xac8, direction::NSM},     {0xac9, 0xac9, direction::L},
    {0xacb, 0xacc, direction::L},       {0xacd, 0xacd, direction::NSM},
    {0xad0, 0xad0, direction::L},       {0xae0, 0xae1, direction::L},
    {0xae2, 0xae3, direction::NSM},     {0xae6, 0xaf0, direction::L},
    {0xaf1, 0xaf1, direction::ET},      {0xaf9, 0xaf9, direction::L},
    {0xafa, 0xaff, direction::NSM},     {0xb01, 0xb01, direction::NSM},
    {0xb02, 0xb03, direction::L},       {0xb05, 0xb0c, direction::L},
    {0xb0f, 0xb10, direction::L},       {0xb13, 0xb28, direction::L},
    {0xb2a, 0xb30, direction::L},       {0xb32, 0xb33, direction::L},
    {0xb35, 0xb39, direction::L},       {0xb3c, 0xb3c, direction::NSM},
    {0xb3d, 0xb3e, direction::L},       {0xb3f, 0xb3f, direction::NSM},
    {0xb40, 0xb40, direction::L},       {0xb41, 0xb44, direction::NSM},
    {0xb47, 0xb48, direction::L},       {0xb4b, 0xb4c, direction::L},
    {0xb4d, 0xb4d, direction::NSM},     {0xb55, 0xb56, direction::NSM},
    {0xb57, 0xb57, direction::L},       {0xb5c, 0xb5d, direction::L},
    {0xb5f, 0xb61, direction::L},       {0xb62, 0xb63, direction::NSM},
    {0xb66, 0xb77, direction::L},       {0xb82, 0xb82, direction::NSM},
    {0xb83, 0xb83, direction::L},       {0xb85, 0xb8a, direction::L},
    {0xb8e, 0xb90, direction::L},       {0xb92, 0xb95, direction::L},
    {0xb99, 0xb9a, direction::L},       {0xb9c, 0xb9c, direction::L},
    {0xb9e, 0xb9f, direction::L},       {0xba3, 0xba4, direction::L},
    {0xba8, 0xbaa, direction::L},       {0xbae, 0xbb9, direction::L},
    {0xbbe, 0xbbf, direction::L},       {0xbc0, 0xbc0, direction::NSM},
    {0xbc1, 0xbc2, direction::L},       {0xbc6, 0xbc8, direction::L},
    {0xbca, 0xbcc, direction::L},       {0xbcd, 0xbcd, direction::NSM},
    {0xbd0, 0xbd0, direction::L},       {0xbd7, 0xbd7, direction::L},
    {0xbe6, 0xbf2, direction::L},       {0xbf3, 0xbf8, direction::ON},
    {0xbf9, 0xbf9, direction::ET},      {0xbfa, 0xbfa, direction::ON},
    {0xc00, 0xc00, direction::NSM},     {0xc01, 0xc03, direction::L},
    {0xc04, 0xc04, direction::NSM},     {0xc05, 0xc0c, direction::L},
    {0xc0e, 0xc10, direction::L},       {0xc12, 0xc28, direction::L},
    {0xc2a, 0xc39, direction::L},       {0xc3d, 0xc3d, direction::L},
    {0xc3e, 0xc40, direction::NSM},     {0xc41, 0xc44, direction::L},
    {0xc46, 0xc48, direction::NSM},     {0xc4a, 0xc4d, direction::NSM},
    {0xc55, 0xc56, direction::NSM},     {0xc58, 0xc5a, direction::L},
    {0xc60, 0xc61, direction::L},       {0xc62, 0xc63, direction::NSM},
    {0xc66, 0xc6f, direction::L},       {0xc77, 0xc77, direction::L},
    {0xc78, 0xc7e, direction::ON},      {0xc7f, 0xc80, direction::L},
    {0xc81, 0xc81, direction::NSM},     {0xc82, 0xc8c, direction::L},
    {0xc8e, 0xc90, direction::L},       {0xc92, 0xca8, direction::L},
    {0xcaa, 0xcb3, direction::L},       {0xcb5, 0xcb9, direction::L},
    {0xcbc, 0xcbc, direction::NSM},     {0xcbd, 0xcc4, direction::L},
    {0xcc6, 0xcc8, direction::L},       {0xcca, 0xccb, direction::L},
    {0xccc, 0xccd, direction::NSM},     {0xcd5, 0xcd6, direction::L},
    {0xcde, 0xcde, direction::L},       {0xce0, 0xce1, direction::L},
    {0xce2, 0xce3, direction::NSM},     {0xce6, 0xcef, direction::L},
    {0xcf1, 0xcf2, direction::L},       {0xd00, 0xd01, direction::NSM},
    {0xd02, 0xd0c, direction::L},       {0xd0e, 0xd10, direction::L},
    {0xd12, 0xd3a, direction::L},       {0xd3b, 0xd3c, direction::NSM},
    {0xd3d, 0xd40, direction::L},       {0xd41, 0xd44, direction::NSM},
    {0xd46, 0xd48, direction::L},       {0xd4a, 0xd4c, direction::L},
    {0xd4d, 0xd4d, direction::NSM},     {0xd4e, 0xd4f, direction::L},
    {0xd54, 0xd61, direction::L},       {0xd62, 0xd63, direction::NSM},
    {0xd66, 0xd7f, direction::L},       {0xd81, 0xd81, direction::NSM},
    {0xd82, 0xd83, direction::L},       {0xd85, 0xd96, direction::L},
    {0xd9a, 0xdb1, direction::L},       {0xdb3, 0xdbb, direction::L},
    {0xdbd, 0xdbd, direction::L},       {0xdc0, 0xdc6, direction::L},
    {0xdca, 0xdca, direction::NSM},     {0xdcf, 0xdd1, direction::L},
    {0xdd2, 0xdd4, direction::NSM},     {0xdd6, 0xdd6, direction::NSM},
    {0xdd8, 0xddf, direction::L},       {0xde6, 0xdef, direction::L},
    {0xdf2, 0xdf4, direction::L},       {0xe01, 0xe30, direction::L},
    {0xe31, 0xe31, direction::NSM},     {0xe32, 0xe33, direction::L},
    {0xe34, 0xe3a, direction::NSM},     {0xe3f, 0xe3f, direction::ET},
    {0xe40, 0xe46, direction::L},       {0xe47, 0xe4e, direction::NSM},
    {0xe4f, 0xe5b, direction::L},       {0xe81, 0xe82, direction::L},
    {0xe84, 0xe84, direction::L},       {0xe86, 0xe8a, direction::L},
    {0xe8c, 0xea3, direction::L},       {0xea5, 0xea5, direction::L},
    {0xea7, 0xeb0, direction::L},       {0xeb1, 0xeb1, direction::NSM},
    {0xeb2, 0xeb3, direction::L},       {0xeb4, 0xebc, direction::NSM},
    {0xebd, 0xebd, direction::L},       {0xec0, 0xec4, direction::L},
    {0xec6, 0xec6, direction::L},       {0xec8, 0xecd, direction::NSM},
    {0xed0, 0xed9, direction::L},       {0xedc, 0xedf, direction::L},
    {0xf00, 0xf17, direction::L},       {0xf18, 0xf19, direction::NSM},
    {0xf1a, 0xf34, direction::L},       {0xf35, 0xf35, direction::NSM},
    {0xf36, 0xf36, direction::L},       {0xf37, 0xf37, direction::NSM},
    {0xf38, 0xf38, direction::L},       {0xf39, 0xf39, direction::NSM},
    {0xf3a, 0xf3d, direction::ON},      {0xf3e, 0xf47, direction::L},
    {0xf49, 0xf6c, direction::L},       {0xf71, 0xf7e, direction::NSM},
    {0xf7f, 0xf7f, direction::L},       {0xf80, 0xf84, direction::NSM},
    {0xf85, 0xf85, direction::L},       {0xf86, 0xf87, direction::NSM},
    {0xf88, 0xf8c, direction::L},       {0xf8d, 0xf97, direction::NSM},
    {0xf99, 0xfbc, direction::NSM},     {0xfbe, 0xfc5, direction::L},
    {0xfc6, 0xfc6, direction::NSM},     {0xfc7, 0xfcc, direction::L},
    {0xfce, 0xfda, direction::L},       {0x1000, 0x102c, direction::L},
    {0x102d, 0x1030, direction::NSM},   {0x1031, 0x1031, direction::L},
    {0x1032, 0x1037, direction::NSM},   {0x1038, 0x1038, direction::L},
    {0x1039, 0x103a, direction::NSM},   {0x103b, 0x103c, direction::L},
    {0x103d, 0x103e, direction::NSM},   {0x103f, 0x1057, direction::L},
    {0x1058, 0x1059, direction::NSM},   {0x105a, 0x105d, direction::L},
    {0x105e, 0x1060, direction::NSM},   {0x1061, 0x1070, direction::L},
    {0x1071, 0x1074, direction::NSM},   {0x1075, 0x1081, direction::L},
    {0x1082, 0x1082, direction::NSM},   {0x1083, 0x1084, direction::L},
    {0x1085, 0x1086, direction::NSM},   {0x1087, 0x108c, direction::L},
    {0x108d, 0x108d, direction::NSM},   {0x108e, 0x109c, direction::L},
    {0x109d, 0x109d, direction::NSM},   {0x109e, 0x10c5, direction::L},
    {0x10c7, 0x10c7, direction::L},     {0x10cd, 0x10cd, direction::L},
    {0x10d0, 0x1248, direction::L},     {0x124a, 0x124d, direction::L},
    {0x1250, 0x1256, direction::L},     {0x1258, 0x1258, direction::L},
    {0x125a, 0x125d, direction::L},     {0x1260, 0x1288, direction::L},
    {0x128a, 0x128d, direction::L},     {0x1290, 0x12b0, direction::L},
    {0x12b2, 0x12b5, direction::L},     {0x12b8, 0x12be, direction::L},
    {0x12c0, 0x12c0, direction::L},     {0x12c2, 0x12c5, direction::L},
    {0x12c8, 0x12d6, direction::L},     {0x12d8, 0x1310, direction::L},
    {0x1312, 0x1315, direction::L},     {0x1318, 0x135a, direction::L},
    {0x135d, 0x135f, direction::NSM},   {0x1360, 0x137c, direction::L},
    {0x1380, 0x138f, direction::L},     {0x1390, 0x1399, direction::ON},
    {0x13a0, 0x13f5, direction::L},     {0x13f8, 0x13fd, direction::L},
    {0x1400, 0x1400, direction::ON},    {0x1401, 0x167f, direction::L},
    {0x1680, 0x1680, direction::WS},    {0x1681, 0x169a, direction::L},
    {0x169b, 0x169c, direction::ON},    {0x16a0, 0x16f8, direction::L},
    {0x1700, 0x170c, direction::L},     {0x170e, 0x1711, direction::L},
    {0x1712, 0x1714, direction::NSM},   {0x1720, 0x1731, direction::L},
    {0x1732, 0x1734, direction::NSM},   {0x1735, 0x1736, direction::L},
    {0x1740, 0x1751, direction::L},     {0x1752, 0x1753, direction::NSM},
    {0x1760, 0x176c, direction::L},     {0x176e, 0x1770, direction::L},
    {0x1772, 0x1773, direction::NSM},   {0x1780, 0x17b3, direction::L},
    {0x17b4, 0x17b5, direction::NSM},   {0x17b6, 0x17b6, direction::L},
    {0x17b7, 0x17bd, direction::NSM},   {0x17be, 0x17c5, direction::L},
    {0x17c6, 0x17c6, direction::NSM},   {0x17c7, 0x17c8, direction::L},
    {0x17c9, 0x17d3, direction::NSM},   {0x17d4, 0x17da, direction::L},
    {0x17db, 0x17db, direction::ET},    {0x17dc, 0x17dc, direction::L},
    {0x17dd, 0x17dd, direction::NSM},   {0x17e0, 0x17e9, direction::L},
    {0x17f0, 0x17f9, direction::ON},    {0x1800, 0x180a, direction::ON},
    {0x180b, 0x180d, direction::NSM},   {0x180e, 0x180e, direction::BN},
    {0x1810, 0x1819, direction::L},     {0x1820, 0x1878, direction::L},
    {0x1880, 0x1884, direction::L},     {0x1885, 0x1886, direction::NSM},
    {0x1887, 0x18a8, direction::L},     {0x18a9, 0x18a9, direction::NSM},
    {0x18aa, 0x18aa, direction::L},     {0x18b0, 0x18f5, direction::L},
    {0x1900, 0x191e, direction::L},     {0x1920, 0x1922, direction::NSM},
    {0x1923, 0x1926, direction::L},     {0x1927, 0x1928, direction::NSM},
    {0x1929, 0x192b, direction::L},     {0x1930, 0x1931, direction::L},
    {0x1932, 0x1932, direction::NSM},   {0x1933, 0x1938, direction::L},
    {0x1939, 0x193b, direction::NSM},   {0x1940, 0x1940, direction::ON},
    {0x1944, 0x1945, direction::ON},    {0x1946, 0x196d, direction::L},
    {0x1970, 0x1974, direction::L},     {0x1980, 0x19ab, direction::L},
    {0x19b0, 0x19c9, direction::L},     {0x19d0, 0x19da, direction::L},
    {0x19de, 0x19ff, direction::ON},    {0x1a00, 0x1a16, direction::L},
    {0x1a17, 0x1a18, direction::NSM},   {0x1a19, 0x1a1a, direction::L},
    {0x1a1b, 0x1a1b, direction::NSM},   {0x1a1e, 0x1a55, direction::L},
    {0x1a56, 0x1a56, direction::NSM},   {0x1a57, 0x1a57, direction::L},
    {0x1a58, 0x1a5e, direction::NSM},   {0x1a60, 0x1a60, direction::NSM},
    {0x1a61, 0x1a61, direction::L},     {0x1a62, 0x1a62, direction::NSM},
    {0x1a63, 0x1a64, direction::L},     {0x1a65, 0x1a6c, direction::NSM},
    {0x1a6d, 0x1a72, direction::L},     {0x1a73, 0x1a7c, direction::NSM},
    {0x1a7f, 0x1a7f, direction::NSM},   {0x1a80, 0x1a89, direction::L},
    {0x1a90, 0x1a99, direction::L},     {0x1aa0, 0x1aad, direction::L},
    {0x1ab0, 0x1ac0, direction::NSM},   {0x1b00, 0x1b03, direction::NSM},
    {0x1b04, 0x1b33, direction::L},     {0x1b34, 0x1b34, direction::NSM},
    {0x1b35, 0x1b35, direction::L},     {0x1b36, 0x1b3a, direction::NSM},
    {0x1b3b, 0x1b3b, direction::L},     {0x1b3c, 0x1b3c, direction::NSM},
    {0x1b3d, 0x1b41, direction::L},     {0x1b42, 0x1b42, direction::NSM},
    {0x1b43, 0x1b4b, direction::L},     {0x1b50, 0x1b6a, direction::L},
    {0x1b6b, 0x1b73, direction::NSM},   {0x1b74, 0x1b7c, direction::L},
    {0x1b80, 0x1b81, direction::NSM},   {0x1b82, 0x1ba1, direction::L},
    {0x1ba2, 0x1ba5, direction::NSM},   {0x1ba6, 0x1ba7, direction::L},
    {0x1ba8, 0x1ba9, direction::NSM},   {0x1baa, 0x1baa, direction::L},
    {0x1bab, 0x1bad, direction::NSM},   {0x1bae, 0x1be5, direction::L},
    {0x1be6, 0x1be6, direction::NSM},   {0x1be7, 0x1be7, direction::L},
    {0x1be8, 0x1be9, direction::NSM},   {0x1bea, 0x1bec, direction::L},
    {0x1bed, 0x1bed, direction::NSM},   {0x1bee, 0x1bee, direction::L},
    {0x1bef, 0x1bf1, direction::NSM},   {0x1bf2, 0x1bf3, direction::L},
    {0x1bfc, 0x1c2b, direction::L},     {0x1c2c, 0x1c33, direction::NSM},
    {0x1c34, 0x1c35, direction::L},     {0x1c36, 0x1c37, direction::NSM},
    {0x1c3b, 0x1c49, direction::L},     {0x1c4d, 0x1c88, direction::L},
    {0x1c90, 0x1cba, direction::L},     {0x1cbd, 0x1cc7, direction::L},
    {0x1cd0, 0x1cd2, direction::NSM},   {0x1cd3, 0x1cd3, direction::L},
    {0x1cd4, 0x1ce0, direction::NSM},   {0x1ce1, 0x1ce1, direction::L},
    {0x1ce2, 0x1ce8, direction::NSM},   {0x1ce9, 0x1cec, direction::L},
    {0x1ced, 0x1ced, direction::NSM},   {0x1cee, 0x1cf3, direction::L},
    {0x1cf4, 0x1cf4, direction::NSM},   {0x1cf5, 0x1cf7, direction::L},
    {0x1cf8, 0x1cf9, direction::NSM},   {0x1cfa, 0x1cfa, direction::L},
    {0x1d00, 0x1dbf, direction::L},     {0x1dc0, 0x1df9, direction::NSM},
    {0x1dfb, 0x1dff, direction::NSM},   {0x1e00, 0x1f15, direction::L},
    {0x1f18, 0x1f1d, direction::L},     {0x1f20, 0x1f45, direction::L},
    {0x1f48, 0x1f4d, direction::L},     {0x1f50, 0x1f57, direction::L},
    {0x1f59, 0x1f59, direction::L},     {0x1f5b, 0x1f5b, direction::L},
    {0x1f5d, 0x1f5d, direction::L},     {0x1f5f, 0x1f7d, direction::L},
    {0x1f80, 0x1fb4, direction::L},     {0x1fb6, 0x1fbc, direction::L},
    {0x1fbd, 0x1fbd, direction::ON},    {0x1fbe, 0x1fbe, direction::L},
    {0x1fbf, 0x1fc1, direction::ON},    {0x1fc2, 0x1fc4, direction::L},
    {0x1fc6, 0x1fcc, direction::L},     {0x1fcd, 0x1fcf, direction::ON},
    {0x1fd0, 0x1fd3, direction::L},     {0x1fd6, 0x1fdb, direction::L},
    {0x1fdd, 0x1fdf, direction::ON},    {0x1fe0, 0x1fec, direction::L},
    {0x1fed, 0x1fef, direction::ON},    {0x1ff2, 0x1ff4, direction::L},
    {0x1ff6, 0x1ffc, direction::L},     {0x1ffd, 0x1ffe, direction::ON},
    {0x2000, 0x200a, direction::WS},    {0x200b, 0x200d, direction::BN},
    {0x200e, 0x200e, direction::L},     {0x200f, 0x200f, direction::R},
    {0x2010, 0x2027, direction::ON},    {0x2028, 0x2028, direction::WS},
    {0x2029, 0x2029, direction::B},     {0x202a, 0x202a, direction::LRE},
    {0x202b, 0x202b, direction::RLE},   {0x202c, 0x202c, direction::PDF},
    {0x202d, 0x202d, direction::LRO},   {0x202e, 0x202e, direction::RLO},
    {0x202f, 0x202f, direction::CS},    {0x2030, 0x2034, direction::ET},
    {0x2035, 0x2043, direction::ON},    {0x2044, 0x2044, direction::CS},
    {0x2045, 0x205e, direction::ON},    {0x205f, 0x205f, direction::WS},
    {0x2060, 0x2064, direction::BN},    {0x2066, 0x2066, direction::LRI},
    {0x2067, 0x2067, direction::RLI},   {0x2068, 0x2068, direction::FSI},
    {0x2069, 0x2069, direction::PDI},   {0x206a, 0x206f, direction::BN},
    {0x2070, 0x2070, direction::EN},    {0x2071, 0x2071, direction::L},
    {0x2074, 0x2079, direction::EN},    {0x207a, 0x207b, direction::ES},
    {0x207c, 0x207e, direction::ON},    {0x207f, 0x207f, direction::L},
    {0x2080, 0x2089, direction::EN},    {0x208a, 0x208b, direction::ES},
    {0x208c, 0x208e, direction::ON},    {0x2090, 0x209c, direction::L},
    {0x20a0, 0x20bf, direction::ET},    {0x20d0, 0x20f0, direction::NSM},
    {0x2100, 0x2101, direction::ON},    {0x2102, 0x2102, direction::L},
    {0x2103, 0x2106, direction::ON},    {0x2107, 0x2107, direction::L},
    {0x2108, 0x2109, direction::ON},    {0x210a, 0x2113, direction::L},
    {0x2114, 0x2114, direction::ON},    {0x2115, 0x2115, direction::L},
    {0x2116, 0x2118, direction::ON},    {0x2119, 0x211d, direction::L},
    {0x211e, 0x2123, direction::ON},    {0x2124, 0x2124, direction::L},
    {0x2125, 0x2125, direction::ON},    {0x2126, 0x2126, direction::L},
    {0x2127, 0x2127, direction::ON},    {0x2128, 0x2128, direction::L},
    {0x2129, 0x2129, direction::ON},    {0x212a, 0x212d, direction::L},
    {0x212e, 0x212e, direction::ET},    {0x212f, 0x2139, direction::L},
    {0x213a, 0x213b, direction::ON},    {0x213c, 0x213f, direction::L},
    {0x2140, 0x2144, direction::ON},    {0x2145, 0x2149, direction::L},
    {0x214a, 0x214d, direction::ON},    {0x214e, 0x214f, direction::L},
    {0x2150, 0x215f, direction::ON},    {0x2160, 0x2188, direction::L},
    {0x2189, 0x218b, direction::ON},    {0x2190, 0x2211, direction::ON},
    {0x2212, 0x2212, direction::ES},    {0x2213, 0x2213, direction::ET},
    {0x2214, 0x2335, direction::ON},    {0x2336, 0x237a, direction::L},
    {0x237b, 0x2394, direction::ON},    {0x2395, 0x2395, direction::L},
    {0x2396, 0x2426, direction::ON},    {0x2440, 0x244a, direction::ON},
    {0x2460, 0x2487, direction::ON},    {0x2488, 0x249b, direction::EN},
    {0x249c, 0x24e9, direction::L},     {0x24ea, 0x26ab, direction::ON},
    {0x26ac, 0x26ac, direction::L},     {0x26ad, 0x27ff, direction::ON},
    {0x2800, 0x28ff, direction::L},     {0x2900, 0x2b73, direction::ON},
    {0x2b76, 0x2b95, direction::ON},    {0x2b97, 0x2bff, direction::ON},
    {0x2c00, 0x2c2e, direction::L},     {0x2c30, 0x2c5e, direction::L},
    {0x2c60, 0x2ce4, direction::L},     {0x2ce5, 0x2cea, direction::ON},
    {0x2ceb, 0x2cee, direction::L},     {0x2cef, 0x2cf1, direction::NSM},
    {0x2cf2, 0x2cf3, direction::L},     {0x2cf9, 0x2cff, direction::ON},
    {0x2d00, 0x2d25, direction::L},     {0x2d27, 0x2d27, direction::L},
    {0x2d2d, 0x2d2d, direction::L},     {0x2d30, 0x2d67, direction::L},
    {0x2d6f, 0x2d70, direction::L},     {0x2d7f, 0x2d7f, direction::NSM},
    {0x2d80, 0x2d96, direction::L},     {0x2da0, 0x2da6, direction::L},
    {0x2da8, 0x2dae, direction::L},     {0x2db0, 0x2db6, direction::L},
    {0x2db8, 0x2dbe, direction::L},     {0x2dc0, 0x2dc6, direction::L},
    {0x2dc8, 0x2dce, direction::L},     {0x2dd0, 0x2dd6, direction::L},
    {0x2dd8, 0x2dde, direction::L},     {0x2de0, 0x2dff, direction::NSM},
    {0x2e00, 0x2e52, direction::ON},    {0x2e80, 0x2e99, direction::ON},
    {0x2e9b, 0x2ef3, direction::ON},    {0x2f00, 0x2fd5, direction::ON},
    {0x2ff0, 0x2ffb, direction::ON},    {0x3000, 0x3000, direction::WS},
    {0x3001, 0x3004, direction::ON},    {0x3005, 0x3007, direction::L},
    {0x3008, 0x3020, direction::ON},    {0x3021, 0x3029, direction::L},
    {0x302a, 0x302d, direction::NSM},   {0x302e, 0x302f, direction::L},
    {0x3030, 0x3030, direction::ON},    {0x3031, 0x3035, direction::L},
    {0x3036, 0x3037, direction::ON},    {0x3038, 0x303c, direction::L},
    {0x303d, 0x303f, direction::ON},    {0x3041, 0x3096, direction::L},
    {0x3099, 0x309a, direction::NSM},   {0x309b, 0x309c, direction::ON},
    {0x309d, 0x309f, direction::L},     {0x30a0, 0x30a0, direction::ON},
    {0x30a1, 0x30fa, direction::L},     {0x30fb, 0x30fb, direction::ON},
    {0x30fc, 0x30ff, direction::L},     {0x3105, 0x312f, direction::L},
    {0x3131, 0x318e, direction::L},     {0x3190, 0x31bf, direction::L},
    {0x31c0, 0x31e3, direction::ON},    {0x31f0, 0x321c, direction::L},
    {0x321d, 0x321e, direction::ON},    {0x3220, 0x324f, direction::L},
    {0x3250, 0x325f, direction::ON},    {0x3260, 0x327b, direction::L},
    {0x327c, 0x327e, direction::ON},    {0x327f, 0x32b0, direction::L},
    {0x32b1, 0x32bf, direction::ON},    {0x32c0, 0x32cb, direction::L},
    {0x32cc, 0x32cf, direction::ON},    {0x32d0, 0x3376, direction::L},
    {0x3377, 0x337a, direction::ON},    {0x337b, 0x33dd, direction::L},
    {0x33de, 0x33df, direction::ON},    {0x33e0, 0x33fe, direction::L},
    {0x33ff, 0x33ff, direction::ON},    {0x3400, 0x4dbf, direction::L},
    {0x4dc0, 0x4dff, direction::ON},    {0x4e00, 0x9ffc, direction::L},
    {0xa000, 0xa48c, direction::L},     {0xa490, 0xa4c6, direction::ON},
    {0xa4d0, 0xa60c, direction::L},     {0xa60d, 0xa60f, direction::ON},
    {0xa610, 0xa62b, direction::L},     {0xa640, 0xa66e, direction::L},
    {0xa66f, 0xa672, direction::NSM},   {0xa673, 0xa673, direction::ON},
    {0xa674, 0xa67d, direction::NSM},   {0xa67e, 0xa67f, direction::ON},
    {0xa680, 0xa69d, direction::L},     {0xa69e, 0xa69f, direction::NSM},
    {0xa6a0, 0xa6ef, direction::L},     {0xa6f0, 0xa6f1, direction::NSM},
    {0xa6f2, 0xa6f7, direction::L},     {0xa700, 0xa721, direction::ON},
    {0xa722, 0xa787, direction::L},     {0xa788, 0xa788, direction::ON},
    {0xa789, 0xa7bf, direction::L},     {0xa7c2, 0xa7ca, direction::L},
    {0xa7f5, 0xa801, direction::L},     {0xa802, 0xa802, direction::NSM},
    {0xa803, 0xa805, direction::L},     {0xa806, 0xa806, direction::NSM},
    {0xa807, 0xa80a, direction::L},     {0xa80b, 0xa80b, direction::NSM},
    {0xa80c, 0xa824, direction::L},     {0xa825, 0xa826, direction::NSM},
    {0xa827, 0xa827, direction::L},     {0xa828, 0xa82b, direction::ON},
    {0xa82c, 0xa82c, direction::NSM},   {0xa830, 0xa837, direction::L},
    {0xa838, 0xa839, direction::ET},    {0xa840, 0xa873, direction::L},
    {0xa874, 0xa877, direction::ON},    {0xa880, 0xa8c3, direction::L},
    {0xa8c4, 0xa8c5, direction::NSM},   {0xa8ce, 0xa8d9, direction::L},
    {0xa8e0, 0xa8f1, direction::NSM},   {0xa8f2, 0xa8fe, direction::L},
    {0xa8ff, 0xa8ff, direction::NSM},   {0xa900, 0xa925, direction::L},
    {0xa926, 0xa92d, direction::NSM},   {0xa92e, 0xa946, direction::L},
    {0xa947, 0xa951, direction::NSM},   {0xa952, 0xa953, direction::L},
    {0xa95f, 0xa97c, direction::L},     {0xa980, 0xa982, direction::NSM},
    {0xa983, 0xa9b2, direction::L},     {0xa9b3, 0xa9b3, direction::NSM},
    {0xa9b4, 0xa9b5, direction::L},     {0xa9b6, 0xa9b9, direction::NSM},
    {0xa9ba, 0xa9bb, direction::L},     {0xa9bc, 0xa9bd, direction::NSM},
    {0xa9be, 0xa9cd, direction::L},     {0xa9cf, 0xa9d9, direction::L},
    {0xa9de, 0xa9e4, direction::L},     {0xa9e5, 0xa9e5, direction::NSM},
    {0xa9e6, 0xa9fe, direction::L},     {0xaa00, 0xaa28, direction::L},
    {0xaa29, 0xaa2e, direction::NSM},   {0xaa2f, 0xaa30, direction::L},
    {0xaa31, 0xaa32, direction::NSM},   {0xaa33, 0xaa34, direction::L},
    {0xaa35, 0xaa36, direction::NSM},   {0xaa40, 0xaa42, direction::L},
    {0xaa43, 0xaa43, direction::NSM},   {0xaa44, 0xaa4b, direction::L},
    {0xaa4c, 0xaa4c, direction::NSM},   {0xaa4d, 0xaa4d, direction::L},
    {0xaa50, 0xaa59, direction::L},     {0xaa5c, 0xaa7b, direction::L},
    {0xaa7c, 0xaa7c, direction::NSM},   {0xaa7d, 0xaaaf, direction::L},
    {0xaab0, 0xaab0, direction::NSM},   {0xaab1, 0xaab1, direction::L},
    {0xaab2, 0xaab4, direction::NSM},   {0xaab5, 0xaab6, direction::L},
    {0xaab7, 0xaab8, direction::NSM},   {0xaab9, 0xaabd, direction::L},
    {0xaabe, 0xaabf, direction::NSM},   {0xaac0, 0xaac0, direction::L},
    {0xaac1, 0xaac1, direction::NSM},   {0xaac2, 0xaac2, direction::L},
    {0xaadb, 0xaaeb, direction::L},     {0xaaec, 0xaaed, direction::NSM},
    {0xaaee, 0xaaf5, direction::L},     {0xaaf6, 0xaaf6, direction::NSM},
    {0xab01, 0xab06, direction::L},     {0xab09, 0xab0e, direction::L},
    {0xab11, 0xab16, direction::L},     {0xab20, 0xab26, direction::L},
    {0xab28, 0xab2e, direction::L},     {0xab30, 0xab69, direction::L},
    {0xab6a, 0xab6b, direction::ON},    {0xab70, 0xabe4, direction::L},
    {0xabe5, 0xabe5, direction::NSM},   {0xabe6, 0xabe7, direction::L},
    {0xabe8, 0xabe8, direction::NSM},   {0xabe9, 0xabec, direction::L},
    {0xabed, 0xabed, direction::NSM},   {0xabf0, 0xabf9, direction::L},
    {0xac00, 0xd7a3, direction::L},     {0xd7b0, 0xd7c6, direction::L},
    {0xd7cb, 0xd7fb, direction::L},     {0xd800, 0xfa6d, direction::L},
    {0xfa70, 0xfad9, direction::L},     {0xfb00, 0xfb06, direction::L},
    {0xfb13, 0xfb17, direction::L},     {0xfb1d, 0xfb1d, direction::R},
    {0xfb1e, 0xfb1e, direction::NSM},   {0xfb1f, 0xfb28, direction::R},
    {0xfb29, 0xfb29, direction::ES},    {0xfb2a, 0xfb36, direction::R},
    {0xfb38, 0xfb3c, direction::R},     {0xfb3e, 0xfb3e, direction::R},
    {0xfb40, 0xfb41, direction::R},     {0xfb43, 0xfb44, direction::R},
    {0xfb46, 0xfb4f, direction::R},     {0xfb50, 0xfbc1, direction::AL},
    {0xfbd3, 0xfd3d, direction::AL},    {0xfd3e, 0xfd3f, direction::ON},
    {0xfd50, 0xfd8f, direction::AL},    {0xfd92, 0xfdc7, direction::AL},
    {0xfdf0, 0xfdfc, direction::AL},    {0xfdfd, 0xfdfd, direction::ON},
    {0xfe00, 0xfe0f, direction::NSM},   {0xfe10, 0xfe19, direction::ON},
    {0xfe20, 0xfe2f, direction::NSM},   {0xfe30, 0xfe4f, direction::ON},
    {0xfe50, 0xfe50, direction::CS},    {0xfe51, 0xfe51, direction::ON},
    {0xfe52, 0xfe52, direction::CS},    {0xfe54, 0xfe54, direction::ON},
    {0xfe55, 0xfe55, direction::CS},    {0xfe56, 0xfe5e, direction::ON},
    {0xfe5f, 0xfe5f, direction::ET},    {0xfe60, 0xfe61, direction::ON},
    {0xfe62, 0xfe63, direction::ES},    {0xfe64, 0xfe66, direction::ON},
    {0xfe68, 0xfe68, direction::ON},    {0xfe69, 0xfe6a, direction::ET},
    {0xfe6b, 0xfe6b, direction::ON},    {0xfe70, 0xfe74, direction::AL},
    {0xfe76, 0xfefc, direction::AL},    {0xfeff, 0xfeff, direction::BN},
    {0xff01, 0xff02, direction::ON},    {0xff03, 0xff05, direction::ET},
    {0xff06, 0xff0a, direction::ON},    {0xff0b, 0xff0b, direction::ES},
    {0xff0c, 0xff0c, direction::CS},    {0xff0d, 0xff0d, direction::ES},
    {0xff0e, 0xff0f, direction::CS},    {0xff10, 0xff19, direction::EN},
    {0xff1a, 0xff1a, direction::CS},    {0xff1b, 0xff20, direction::ON},
    {0xff21, 0xff3a, direction::L},     {0xff3b, 0xff40, direction::ON},
    {0xff41, 0xff5a, direction::L},     {0xff5b, 0xff65, direction::ON},
    {0xff66, 0xffbe, direction::L},     {0xffc2, 0xffc7, direction::L},
    {0xffca, 0xffcf, direction::L},     {0xffd2, 0xffd7, direction::L},
    {0xffda, 0xffdc, direction::L},     {0xffe0, 0xffe1, direction::ET},
    {0xffe2, 0xffe4, direction::ON},    {0xffe5, 0xffe6, direction::ET},
    {0xffe8, 0xffee, direction::ON},    {0xfff9, 0xfffd, direction::ON},
    {0x10000, 0x1000b, direction::L},   {0x1000d, 0x10026, direction::L},
    {0x10028, 0x1003a, direction::L},   {0x1003c, 0x1003d, direction::L},
    {0x1003f, 0x1004d, direction::L},   {0x10050, 0x1005d, direction::L},
    {0x10080, 0x100fa, direction::L},   {0x10100, 0x10100, direction::L},
    {0x10101, 0x10101, direction::ON},  {0x10102, 0x10102, direction::L},
    {0x10107, 0x10133, direction::L},   {0x10137, 0x1013f, direction::L},
    {0x10140, 0x1018c, direction::ON},  {0x1018d, 0x1018e, direction::L},
    {0x10190, 0x1019c, direction::ON},  {0x101a0, 0x101a0, direction::ON},
    {0x101d0, 0x101fc, direction::L},   {0x101fd, 0x101fd, direction::NSM},
    {0x10280, 0x1029c, direction::L},   {0x102a0, 0x102d0, direction::L},
    {0x102e0, 0x102e0, direction::NSM}, {0x102e1, 0x102fb, direction::EN},
    {0x10300, 0x10323, direction::L},   {0x1032d, 0x1034a, direction::L},
    {0x10350, 0x10375, direction::L},   {0x10376, 0x1037a, direction::NSM},
    {0x10380, 0x1039d, direction::L},   {0x1039f, 0x103c3, direction::L},
    {0x103c8, 0x103d5, direction::L},   {0x10400, 0x1049d, direction::L},
    {0x104a0, 0x104a9, direction::L},   {0x104b0, 0x104d3, direction::L},
    {0x104d8, 0x104fb, direction::L},   {0x10500, 0x10527, direction::L},
    {0x10530, 0x10563, direction::L},   {0x1056f, 0x1056f, direction::L},
    {0x10600, 0x10736, direction::L},   {0x10740, 0x10755, direction::L},
    {0x10760, 0x10767, direction::L},   {0x10800, 0x10805, direction::R},
    {0x10808, 0x10808, direction::R},   {0x1080a, 0x10835, direction::R},
    {0x10837, 0x10838, direction::R},   {0x1083c, 0x1083c, direction::R},
    {0x1083f, 0x10855, direction::R},   {0x10857, 0x1089e, direction::R},
    {0x108a7, 0x108af, direction::R},   {0x108e0, 0x108f2, direction::R},
    {0x108f4, 0x108f5, direction::R},   {0x108fb, 0x1091b, direction::R},
    {0x1091f, 0x1091f, direction::ON},  {0x10920, 0x10939, direction::R},
    {0x1093f, 0x1093f, direction::R},   {0x10980, 0x109b7, direction::R},
    {0x109bc, 0x109cf, direction::R},   {0x109d2, 0x10a00, direction::R},
    {0x10a01, 0x10a03, direction::NSM}, {0x10a05, 0x10a06, direction::NSM},
    {0x10a0c, 0x10a0f, direction::NSM}, {0x10a10, 0x10a13, direction::R},
    {0x10a15, 0x10a17, direction::R},   {0x10a19, 0x10a35, direction::R},
    {0x10a38, 0x10a3a, direction::NSM}, {0x10a3f, 0x10a3f, direction::NSM},
    {0x10a40, 0x10a48, direction::R},   {0x10a50, 0x10a58, direction::R},
    {0x10a60, 0x10a9f, direction::R},   {0x10ac0, 0x10ae4, direction::R},
    {0x10ae5, 0x10ae6, direction::NSM}, {0x10aeb, 0x10af6, direction::R},
    {0x10b00, 0x10b35, direction::R},   {0x10b39, 0x10b3f, direction::ON},
    {0x10b40, 0x10b55, direction::R},   {0x10b58, 0x10b72, direction::R},
    {0x10b78, 0x10b91, direction::R},   {0x10b99, 0x10b9c, direction::R},
    {0x10ba9, 0x10baf, direction::R},   {0x10c00, 0x10c48, direction::R},
    {0x10c80, 0x10cb2, direction::R},   {0x10cc0, 0x10cf2, direction::R},
    {0x10cfa, 0x10cff, direction::R},   {0x10d00, 0x10d23, direction::AL},
    {0x10d24, 0x10d27, direction::NSM}, {0x10d30, 0x10d39, direction::AN},
    {0x10e60, 0x10e7e, direction::AN},  {0x10e80, 0x10ea9, direction::R},
    {0x10eab, 0x10eac, direction::NSM}, {0x10ead, 0x10ead, direction::R},
    {0x10eb0, 0x10eb1, direction::R},   {0x10f00, 0x10f27, direction::R},
    {0x10f30, 0x10f45, direction::AL},  {0x10f46, 0x10f50, direction::NSM},
    {0x10f51, 0x10f59, direction::AL},  {0x10fb0, 0x10fcb, direction::R},
    {0x10fe0, 0x10ff6, direction::R},   {0x11000, 0x11000, direction::L},
    {0x11001, 0x11001, direction::NSM}, {0x11002, 0x11037, direction::L},
    {0x11038, 0x11046, direction::NSM}, {0x11047, 0x1104d, direction::L},
    {0x11052, 0x11065, direction::ON},  {0x11066, 0x1106f, direction::L},
    {0x1107f, 0x11081, direction::NSM}, {0x11082, 0x110b2, direction::L},
    {0x110b3, 0x110b6, direction::NSM}, {0x110b7, 0x110b8, direction::L},
    {0x110b9, 0x110ba, direction::NSM}, {0x110bb, 0x110c1, direction::L},
    {0x110cd, 0x110cd, direction::L},   {0x110d0, 0x110e8, direction::L},
    {0x110f0, 0x110f9, direction::L},   {0x11100, 0x11102, direction::NSM},
    {0x11103, 0x11126, direction::L},   {0x11127, 0x1112b, direction::NSM},
    {0x1112c, 0x1112c, direction::L},   {0x1112d, 0x11134, direction::NSM},
    {0x11136, 0x11147, direction::L},   {0x11150, 0x11172, direction::L},
    {0x11173, 0x11173, direction::NSM}, {0x11174, 0x11176, direction::L},
    {0x11180, 0x11181, direction::NSM}, {0x11182, 0x111b5, direction::L},
    {0x111b6, 0x111be, direction::NSM}, {0x111bf, 0x111c8, direction::L},
    {0x111c9, 0x111cc, direction::NSM}, {0x111cd, 0x111ce, direction::L},
    {0x111cf, 0x111cf, direction::NSM}, {0x111d0, 0x111df, direction::L},
    {0x111e1, 0x111f4, direction::L},   {0x11200, 0x11211, direction::L},
    {0x11213, 0x1122e, direction::L},   {0x1122f, 0x11231, direction::NSM},
    {0x11232, 0x11233, direction::L},   {0x11234, 0x11234, direction::NSM},
    {0x11235, 0x11235, direction::L},   {0x11236, 0x11237, direction::NSM},
    {0x11238, 0x1123d, direction::L},   {0x1123e, 0x1123e, direction::NSM},
    {0x11280, 0x11286, direction::L},   {0x11288, 0x11288, direction::L},
    {0x1128a, 0x1128d, direction::L},   {0x1128f, 0x1129d, direction::L},
    {0x1129f, 0x112a9, direction::L},   {0x112b0, 0x112de, direction::L},
    {0x112df, 0x112df, direction::NSM}, {0x112e0, 0x112e2, direction::L},
    {0x112e3, 0x112ea, direction::NSM}, {0x112f0, 0x112f9, direction::L},
    {0x11300, 0x11301, direction::NSM}, {0x11302, 0x11303, direction::L},
    {0x11305, 0x1130c, direction::L},   {0x1130f, 0x11310, direction::L},
    {0x11313, 0x11328, direction::L},   {0x1132a, 0x11330, direction::L},
    {0x11332, 0x11333, direction::L},   {0x11335, 0x11339, direction::L},
    {0x1133b, 0x1133c, direction::NSM}, {0x1133d, 0x1133f, direction::L},
    {0x11340, 0x11340, direction::NSM}, {0x11341, 0x11344, direction::L},
    {0x11347, 0x11348, direction::L},   {0x1134b, 0x1134d, direction::L},
    {0x11350, 0x11350, direction::L},   {0x11357, 0x11357, direction::L},
    {0x1135d, 0x11363, direction::L},   {0x11366, 0x1136c, direction::NSM},
    {0x11370, 0x11374, direction::NSM}, {0x11400, 0x11437, direction::L},
    {0x11438, 0x1143f, direction::NSM}, {0x11440, 0x11441, direction::L},
    {0x11442, 0x11444, direction::NSM}, {0x11445, 0x11445, direction::L},
    {0x11446, 0x11446, direction::NSM}, {0x11447, 0x1145b, direction::L},
    {0x1145d, 0x1145d, direction::L},   {0x1145e, 0x1145e, direction::NSM},
    {0x1145f, 0x11461, direction::L},   {0x11480, 0x114b2, direction::L},
    {0x114b3, 0x114b8, direction::NSM}, {0x114b9, 0x114b9, direction::L},
    {0x114ba, 0x114ba, direction::NSM}, {0x114bb, 0x114be, direction::L},
    {0x114bf, 0x114c0, direction::NSM}, {0x114c1, 0x114c1, direction::L},
    {0x114c2, 0x114c3, direction::NSM}, {0x114c4, 0x114c7, direction::L},
    {0x114d0, 0x114d9, direction::L},   {0x11580, 0x115b1, direction::L},
    {0x115b2, 0x115b5, direction::NSM}, {0x115b8, 0x115bb, direction::L},
    {0x115bc, 0x115bd, direction::NSM}, {0x115be, 0x115be, direction::L},
    {0x115bf, 0x115c0, direction::NSM}, {0x115c1, 0x115db, direction::L},
    {0x115dc, 0x115dd, direction::NSM}, {0x11600, 0x11632, direction::L},
    {0x11633, 0x1163a, direction::NSM}, {0x1163b, 0x1163c, direction::L},
    {0x1163d, 0x1163d, direction::NSM}, {0x1163e, 0x1163e, direction::L},
    {0x1163f, 0x11640, direction::NSM}, {0x11641, 0x11644, direction::L},
    {0x11650, 0x11659, direction::L},   {0x11660, 0x1166c, direction::ON},
    {0x11680, 0x116aa, direction::L},   {0x116ab, 0x116ab, direction::NSM},
    {0x116ac, 0x116ac, direction::L},   {0x116ad, 0x116ad, direction::NSM},
    {0x116ae, 0x116af, direction::L},   {0x116b0, 0x116b5, direction::NSM},
    {0x116b6, 0x116b6, direction::L},   {0x116b7, 0x116b7, direction::NSM},
    {0x116b8, 0x116b8, direction::L},   {0x116c0, 0x116c9, direction::L},
    {0x11700, 0x1171a, direction::L},   {0x1171d, 0x1171f, direction::NSM},
    {0x11720, 0x11721, direction::L},   {0x11722, 0x11725, direction::NSM},
    {0x11726, 0x11726, direction::L},   {0x11727, 0x1172b, direction::NSM},
    {0x11730, 0x1173f, direction::L},   {0x11800, 0x1182e, direction::L},
    {0x1182f, 0x11837, direction::NSM}, {0x11838, 0x11838, direction::L},
    {0x11839, 0x1183a, direction::NSM}, {0x1183b, 0x1183b, direction::L},
    {0x118a0, 0x118f2, direction::L},   {0x118ff, 0x11906, direction::L},
    {0x11909, 0x11909, direction::L},   {0x1190c, 0x11913, direction::L},
    {0x11915, 0x11916, direction::L},   {0x11918, 0x11935, direction::L},
    {0x11937, 0x11938, direction::L},   {0x1193b, 0x1193c, direction::NSM},
    {0x1193d, 0x1193d, direction::L},   {0x1193e, 0x1193e, direction::NSM},
    {0x1193f, 0x11942, direction::L},   {0x11943, 0x11943, direction::NSM},
    {0x11944, 0x11946, direction::L},   {0x11950, 0x11959, direction::L},
    {0x119a0, 0x119a7, direction::L},   {0x119aa, 0x119d3, direction::L},
    {0x119d4, 0x119d7, direction::NSM}, {0x119da, 0x119db, direction::NSM},
    {0x119dc, 0x119df, direction::L},   {0x119e0, 0x119e0, direction::NSM},
    {0x119e1, 0x119e4, direction::L},   {0x11a00, 0x11a00, direction::L},
    {0x11a01, 0x11a06, direction::NSM}, {0x11a07, 0x11a08, direction::L},
    {0x11a09, 0x11a0a, direction::NSM}, {0x11a0b, 0x11a32, direction::L},
    {0x11a33, 0x11a38, direction::NSM}, {0x11a39, 0x11a3a, direction::L},
    {0x11a3b, 0x11a3e, direction::NSM}, {0x11a3f, 0x11a46, direction::L},
    {0x11a47, 0x11a47, direction::NSM}, {0x11a50, 0x11a50, direction::L},
    {0x11a51, 0x11a56, direction::NSM}, {0x11a57, 0x11a58, direction::L},
    {0x11a59, 0x11a5b, direction::NSM}, {0x11a5c, 0x11a89, direction::L},
    {0x11a8a, 0x11a96, direction::NSM}, {0x11a97, 0x11a97, direction::L},
    {0x11a98, 0x11a99, direction::NSM}, {0x11a9a, 0x11aa2, direction::L},
    {0x11ac0, 0x11af8, direction::L},   {0x11c00, 0x11c08, direction::L},
    {0x11c0a, 0x11c2f, direction::L},   {0x11c30, 0x11c36, direction::NSM},
    {0x11c38, 0x11c3d, direction::NSM}, {0x11c3e, 0x11c45, direction::L},
    {0x11c50, 0x11c6c, direction::L},   {0x11c70, 0x11c8f, direction::L},
    {0x11c92, 0x11ca7, direction::NSM}, {0x11ca9, 0x11ca9, direction::L},
    {0x11caa, 0x11cb0, direction::NSM}, {0x11cb1, 0x11cb1, direction::L},
    {0x11cb2, 0x11cb3, direction::NSM}, {0x11cb4, 0x11cb4, direction::L},
    {0x11cb5, 0x11cb6, direction::NSM}, {0x11d00, 0x11d06, direction::L},
    {0x11d08, 0x11d09, direction::L},   {0x11d0b, 0x11d30, direction::L},
    {0x11d31, 0x11d36, direction::NSM}, {0x11d3a, 0x11d3a, direction::NSM},
    {0x11d3c, 0x11d3d, direction::NSM}, {0x11d3f, 0x11d45, direction::NSM},
    {0x11d46, 0x11d46, direction::L},   {0x11d47, 0x11d47, direction::NSM},
    {0x11d50, 0x11d59, direction::L},   {0x11d60, 0x11d65, direction::L},
    {0x11d67, 0x11d68, direction::L},   {0x11d6a, 0x11d8e, direction::L},
    {0x11d90, 0x11d91, direction::NSM}, {0x11d93, 0x11d94, direction::L},
    {0x11d95, 0x11d95, direction::NSM}, {0x11d96, 0x11d96, direction::L},
    {0x11d97, 0x11d97, direction::NSM}, {0x11d98, 0x11d98, direction::L},
    {0x11da0, 0x11da9, direction::L},   {0x11ee0, 0x11ef2, direction::L},
    {0x11ef3, 0x11ef4, direction::NSM}, {0x11ef5, 0x11ef8, direction::L},
    {0x11fb0, 0x11fb0, direction::L},   {0x11fc0, 0x11fd4, direction::L},
    {0x11fd5, 0x11fdc, direction::ON},  {0x11fdd, 0x11fe0, direction::ET},
    {0x11fe1, 0x11ff1, direction::ON},  {0x11fff, 0x12399, direction::L},
    {0x12400, 0x1246e, direction::L},   {0x12470, 0x12474, direction::L},
    {0x12480, 0x12543, direction::L},   {0x13000, 0x1342e, direction::L},
    {0x13430, 0x13438, direction::L},   {0x14400, 0x14646, direction::L},
    {0x16800, 0x16a38, direction::L},   {0x16a40, 0x16a5e, direction::L},
    {0x16a60, 0x16a69, direction::L},   {0x16a6e, 0x16a6f, direction::L},
    {0x16ad0, 0x16aed, direction::L},   {0x16af0, 0x16af4, direction::NSM},
    {0x16af5, 0x16af5, direction::L},   {0x16b00, 0x16b2f, direction::L},
    {0x16b30, 0x16b36, direction::NSM}, {0x16b37, 0x16b45, direction::L},
    {0x16b50, 0x16b59, direction::L},   {0x16b5b, 0x16b61, direction::L},
    {0x16b63, 0x16b77, direction::L},   {0x16b7d, 0x16b8f, direction::L},
    {0x16e40, 0x16e9a, direction::L},   {0x16f00, 0x16f4a, direction::L},
    {0x16f4f, 0x16f4f, direction::NSM}, {0x16f50, 0x16f87, direction::L},
    {0x16f8f, 0x16f92, direction::NSM}, {0x16f93, 0x16f9f, direction::L},
    {0x16fe0, 0x16fe1, direction::L},   {0x16fe2, 0x16fe2, direction::ON},
    {0x16fe3, 0x16fe3, direction::L},   {0x16fe4, 0x16fe4, direction::NSM},
    {0x16ff0, 0x16ff1, direction::L},   {0x17000, 0x187f7, direction::L},
    {0x18800, 0x18cd5, direction::L},   {0x18d00, 0x18d08, direction::L},
    {0x1b000, 0x1b11e, direction::L},   {0x1b150, 0x1b152, direction::L},
    {0x1b164, 0x1b167, direction::L},   {0x1b170, 0x1b2fb, direction::L},
    {0x1bc00, 0x1bc6a, direction::L},   {0x1bc70, 0x1bc7c, direction::L},
    {0x1bc80, 0x1bc88, direction::L},   {0x1bc90, 0x1bc99, direction::L},
    {0x1bc9c, 0x1bc9c, direction::L},   {0x1bc9d, 0x1bc9e, direction::NSM},
    {0x1bc9f, 0x1bc9f, direction::L},   {0x1bca0, 0x1bca3, direction::BN},
    {0x1d000, 0x1d0f5, direction::L},   {0x1d100, 0x1d126, direction::L},
    {0x1d129, 0x1d166, direction::L},   {0x1d167, 0x1d169, direction::NSM},
    {0x1d16a, 0x1d172, direction::L},   {0x1d173, 0x1d17a, direction::BN},
    {0x1d17b, 0x1d182, direction::NSM}, {0x1d183, 0x1d184, direction::L},
    {0x1d185, 0x1d18b, direction::NSM}, {0x1d18c, 0x1d1a9, direction::L},
    {0x1d1aa, 0x1d1ad, direction::NSM}, {0x1d1ae, 0x1d1e8, direction::L},
    {0x1d200, 0x1d241, direction::ON},  {0x1d242, 0x1d244, direction::NSM},
    {0x1d245, 0x1d245, direction::ON},  {0x1d2e0, 0x1d2f3, direction::L},
    {0x1d300, 0x1d356, direction::ON},  {0x1d360, 0x1d378, direction::L},
    {0x1d400, 0x1d454, direction::L},   {0x1d456, 0x1d49c, direction::L},
    {0x1d49e, 0x1d49f, direction::L},   {0x1d4a2, 0x1d4a2, direction::L},
    {0x1d4a5, 0x1d4a6, direction::L},   {0x1d4a9, 0x1d4ac, direction::L},
    {0x1d4ae, 0x1d4b9, direction::L},   {0x1d4bb, 0x1d4bb, direction::L},
    {0x1d4bd, 0x1d4c3, direction::L},   {0x1d4c5, 0x1d505, direction::L},
    {0x1d507, 0x1d50a, direction::L},   {0x1d50d, 0x1d514, direction::L},
    {0x1d516, 0x1d51c, direction::L},   {0x1d51e, 0x1d539, direction::L},
    {0x1d53b, 0x1d53e, direction::L},   {0x1d540, 0x1d544, direction::L},
    {0x1d546, 0x1d546, direction::L},   {0x1d54a, 0x1d550, direction::L},
    {0x1d552, 0x1d6a5, direction::L},   {0x1d6a8, 0x1d6da, direction::L},
    {0x1d6db, 0x1d6db, direction::ON},  {0x1d6dc, 0x1d714, direction::L},
    {0x1d715, 0x1d715, direction::ON},  {0x1d716, 0x1d74e, direction::L},
    {0x1d74f, 0x1d74f, direction::ON},  {0x1d750, 0x1d788, direction::L},
    {0x1d789, 0x1d789, direction::ON},  {0x1d78a, 0x1d7c2, direction::L},
    {0x1d7c3, 0x1d7c3, direction::ON},  {0x1d7c4, 0x1d7cb, direction::L},
    {0x1d7ce, 0x1d7ff, direction::EN},  {0x1d800, 0x1d9ff, direction::L},
    {0x1da00, 0x1da36, direction::NSM}, {0x1da37, 0x1da3a, direction::L},
    {0x1da3b, 0x1da6c, direction::NSM}, {0x1da6d, 0x1da74, direction::L},
    {0x1da75, 0x1da75, direction::NSM}, {0x1da76, 0x1da83, direction::L},
    {0x1da84, 0x1da84, direction::NSM}, {0x1da85, 0x1da8b, direction::L},
    {0x1da9b, 0x1da9f, direction::NSM}, {0x1daa1, 0x1daaf, direction::NSM},
    {0x1e000, 0x1e006, direction::NSM}, {0x1e008, 0x1e018, direction::NSM},
    {0x1e01b, 0x1e021, direction::NSM}, {0x1e023, 0x1e024, direction::NSM},
    {0x1e026, 0x1e02a, direction::NSM}, {0x1e100, 0x1e12c, direction::L},
    {0x1e130, 0x1e136, direction::NSM}, {0x1e137, 0x1e13d, direction::L},
    {0x1e140, 0x1e149, direction::L},   {0x1e14e, 0x1e14f, direction::L},
    {0x1e2c0, 0x1e2eb, direction::L},   {0x1e2ec, 0x1e2ef, direction::NSM},
    {0x1e2f0, 0x1e2f9, direction::L},   {0x1e2ff, 0x1e2ff, direction::ET},
    {0x1e800, 0x1e8c4, direction::R},   {0x1e8c7, 0x1e8cf, direction::R},
    {0x1e8d0, 0x1e8d6, direction::NSM}, {0x1e900, 0x1e943, direction::R},
    {0x1e944, 0x1e94a, direction::NSM}, {0x1e94b, 0x1e94b, direction::R},
    {0x1e950, 0x1e959, direction::R},   {0x1e95e, 0x1e95f, direction::R},
    {0x1ec71, 0x1ecb4, direction::AL},  {0x1ed01, 0x1ed3d, direction::AL},
    {0x1ee00, 0x1ee03, direction::AL},  {0x1ee05, 0x1ee1f, direction::AL},
    {0x1ee21, 0x1ee22, direction::AL},  {0x1ee24, 0x1ee24, direction::AL},
    {0x1ee27, 0x1ee27, direction::AL},  {0x1ee29, 0x1ee32, direction::AL},
    {0x1ee34, 0x1ee37, direction::AL},  {0x1ee39, 0x1ee39, direction::AL},
    {0x1ee3b, 0x1ee3b, direction::AL},  {0x1ee42, 0x1ee42, direction::AL},
    {0x1ee47, 0x1ee47, direction::AL},  {0x1ee49, 0x1ee49, direction::AL},
    {0x1ee4b, 0x1ee4b, direction::AL},  {0x1ee4d, 0x1ee4f, direction::AL},
    {0x1ee51, 0x1ee52, direction::AL},  {0x1ee54, 0x1ee54, direction::AL},
    {0x1ee57, 0x1ee57, direction::AL},  {0x1ee59, 0x1ee59, direction::AL},
    {0x1ee5b, 0x1ee5b, direction::AL},  {0x1ee5d, 0x1ee5d, direction::AL},
    {0x1ee5f, 0x1ee5f, direction::AL},  {0x1ee61, 0x1ee62, direction::AL},
    {0x1ee64, 0x1ee64, direction::AL},  {0x1ee67, 0x1ee6a, direction::AL},
    {0x1ee6c, 0x1ee72, direction::AL},  {0x1ee74, 0x1ee77, direction::AL},
    {0x1ee79, 0x1ee7c, direction::AL},  {0x1ee7e, 0x1ee7e, direction::AL},
    {0x1ee80, 0x1ee89, direction::AL},  {0x1ee8b, 0x1ee9b, direction::AL},
    {0x1eea1, 0x1eea3, direction::AL},  {0x1eea5, 0x1eea9, direction::AL},
    {0x1eeab, 0x1eebb, direction::AL},  {0x1eef0, 0x1eef1, direction::ON},
    {0x1f000, 0x1f02b, direction::ON},  {0x1f030, 0x1f093, direction::ON},
    {0x1f0a0, 0x1f0ae, direction::ON},  {0x1f0b1, 0x1f0bf, direction::ON},
    {0x1f0c1, 0x1f0cf, direction::ON},  {0x1f0d1, 0x1f0f5, direction::ON},
    {0x1f100, 0x1f10a, direction::EN},  {0x1f10b, 0x1f10f, direction::ON},
    {0x1f110, 0x1f12e, direction::L},   {0x1f12f, 0x1f12f, direction::ON},
    {0x1f130, 0x1f169, direction::L},   {0x1f16a, 0x1f16f, direction::ON},
    {0x1f170, 0x1f1ac, direction::L},   {0x1f1ad, 0x1f1ad, direction::ON},
    {0x1f1e6, 0x1f202, direction::L},   {0x1f210, 0x1f23b, direction::L},
    {0x1f240, 0x1f248, direction::L},   {0x1f250, 0x1f251, direction::L},
    {0x1f260, 0x1f265, direction::ON},  {0x1f300, 0x1f6d7, direction::ON},
    {0x1f6e0, 0x1f6ec, direction::ON},  {0x1f6f0, 0x1f6fc, direction::ON},
    {0x1f700, 0x1f773, direction::ON},  {0x1f780, 0x1f7d8, direction::ON},
    {0x1f7e0, 0x1f7eb, direction::ON},  {0x1f800, 0x1f80b, direction::ON},
    {0x1f810, 0x1f847, direction::ON},  {0x1f850, 0x1f859, direction::ON},
    {0x1f860, 0x1f887, direction::ON},  {0x1f890, 0x1f8ad, direction::ON},
    {0x1f8b0, 0x1f8b1, direction::ON},  {0x1f900, 0x1f978, direction::ON},
    {0x1f97a, 0x1f9cb, direction::ON},  {0x1f9cd, 0x1fa53, direction::ON},
    {0x1fa60, 0x1fa6d, direction::ON},  {0x1fa70, 0x1fa74, direction::ON},
    {0x1fa78, 0x1fa7a, direction::ON},  {0x1fa80, 0x1fa86, direction::ON},
    {0x1fa90, 0x1faa8, direction::ON},  {0x1fab0, 0x1fab6, direction::ON},
    {0x1fac0, 0x1fac2, direction::ON},  {0x1fad0, 0x1fad6, direction::ON},
    {0x1fb00, 0x1fb92, direction::ON},  {0x1fb94, 0x1fbca, direction::ON},
    {0x1fbf0, 0x1fbf9, direction::EN},  {0x20000, 0x2a6dd, direction::L},
    {0x2a700, 0x2b734, direction::L},   {0x2b740, 0x2b81d, direction::L},
    {0x2b820, 0x2cea1, direction::L},   {0x2ceb0, 0x2ebe0, direction::L},
    {0x2f800, 0x2fa1d, direction::L},   {0x30000, 0x3134a, direction::L},
    {0xe0001, 0xe0001, direction::BN},  {0xe0020, 0xe007f, direction::BN},
    {0xe0100, 0xe01ef, direction::NSM}, {0xf0000, 0xffffd, direction::L},
    {0x100000, 0x10fffd, direction::L}};

// CheckJoiners and CheckBidi are true for URL specification.

inline static direction find_direction(uint32_t code_point) noexcept {
  auto it = std::lower_bound(
      std::begin(dir_table), std::end(dir_table), code_point,
      [](const directions& d, uint32_t c) { return d.final_code < c; });

  // next check is almost surely in vain, but we use it for safety.
  if (it == std::end(dir_table)) {
    return direction::NONE;
  }
  // We have that d.final_code >= c.
  if (code_point >= it->start_code) {
    return it->direct;
  }
  return direction::NONE;
}

inline static size_t find_last_not_of_nsm(
    const std::u32string_view label) noexcept {
  for (int i = label.size() - 1; i >= 0; i--)
    if (find_direction(label[i]) != direction::NSM) return i;

  return std::u32string_view::npos;
}

// An RTL label is a label that contains at least one character of type R, AL,
// or AN. https://www.rfc-editor.org/rfc/rfc5893#section-2
inline static bool is_rtl_label(const std::u32string_view label) noexcept {
  const size_t mask =
      (1u << direction::R) | (1u << direction::AL) | (1u << direction::AN);

  size_t directions = 0;
  for (size_t i = 0; i < label.size(); i++) {
    directions |= 1u << find_direction(label[i]);
  }
  return (directions & mask) != 0;
}

bool is_label_valid(const std::u32string_view label) {
  if (label.empty()) {
    return true;
  }

  ///////////////
  // We have a normalization step which ensures that we are in NFC.
  // If we receive punycode, we normalize and check that the normalized
  // version matches the original.
  // --------------------------------------
  // The label must be in Unicode Normalization Form NFC.

  // Current URL standard indicatest that CheckHyphens is set to false.
  // ---------------------------------------
  // If CheckHyphens, the label must not contain a U+002D HYPHEN-MINUS character
  // in both the third and fourth positions. If CheckHyphens, the label must
  // neither begin nor end with a U+002D HYPHEN-MINUS character.

  // This is not necessary because we segment the
  // labels by '.'.
  // ---------------------------------------
  // The label must not contain a U+002E ( . ) FULL STOP.
  // if (label.find('.') != std::string_view::npos) return false;

  // The label must not begin with a combining mark, that is:
  // General_Category=Mark.
  constexpr static uint32_t combining[] = {
      0x300,   0x301,   0x302,   0x303,   0x304,   0x305,   0x306,   0x307,
      0x308,   0x309,   0x30a,   0x30b,   0x30c,   0x30d,   0x30e,   0x30f,
      0x310,   0x311,   0x312,   0x313,   0x314,   0x315,   0x316,   0x317,
      0x318,   0x319,   0x31a,   0x31b,   0x31c,   0x31d,   0x31e,   0x31f,
      0x320,   0x321,   0x322,   0x323,   0x324,   0x325,   0x326,   0x327,
      0x328,   0x329,   0x32a,   0x32b,   0x32c,   0x32d,   0x32e,   0x32f,
      0x330,   0x331,   0x332,   0x333,   0x334,   0x335,   0x336,   0x337,
      0x338,   0x339,   0x33a,   0x33b,   0x33c,   0x33d,   0x33e,   0x33f,
      0x340,   0x341,   0x342,   0x343,   0x344,   0x345,   0x346,   0x347,
      0x348,   0x349,   0x34a,   0x34b,   0x34c,   0x34d,   0x34e,   0x34f,
      0x350,   0x351,   0x352,   0x353,   0x354,   0x355,   0x356,   0x357,
      0x358,   0x359,   0x35a,   0x35b,   0x35c,   0x35d,   0x35e,   0x35f,
      0x360,   0x361,   0x362,   0x363,   0x364,   0x365,   0x366,   0x367,
      0x368,   0x369,   0x36a,   0x36b,   0x36c,   0x36d,   0x36e,   0x36f,
      0x483,   0x484,   0x485,   0x486,   0x487,   0x488,   0x489,   0x591,
      0x592,   0x593,   0x594,   0x595,   0x596,   0x597,   0x598,   0x599,
      0x59a,   0x59b,   0x59c,   0x59d,   0x59e,   0x59f,   0x5a0,   0x5a1,
      0x5a2,   0x5a3,   0x5a4,   0x5a5,   0x5a6,   0x5a7,   0x5a8,   0x5a9,
      0x5aa,   0x5ab,   0x5ac,   0x5ad,   0x5ae,   0x5af,   0x5b0,   0x5b1,
      0x5b2,   0x5b3,   0x5b4,   0x5b5,   0x5b6,   0x5b7,   0x5b8,   0x5b9,
      0x5ba,   0x5bb,   0x5bc,   0x5bd,   0x5bf,   0x5c1,   0x5c2,   0x5c4,
      0x5c5,   0x5c7,   0x610,   0x611,   0x612,   0x613,   0x614,   0x615,
      0x616,   0x617,   0x618,   0x619,   0x61a,   0x64b,   0x64c,   0x64d,
      0x64e,   0x64f,   0x650,   0x651,   0x652,   0x653,   0x654,   0x655,
      0x656,   0x657,   0x658,   0x659,   0x65a,   0x65b,   0x65c,   0x65d,
      0x65e,   0x65f,   0x670,   0x6d6,   0x6d7,   0x6d8,   0x6d9,   0x6da,
      0x6db,   0x6dc,   0x6df,   0x6e0,   0x6e1,   0x6e2,   0x6e3,   0x6e4,
      0x6e7,   0x6e8,   0x6ea,   0x6eb,   0x6ec,   0x6ed,   0x711,   0x730,
      0x731,   0x732,   0x733,   0x734,   0x735,   0x736,   0x737,   0x738,
      0x739,   0x73a,   0x73b,   0x73c,   0x73d,   0x73e,   0x73f,   0x740,
      0x741,   0x742,   0x743,   0x744,   0x745,   0x746,   0x747,   0x748,
      0x749,   0x74a,   0x7a6,   0x7a7,   0x7a8,   0x7a9,   0x7aa,   0x7ab,
      0x7ac,   0x7ad,   0x7ae,   0x7af,   0x7b0,   0x7eb,   0x7ec,   0x7ed,
      0x7ee,   0x7ef,   0x7f0,   0x7f1,   0x7f2,   0x7f3,   0x7fd,   0x816,
      0x817,   0x818,   0x819,   0x81b,   0x81c,   0x81d,   0x81e,   0x81f,
      0x820,   0x821,   0x822,   0x823,   0x825,   0x826,   0x827,   0x829,
      0x82a,   0x82b,   0x82c,   0x82d,   0x859,   0x85a,   0x85b,   0x8d3,
      0x8d4,   0x8d5,   0x8d6,   0x8d7,   0x8d8,   0x8d9,   0x8da,   0x8db,
      0x8dc,   0x8dd,   0x8de,   0x8df,   0x8e0,   0x8e1,   0x8e3,   0x8e4,
      0x8e5,   0x8e6,   0x8e7,   0x8e8,   0x8e9,   0x8ea,   0x8eb,   0x8ec,
      0x8ed,   0x8ee,   0x8ef,   0x8f0,   0x8f1,   0x8f2,   0x8f3,   0x8f4,
      0x8f5,   0x8f6,   0x8f7,   0x8f8,   0x8f9,   0x8fa,   0x8fb,   0x8fc,
      0x8fd,   0x8fe,   0x8ff,   0x900,   0x901,   0x902,   0x903,   0x93a,
      0x93b,   0x93c,   0x93e,   0x93f,   0x940,   0x941,   0x942,   0x943,
      0x944,   0x945,   0x946,   0x947,   0x948,   0x949,   0x94a,   0x94b,
      0x94c,   0x94d,   0x94e,   0x94f,   0x951,   0x952,   0x953,   0x954,
      0x955,   0x956,   0x957,   0x962,   0x963,   0x981,   0x982,   0x983,
      0x9bc,   0x9be,   0x9bf,   0x9c0,   0x9c1,   0x9c2,   0x9c3,   0x9c4,
      0x9c7,   0x9c8,   0x9cb,   0x9cc,   0x9cd,   0x9d7,   0x9e2,   0x9e3,
      0x9fe,   0xa01,   0xa02,   0xa03,   0xa3c,   0xa3e,   0xa3f,   0xa40,
      0xa41,   0xa42,   0xa47,   0xa48,   0xa4b,   0xa4c,   0xa4d,   0xa51,
      0xa70,   0xa71,   0xa75,   0xa81,   0xa82,   0xa83,   0xabc,   0xabe,
      0xabf,   0xac0,   0xac1,   0xac2,   0xac3,   0xac4,   0xac5,   0xac7,
      0xac8,   0xac9,   0xacb,   0xacc,   0xacd,   0xae2,   0xae3,   0xafa,
      0xafb,   0xafc,   0xafd,   0xafe,   0xaff,   0xb01,   0xb02,   0xb03,
      0xb3c,   0xb3e,   0xb3f,   0xb40,   0xb41,   0xb42,   0xb43,   0xb44,
      0xb47,   0xb48,   0xb4b,   0xb4c,   0xb4d,   0xb55,   0xb56,   0xb57,
      0xb62,   0xb63,   0xb82,   0xbbe,   0xbbf,   0xbc0,   0xbc1,   0xbc2,
      0xbc6,   0xbc7,   0xbc8,   0xbca,   0xbcb,   0xbcc,   0xbcd,   0xbd7,
      0xc00,   0xc01,   0xc02,   0xc03,   0xc04,   0xc3e,   0xc3f,   0xc40,
      0xc41,   0xc42,   0xc43,   0xc44,   0xc46,   0xc47,   0xc48,   0xc4a,
      0xc4b,   0xc4c,   0xc4d,   0xc55,   0xc56,   0xc62,   0xc63,   0xc81,
      0xc82,   0xc83,   0xcbc,   0xcbe,   0xcbf,   0xcc0,   0xcc1,   0xcc2,
      0xcc3,   0xcc4,   0xcc6,   0xcc7,   0xcc8,   0xcca,   0xccb,   0xccc,
      0xccd,   0xcd5,   0xcd6,   0xce2,   0xce3,   0xd00,   0xd01,   0xd02,
      0xd03,   0xd3b,   0xd3c,   0xd3e,   0xd3f,   0xd40,   0xd41,   0xd42,
      0xd43,   0xd44,   0xd46,   0xd47,   0xd48,   0xd4a,   0xd4b,   0xd4c,
      0xd4d,   0xd57,   0xd62,   0xd63,   0xd81,   0xd82,   0xd83,   0xdca,
      0xdcf,   0xdd0,   0xdd1,   0xdd2,   0xdd3,   0xdd4,   0xdd6,   0xdd8,
      0xdd9,   0xdda,   0xddb,   0xddc,   0xddd,   0xdde,   0xddf,   0xdf2,
      0xdf3,   0xe31,   0xe34,   0xe35,   0xe36,   0xe37,   0xe38,   0xe39,
      0xe3a,   0xe47,   0xe48,   0xe49,   0xe4a,   0xe4b,   0xe4c,   0xe4d,
      0xe4e,   0xeb1,   0xeb4,   0xeb5,   0xeb6,   0xeb7,   0xeb8,   0xeb9,
      0xeba,   0xebb,   0xebc,   0xec8,   0xec9,   0xeca,   0xecb,   0xecc,
      0xecd,   0xf18,   0xf19,   0xf35,   0xf37,   0xf39,   0xf3e,   0xf3f,
      0xf71,   0xf72,   0xf73,   0xf74,   0xf75,   0xf76,   0xf77,   0xf78,
      0xf79,   0xf7a,   0xf7b,   0xf7c,   0xf7d,   0xf7e,   0xf7f,   0xf80,
      0xf81,   0xf82,   0xf83,   0xf84,   0xf86,   0xf87,   0xf8d,   0xf8e,
      0xf8f,   0xf90,   0xf91,   0xf92,   0xf93,   0xf94,   0xf95,   0xf96,
      0xf97,   0xf99,   0xf9a,   0xf9b,   0xf9c,   0xf9d,   0xf9e,   0xf9f,
      0xfa0,   0xfa1,   0xfa2,   0xfa3,   0xfa4,   0xfa5,   0xfa6,   0xfa7,
      0xfa8,   0xfa9,   0xfaa,   0xfab,   0xfac,   0xfad,   0xfae,   0xfaf,
      0xfb0,   0xfb1,   0xfb2,   0xfb3,   0xfb4,   0xfb5,   0xfb6,   0xfb7,
      0xfb8,   0xfb9,   0xfba,   0xfbb,   0xfbc,   0xfc6,   0x102b,  0x102c,
      0x102d,  0x102e,  0x102f,  0x1030,  0x1031,  0x1032,  0x1033,  0x1034,
      0x1035,  0x1036,  0x1037,  0x1038,  0x1039,  0x103a,  0x103b,  0x103c,
      0x103d,  0x103e,  0x1056,  0x1057,  0x1058,  0x1059,  0x105e,  0x105f,
      0x1060,  0x1062,  0x1063,  0x1064,  0x1067,  0x1068,  0x1069,  0x106a,
      0x106b,  0x106c,  0x106d,  0x1071,  0x1072,  0x1073,  0x1074,  0x1082,
      0x1083,  0x1084,  0x1085,  0x1086,  0x1087,  0x1088,  0x1089,  0x108a,
      0x108b,  0x108c,  0x108d,  0x108f,  0x109a,  0x109b,  0x109c,  0x109d,
      0x135d,  0x135e,  0x135f,  0x1712,  0x1713,  0x1714,  0x1732,  0x1733,
      0x1734,  0x1752,  0x1753,  0x1772,  0x1773,  0x17b4,  0x17b5,  0x17b6,
      0x17b7,  0x17b8,  0x17b9,  0x17ba,  0x17bb,  0x17bc,  0x17bd,  0x17be,
      0x17bf,  0x17c0,  0x17c1,  0x17c2,  0x17c3,  0x17c4,  0x17c5,  0x17c6,
      0x17c7,  0x17c8,  0x17c9,  0x17ca,  0x17cb,  0x17cc,  0x17cd,  0x17ce,
      0x17cf,  0x17d0,  0x17d1,  0x17d2,  0x17d3,  0x17dd,  0x180b,  0x180c,
      0x180d,  0x1885,  0x1886,  0x18a9,  0x1920,  0x1921,  0x1922,  0x1923,
      0x1924,  0x1925,  0x1926,  0x1927,  0x1928,  0x1929,  0x192a,  0x192b,
      0x1930,  0x1931,  0x1932,  0x1933,  0x1934,  0x1935,  0x1936,  0x1937,
      0x1938,  0x1939,  0x193a,  0x193b,  0x1a17,  0x1a18,  0x1a19,  0x1a1a,
      0x1a1b,  0x1a55,  0x1a56,  0x1a57,  0x1a58,  0x1a59,  0x1a5a,  0x1a5b,
      0x1a5c,  0x1a5d,  0x1a5e,  0x1a60,  0x1a61,  0x1a62,  0x1a63,  0x1a64,
      0x1a65,  0x1a66,  0x1a67,  0x1a68,  0x1a69,  0x1a6a,  0x1a6b,  0x1a6c,
      0x1a6d,  0x1a6e,  0x1a6f,  0x1a70,  0x1a71,  0x1a72,  0x1a73,  0x1a74,
      0x1a75,  0x1a76,  0x1a77,  0x1a78,  0x1a79,  0x1a7a,  0x1a7b,  0x1a7c,
      0x1a7f,  0x1ab0,  0x1ab1,  0x1ab2,  0x1ab3,  0x1ab4,  0x1ab5,  0x1ab6,
      0x1ab7,  0x1ab8,  0x1ab9,  0x1aba,  0x1abb,  0x1abc,  0x1abd,  0x1abe,
      0x1abf,  0x1ac0,  0x1b00,  0x1b01,  0x1b02,  0x1b03,  0x1b04,  0x1b34,
      0x1b35,  0x1b36,  0x1b37,  0x1b38,  0x1b39,  0x1b3a,  0x1b3b,  0x1b3c,
      0x1b3d,  0x1b3e,  0x1b3f,  0x1b40,  0x1b41,  0x1b42,  0x1b43,  0x1b44,
      0x1b6b,  0x1b6c,  0x1b6d,  0x1b6e,  0x1b6f,  0x1b70,  0x1b71,  0x1b72,
      0x1b73,  0x1b80,  0x1b81,  0x1b82,  0x1ba1,  0x1ba2,  0x1ba3,  0x1ba4,
      0x1ba5,  0x1ba6,  0x1ba7,  0x1ba8,  0x1ba9,  0x1baa,  0x1bab,  0x1bac,
      0x1bad,  0x1be6,  0x1be7,  0x1be8,  0x1be9,  0x1bea,  0x1beb,  0x1bec,
      0x1bed,  0x1bee,  0x1bef,  0x1bf0,  0x1bf1,  0x1bf2,  0x1bf3,  0x1c24,
      0x1c25,  0x1c26,  0x1c27,  0x1c28,  0x1c29,  0x1c2a,  0x1c2b,  0x1c2c,
      0x1c2d,  0x1c2e,  0x1c2f,  0x1c30,  0x1c31,  0x1c32,  0x1c33,  0x1c34,
      0x1c35,  0x1c36,  0x1c37,  0x1cd0,  0x1cd1,  0x1cd2,  0x1cd4,  0x1cd5,
      0x1cd6,  0x1cd7,  0x1cd8,  0x1cd9,  0x1cda,  0x1cdb,  0x1cdc,  0x1cdd,
      0x1cde,  0x1cdf,  0x1ce0,  0x1ce1,  0x1ce2,  0x1ce3,  0x1ce4,  0x1ce5,
      0x1ce6,  0x1ce7,  0x1ce8,  0x1ced,  0x1cf4,  0x1cf7,  0x1cf8,  0x1cf9,
      0x1dc0,  0x1dc1,  0x1dc2,  0x1dc3,  0x1dc4,  0x1dc5,  0x1dc6,  0x1dc7,
      0x1dc8,  0x1dc9,  0x1dca,  0x1dcb,  0x1dcc,  0x1dcd,  0x1dce,  0x1dcf,
      0x1dd0,  0x1dd1,  0x1dd2,  0x1dd3,  0x1dd4,  0x1dd5,  0x1dd6,  0x1dd7,
      0x1dd8,  0x1dd9,  0x1dda,  0x1ddb,  0x1ddc,  0x1ddd,  0x1dde,  0x1ddf,
      0x1de0,  0x1de1,  0x1de2,  0x1de3,  0x1de4,  0x1de5,  0x1de6,  0x1de7,
      0x1de8,  0x1de9,  0x1dea,  0x1deb,  0x1dec,  0x1ded,  0x1dee,  0x1def,
      0x1df0,  0x1df1,  0x1df2,  0x1df3,  0x1df4,  0x1df5,  0x1df6,  0x1df7,
      0x1df8,  0x1df9,  0x1dfb,  0x1dfc,  0x1dfd,  0x1dfe,  0x1dff,  0x20d0,
      0x20d1,  0x20d2,  0x20d3,  0x20d4,  0x20d5,  0x20d6,  0x20d7,  0x20d8,
      0x20d9,  0x20da,  0x20db,  0x20dc,  0x20dd,  0x20de,  0x20df,  0x20e0,
      0x20e1,  0x20e2,  0x20e3,  0x20e4,  0x20e5,  0x20e6,  0x20e7,  0x20e8,
      0x20e9,  0x20ea,  0x20eb,  0x20ec,  0x20ed,  0x20ee,  0x20ef,  0x20f0,
      0x2cef,  0x2cf0,  0x2cf1,  0x2d7f,  0x2de0,  0x2de1,  0x2de2,  0x2de3,
      0x2de4,  0x2de5,  0x2de6,  0x2de7,  0x2de8,  0x2de9,  0x2dea,  0x2deb,
      0x2dec,  0x2ded,  0x2dee,  0x2def,  0x2df0,  0x2df1,  0x2df2,  0x2df3,
      0x2df4,  0x2df5,  0x2df6,  0x2df7,  0x2df8,  0x2df9,  0x2dfa,  0x2dfb,
      0x2dfc,  0x2dfd,  0x2dfe,  0x2dff,  0x302a,  0x302b,  0x302c,  0x302d,
      0x302e,  0x302f,  0x3099,  0x309a,  0xa66f,  0xa670,  0xa671,  0xa672,
      0xa674,  0xa675,  0xa676,  0xa677,  0xa678,  0xa679,  0xa67a,  0xa67b,
      0xa67c,  0xa67d,  0xa69e,  0xa69f,  0xa6f0,  0xa6f1,  0xa802,  0xa806,
      0xa80b,  0xa823,  0xa824,  0xa825,  0xa826,  0xa827,  0xa82c,  0xa880,
      0xa881,  0xa8b4,  0xa8b5,  0xa8b6,  0xa8b7,  0xa8b8,  0xa8b9,  0xa8ba,
      0xa8bb,  0xa8bc,  0xa8bd,  0xa8be,  0xa8bf,  0xa8c0,  0xa8c1,  0xa8c2,
      0xa8c3,  0xa8c4,  0xa8c5,  0xa8e0,  0xa8e1,  0xa8e2,  0xa8e3,  0xa8e4,
      0xa8e5,  0xa8e6,  0xa8e7,  0xa8e8,  0xa8e9,  0xa8ea,  0xa8eb,  0xa8ec,
      0xa8ed,  0xa8ee,  0xa8ef,  0xa8f0,  0xa8f1,  0xa8ff,  0xa926,  0xa927,
      0xa928,  0xa929,  0xa92a,  0xa92b,  0xa92c,  0xa92d,  0xa947,  0xa948,
      0xa949,  0xa94a,  0xa94b,  0xa94c,  0xa94d,  0xa94e,  0xa94f,  0xa950,
      0xa951,  0xa952,  0xa953,  0xa980,  0xa981,  0xa982,  0xa983,  0xa9b3,
      0xa9b4,  0xa9b5,  0xa9b6,  0xa9b7,  0xa9b8,  0xa9b9,  0xa9ba,  0xa9bb,
      0xa9bc,  0xa9bd,  0xa9be,  0xa9bf,  0xa9c0,  0xa9e5,  0xaa29,  0xaa2a,
      0xaa2b,  0xaa2c,  0xaa2d,  0xaa2e,  0xaa2f,  0xaa30,  0xaa31,  0xaa32,
      0xaa33,  0xaa34,  0xaa35,  0xaa36,  0xaa43,  0xaa4c,  0xaa4d,  0xaa7b,
      0xaa7c,  0xaa7d,  0xaab0,  0xaab2,  0xaab3,  0xaab4,  0xaab7,  0xaab8,
      0xaabe,  0xaabf,  0xaac1,  0xaaeb,  0xaaec,  0xaaed,  0xaaee,  0xaaef,
      0xaaf5,  0xaaf6,  0xabe3,  0xabe4,  0xabe5,  0xabe6,  0xabe7,  0xabe8,
      0xabe9,  0xabea,  0xabec,  0xabed,  0xfb1e,  0xfe00,  0xfe01,  0xfe02,
      0xfe03,  0xfe04,  0xfe05,  0xfe06,  0xfe07,  0xfe08,  0xfe09,  0xfe0a,
      0xfe0b,  0xfe0c,  0xfe0d,  0xfe0e,  0xfe0f,  0xfe20,  0xfe21,  0xfe22,
      0xfe23,  0xfe24,  0xfe25,  0xfe26,  0xfe27,  0xfe28,  0xfe29,  0xfe2a,
      0xfe2b,  0xfe2c,  0xfe2d,  0xfe2e,  0xfe2f,  0x101fd, 0x102e0, 0x10376,
      0x10377, 0x10378, 0x10379, 0x1037a, 0x10a01, 0x10a02, 0x10a03, 0x10a05,
      0x10a06, 0x10a0c, 0x10a0d, 0x10a0e, 0x10a0f, 0x10a38, 0x10a39, 0x10a3a,
      0x10a3f, 0x10ae5, 0x10ae6, 0x10d24, 0x10d25, 0x10d26, 0x10d27, 0x10eab,
      0x10eac, 0x10f46, 0x10f47, 0x10f48, 0x10f49, 0x10f4a, 0x10f4b, 0x10f4c,
      0x10f4d, 0x10f4e, 0x10f4f, 0x10f50, 0x11000, 0x11001, 0x11002, 0x11038,
      0x11039, 0x1103a, 0x1103b, 0x1103c, 0x1103d, 0x1103e, 0x1103f, 0x11040,
      0x11041, 0x11042, 0x11043, 0x11044, 0x11045, 0x11046, 0x1107f, 0x11080,
      0x11081, 0x11082, 0x110b0, 0x110b1, 0x110b2, 0x110b3, 0x110b4, 0x110b5,
      0x110b6, 0x110b7, 0x110b8, 0x110b9, 0x110ba, 0x11100, 0x11101, 0x11102,
      0x11127, 0x11128, 0x11129, 0x1112a, 0x1112b, 0x1112c, 0x1112d, 0x1112e,
      0x1112f, 0x11130, 0x11131, 0x11132, 0x11133, 0x11134, 0x11145, 0x11146,
      0x11173, 0x11180, 0x11181, 0x11182, 0x111b3, 0x111b4, 0x111b5, 0x111b6,
      0x111b7, 0x111b8, 0x111b9, 0x111ba, 0x111bb, 0x111bc, 0x111bd, 0x111be,
      0x111bf, 0x111c0, 0x111c9, 0x111ca, 0x111cb, 0x111cc, 0x111ce, 0x111cf,
      0x1122c, 0x1122d, 0x1122e, 0x1122f, 0x11230, 0x11231, 0x11232, 0x11233,
      0x11234, 0x11235, 0x11236, 0x11237, 0x1123e, 0x112df, 0x112e0, 0x112e1,
      0x112e2, 0x112e3, 0x112e4, 0x112e5, 0x112e6, 0x112e7, 0x112e8, 0x112e9,
      0x112ea, 0x11300, 0x11301, 0x11302, 0x11303, 0x1133b, 0x1133c, 0x1133e,
      0x1133f, 0x11340, 0x11341, 0x11342, 0x11343, 0x11344, 0x11347, 0x11348,
      0x1134b, 0x1134c, 0x1134d, 0x11357, 0x11362, 0x11363, 0x11366, 0x11367,
      0x11368, 0x11369, 0x1136a, 0x1136b, 0x1136c, 0x11370, 0x11371, 0x11372,
      0x11373, 0x11374, 0x11435, 0x11436, 0x11437, 0x11438, 0x11439, 0x1143a,
      0x1143b, 0x1143c, 0x1143d, 0x1143e, 0x1143f, 0x11440, 0x11441, 0x11442,
      0x11443, 0x11444, 0x11445, 0x11446, 0x1145e, 0x114b0, 0x114b1, 0x114b2,
      0x114b3, 0x114b4, 0x114b5, 0x114b6, 0x114b7, 0x114b8, 0x114b9, 0x114ba,
      0x114bb, 0x114bc, 0x114bd, 0x114be, 0x114bf, 0x114c0, 0x114c1, 0x114c2,
      0x114c3, 0x115af, 0x115b0, 0x115b1, 0x115b2, 0x115b3, 0x115b4, 0x115b5,
      0x115b8, 0x115b9, 0x115ba, 0x115bb, 0x115bc, 0x115bd, 0x115be, 0x115bf,
      0x115c0, 0x115dc, 0x115dd, 0x11630, 0x11631, 0x11632, 0x11633, 0x11634,
      0x11635, 0x11636, 0x11637, 0x11638, 0x11639, 0x1163a, 0x1163b, 0x1163c,
      0x1163d, 0x1163e, 0x1163f, 0x11640, 0x116ab, 0x116ac, 0x116ad, 0x116ae,
      0x116af, 0x116b0, 0x116b1, 0x116b2, 0x116b3, 0x116b4, 0x116b5, 0x116b6,
      0x116b7, 0x1171d, 0x1171e, 0x1171f, 0x11720, 0x11721, 0x11722, 0x11723,
      0x11724, 0x11725, 0x11726, 0x11727, 0x11728, 0x11729, 0x1172a, 0x1172b,
      0x1182c, 0x1182d, 0x1182e, 0x1182f, 0x11830, 0x11831, 0x11832, 0x11833,
      0x11834, 0x11835, 0x11836, 0x11837, 0x11838, 0x11839, 0x1183a, 0x11930,
      0x11931, 0x11932, 0x11933, 0x11934, 0x11935, 0x11937, 0x11938, 0x1193b,
      0x1193c, 0x1193d, 0x1193e, 0x11940, 0x11942, 0x11943, 0x119d1, 0x119d2,
      0x119d3, 0x119d4, 0x119d5, 0x119d6, 0x119d7, 0x119da, 0x119db, 0x119dc,
      0x119dd, 0x119de, 0x119df, 0x119e0, 0x119e4, 0x11a01, 0x11a02, 0x11a03,
      0x11a04, 0x11a05, 0x11a06, 0x11a07, 0x11a08, 0x11a09, 0x11a0a, 0x11a33,
      0x11a34, 0x11a35, 0x11a36, 0x11a37, 0x11a38, 0x11a39, 0x11a3b, 0x11a3c,
      0x11a3d, 0x11a3e, 0x11a47, 0x11a51, 0x11a52, 0x11a53, 0x11a54, 0x11a55,
      0x11a56, 0x11a57, 0x11a58, 0x11a59, 0x11a5a, 0x11a5b, 0x11a8a, 0x11a8b,
      0x11a8c, 0x11a8d, 0x11a8e, 0x11a8f, 0x11a90, 0x11a91, 0x11a92, 0x11a93,
      0x11a94, 0x11a95, 0x11a96, 0x11a97, 0x11a98, 0x11a99, 0x11c2f, 0x11c30,
      0x11c31, 0x11c32, 0x11c33, 0x11c34, 0x11c35, 0x11c36, 0x11c38, 0x11c39,
      0x11c3a, 0x11c3b, 0x11c3c, 0x11c3d, 0x11c3e, 0x11c3f, 0x11c92, 0x11c93,
      0x11c94, 0x11c95, 0x11c96, 0x11c97, 0x11c98, 0x11c99, 0x11c9a, 0x11c9b,
      0x11c9c, 0x11c9d, 0x11c9e, 0x11c9f, 0x11ca0, 0x11ca1, 0x11ca2, 0x11ca3,
      0x11ca4, 0x11ca5, 0x11ca6, 0x11ca7, 0x11ca9, 0x11caa, 0x11cab, 0x11cac,
      0x11cad, 0x11cae, 0x11caf, 0x11cb0, 0x11cb1, 0x11cb2, 0x11cb3, 0x11cb4,
      0x11cb5, 0x11cb6, 0x11d31, 0x11d32, 0x11d33, 0x11d34, 0x11d35, 0x11d36,
      0x11d3a, 0x11d3c, 0x11d3d, 0x11d3f, 0x11d40, 0x11d41, 0x11d42, 0x11d43,
      0x11d44, 0x11d45, 0x11d47, 0x11d8a, 0x11d8b, 0x11d8c, 0x11d8d, 0x11d8e,
      0x11d90, 0x11d91, 0x11d93, 0x11d94, 0x11d95, 0x11d96, 0x11d97, 0x11ef3,
      0x11ef4, 0x11ef5, 0x11ef6, 0x16af0, 0x16af1, 0x16af2, 0x16af3, 0x16af4,
      0x16b30, 0x16b31, 0x16b32, 0x16b33, 0x16b34, 0x16b35, 0x16b36, 0x16f4f,
      0x16f51, 0x16f52, 0x16f53, 0x16f54, 0x16f55, 0x16f56, 0x16f57, 0x16f58,
      0x16f59, 0x16f5a, 0x16f5b, 0x16f5c, 0x16f5d, 0x16f5e, 0x16f5f, 0x16f60,
      0x16f61, 0x16f62, 0x16f63, 0x16f64, 0x16f65, 0x16f66, 0x16f67, 0x16f68,
      0x16f69, 0x16f6a, 0x16f6b, 0x16f6c, 0x16f6d, 0x16f6e, 0x16f6f, 0x16f70,
      0x16f71, 0x16f72, 0x16f73, 0x16f74, 0x16f75, 0x16f76, 0x16f77, 0x16f78,
      0x16f79, 0x16f7a, 0x16f7b, 0x16f7c, 0x16f7d, 0x16f7e, 0x16f7f, 0x16f80,
      0x16f81, 0x16f82, 0x16f83, 0x16f84, 0x16f85, 0x16f86, 0x16f87, 0x16f8f,
      0x16f90, 0x16f91, 0x16f92, 0x16fe4, 0x16ff0, 0x16ff1, 0x1bc9d, 0x1bc9e,
      0x1d165, 0x1d166, 0x1d167, 0x1d168, 0x1d169, 0x1d16d, 0x1d16e, 0x1d16f,
      0x1d170, 0x1d171, 0x1d172, 0x1d17b, 0x1d17c, 0x1d17d, 0x1d17e, 0x1d17f,
      0x1d180, 0x1d181, 0x1d182, 0x1d185, 0x1d186, 0x1d187, 0x1d188, 0x1d189,
      0x1d18a, 0x1d18b, 0x1d1aa, 0x1d1ab, 0x1d1ac, 0x1d1ad, 0x1d242, 0x1d243,
      0x1d244, 0x1da00, 0x1da01, 0x1da02, 0x1da03, 0x1da04, 0x1da05, 0x1da06,
      0x1da07, 0x1da08, 0x1da09, 0x1da0a, 0x1da0b, 0x1da0c, 0x1da0d, 0x1da0e,
      0x1da0f, 0x1da10, 0x1da11, 0x1da12, 0x1da13, 0x1da14, 0x1da15, 0x1da16,
      0x1da17, 0x1da18, 0x1da19, 0x1da1a, 0x1da1b, 0x1da1c, 0x1da1d, 0x1da1e,
      0x1da1f, 0x1da20, 0x1da21, 0x1da22, 0x1da23, 0x1da24, 0x1da25, 0x1da26,
      0x1da27, 0x1da28, 0x1da29, 0x1da2a, 0x1da2b, 0x1da2c, 0x1da2d, 0x1da2e,
      0x1da2f, 0x1da30, 0x1da31, 0x1da32, 0x1da33, 0x1da34, 0x1da35, 0x1da36,
      0x1da3b, 0x1da3c, 0x1da3d, 0x1da3e, 0x1da3f, 0x1da40, 0x1da41, 0x1da42,
      0x1da43, 0x1da44, 0x1da45, 0x1da46, 0x1da47, 0x1da48, 0x1da49, 0x1da4a,
      0x1da4b, 0x1da4c, 0x1da4d, 0x1da4e, 0x1da4f, 0x1da50, 0x1da51, 0x1da52,
      0x1da53, 0x1da54, 0x1da55, 0x1da56, 0x1da57, 0x1da58, 0x1da59, 0x1da5a,
      0x1da5b, 0x1da5c, 0x1da5d, 0x1da5e, 0x1da5f, 0x1da60, 0x1da61, 0x1da62,
      0x1da63, 0x1da64, 0x1da65, 0x1da66, 0x1da67, 0x1da68, 0x1da69, 0x1da6a,
      0x1da6b, 0x1da6c, 0x1da75, 0x1da84, 0x1da9b, 0x1da9c, 0x1da9d, 0x1da9e,
      0x1da9f, 0x1daa1, 0x1daa2, 0x1daa3, 0x1daa4, 0x1daa5, 0x1daa6, 0x1daa7,
      0x1daa8, 0x1daa9, 0x1daaa, 0x1daab, 0x1daac, 0x1daad, 0x1daae, 0x1daaf,
      0x1e000, 0x1e001, 0x1e002, 0x1e003, 0x1e004, 0x1e005, 0x1e006, 0x1e008,
      0x1e009, 0x1e00a, 0x1e00b, 0x1e00c, 0x1e00d, 0x1e00e, 0x1e00f, 0x1e010,
      0x1e011, 0x1e012, 0x1e013, 0x1e014, 0x1e015, 0x1e016, 0x1e017, 0x1e018,
      0x1e01b, 0x1e01c, 0x1e01d, 0x1e01e, 0x1e01f, 0x1e020, 0x1e021, 0x1e023,
      0x1e024, 0x1e026, 0x1e027, 0x1e028, 0x1e029, 0x1e02a, 0x1e130, 0x1e131,
      0x1e132, 0x1e133, 0x1e134, 0x1e135, 0x1e136, 0x1e2ec, 0x1e2ed, 0x1e2ee,
      0x1e2ef, 0x1e8d0, 0x1e8d1, 0x1e8d2, 0x1e8d3, 0x1e8d4, 0x1e8d5, 0x1e8d6,
      0x1e944, 0x1e945, 0x1e946, 0x1e947, 0x1e948, 0x1e949, 0x1e94a, 0xe0100,
      0xe0101, 0xe0102, 0xe0103, 0xe0104, 0xe0105, 0xe0106, 0xe0107, 0xe0108,
      0xe0109, 0xe010a, 0xe010b, 0xe010c, 0xe010d, 0xe010e, 0xe010f, 0xe0110,
      0xe0111, 0xe0112, 0xe0113, 0xe0114, 0xe0115, 0xe0116, 0xe0117, 0xe0118,
      0xe0119, 0xe011a, 0xe011b, 0xe011c, 0xe011d, 0xe011e, 0xe011f, 0xe0120,
      0xe0121, 0xe0122, 0xe0123, 0xe0124, 0xe0125, 0xe0126, 0xe0127, 0xe0128,
      0xe0129, 0xe012a, 0xe012b, 0xe012c, 0xe012d, 0xe012e, 0xe012f, 0xe0130,
      0xe0131, 0xe0132, 0xe0133, 0xe0134, 0xe0135, 0xe0136, 0xe0137, 0xe0138,
      0xe0139, 0xe013a, 0xe013b, 0xe013c, 0xe013d, 0xe013e, 0xe013f, 0xe0140,
      0xe0141, 0xe0142, 0xe0143, 0xe0144, 0xe0145, 0xe0146, 0xe0147, 0xe0148,
      0xe0149, 0xe014a, 0xe014b, 0xe014c, 0xe014d, 0xe014e, 0xe014f, 0xe0150,
      0xe0151, 0xe0152, 0xe0153, 0xe0154, 0xe0155, 0xe0156, 0xe0157, 0xe0158,
      0xe0159, 0xe015a, 0xe015b, 0xe015c, 0xe015d, 0xe015e, 0xe015f, 0xe0160,
      0xe0161, 0xe0162, 0xe0163, 0xe0164, 0xe0165, 0xe0166, 0xe0167, 0xe0168,
      0xe0169, 0xe016a, 0xe016b, 0xe016c, 0xe016d, 0xe016e, 0xe016f, 0xe0170,
      0xe0171, 0xe0172, 0xe0173, 0xe0174, 0xe0175, 0xe0176, 0xe0177, 0xe0178,
      0xe0179, 0xe017a, 0xe017b, 0xe017c, 0xe017d, 0xe017e, 0xe017f, 0xe0180,
      0xe0181, 0xe0182, 0xe0183, 0xe0184, 0xe0185, 0xe0186, 0xe0187, 0xe0188,
      0xe0189, 0xe018a, 0xe018b, 0xe018c, 0xe018d, 0xe018e, 0xe018f, 0xe0190,
      0xe0191, 0xe0192, 0xe0193, 0xe0194, 0xe0195, 0xe0196, 0xe0197, 0xe0198,
      0xe0199, 0xe019a, 0xe019b, 0xe019c, 0xe019d, 0xe019e, 0xe019f, 0xe01a0,
      0xe01a1, 0xe01a2, 0xe01a3, 0xe01a4, 0xe01a5, 0xe01a6, 0xe01a7, 0xe01a8,
      0xe01a9, 0xe01aa, 0xe01ab, 0xe01ac, 0xe01ad, 0xe01ae, 0xe01af, 0xe01b0,
      0xe01b1, 0xe01b2, 0xe01b3, 0xe01b4, 0xe01b5, 0xe01b6, 0xe01b7, 0xe01b8,
      0xe01b9, 0xe01ba, 0xe01bb, 0xe01bc, 0xe01bd, 0xe01be, 0xe01bf, 0xe01c0,
      0xe01c1, 0xe01c2, 0xe01c3, 0xe01c4, 0xe01c5, 0xe01c6, 0xe01c7, 0xe01c8,
      0xe01c9, 0xe01ca, 0xe01cb, 0xe01cc, 0xe01cd, 0xe01ce, 0xe01cf, 0xe01d0,
      0xe01d1, 0xe01d2, 0xe01d3, 0xe01d4, 0xe01d5, 0xe01d6, 0xe01d7, 0xe01d8,
      0xe01d9, 0xe01da, 0xe01db, 0xe01dc, 0xe01dd, 0xe01de, 0xe01df, 0xe01e0,
      0xe01e1, 0xe01e2, 0xe01e3, 0xe01e4, 0xe01e5, 0xe01e6, 0xe01e7, 0xe01e8,
      0xe01e9, 0xe01ea, 0xe01eb, 0xe01ec, 0xe01ed, 0xe01ee, 0xe01ef};
  if (std::binary_search(std::begin(combining), std::end(combining),
                         label.front())) {
    return false;
  }
  // We verify this next step as part of the mapping:
  // ---------------------------------------------
  // Each code point in the label must only have certain status values
  // according to Section 5, IDNA Mapping Table:
  // - For Transitional Processing, each value must be valid.
  // - For Nontransitional Processing, each value must be either valid or
  // deviation.

  // If CheckJoiners, the label must satisfy the ContextJ rules from Appendix
  // A, in The Unicode Code Points and Internationalized Domain Names for
  // Applications (IDNA) [IDNA2008].
  constexpr static uint32_t virama[] = {
      0x094D,  0x09CD,  0x0A4D,  0x0ACD,  0x0B4D,  0x0BCD,  0x0C4D,  0x0CCD,
      0x0D3B,  0x0D3C,  0x0D4D,  0x0DCA,  0x0E3A,  0x0EBA,  0x0F84,  0x1039,
      0x103A,  0x1714,  0x1734,  0x17D2,  0x1A60,  0x1B44,  0x1BAA,  0x1BAB,
      0x1BF2,  0x1BF3,  0x2D7F,  0xA806,  0xA82C,  0xA8C4,  0xA953,  0xA9C0,
      0xAAF6,  0xABED,  0x10A3F, 0x11046, 0x1107F, 0x110B9, 0x11133, 0x11134,
      0x111C0, 0x11235, 0x112EA, 0x1134D, 0x11442, 0x114C2, 0x115BF, 0x1163F,
      0x116B6, 0x1172B, 0x11839, 0x1193D, 0x1193E, 0x119E0, 0x11A34, 0x11A47,
      0x11A99, 0x11C3F, 0x11D44, 0x11D45, 0x11D97};
  constexpr static uint32_t R[] = {
      0x622, 0x623, 0x624, 0x625, 0x627, 0x629, 0x62f, 0x630, 0x631,
      0x632, 0x648, 0x671, 0x672, 0x673, 0x675, 0x676, 0x677, 0x688,
      0x689, 0x68a, 0x68b, 0x68c, 0x68d, 0x68e, 0x68f, 0x690, 0x691,
      0x692, 0x693, 0x694, 0x695, 0x696, 0x697, 0x698, 0x699, 0x6c0,
      0x6c3, 0x6c4, 0x6c5, 0x6c6, 0x6c7, 0x6c8, 0x6c9, 0x6ca, 0x6cb,
      0x6cd, 0x6cf, 0x6d2, 0x6d3, 0x6d5, 0x6ee, 0x6ef, 0x710, 0x715,
      0x716, 0x717, 0x718, 0x719, 0x71e, 0x728, 0x72a, 0x72c, 0x72f,
      0x74d, 0x759, 0x75a, 0x75b, 0x854, 0x8aa, 0x8ab, 0x8ac};
  constexpr static uint32_t L[] = {0xa872};
  constexpr static uint32_t D[] = {
      0x620,  0x626,  0x628,  0x62a,  0x62b,  0x62c,  0x62d,  0x62e,  0x633,
      0x634,  0x635,  0x636,  0x637,  0x638,  0x639,  0x63a,  0x63b,  0x63c,
      0x63d,  0x63e,  0x63f,  0x641,  0x642,  0x643,  0x644,  0x645,  0x646,
      0x647,  0x649,  0x64a,  0x66e,  0x66f,  0x678,  0x679,  0x67a,  0x67b,
      0x67c,  0x67d,  0x67e,  0x67f,  0x680,  0x681,  0x682,  0x683,  0x684,
      0x685,  0x686,  0x687,  0x69a,  0x69b,  0x69c,  0x69d,  0x69e,  0x69f,
      0x6a0,  0x6a1,  0x6a2,  0x6a3,  0x6a4,  0x6a5,  0x6a6,  0x6a7,  0x6a8,
      0x6a9,  0x6aa,  0x6ab,  0x6ac,  0x6ad,  0x6ae,  0x6af,  0x6b0,  0x6b1,
      0x6b2,  0x6b3,  0x6b4,  0x6b5,  0x6b6,  0x6b7,  0x6b8,  0x6b9,  0x6ba,
      0x6bb,  0x6bc,  0x6bd,  0x6be,  0x6bf,  0x6c1,  0x6c2,  0x6cc,  0x6ce,
      0x6d0,  0x6d1,  0x6fa,  0x6fb,  0x6fc,  0x6ff,  0x712,  0x713,  0x714,
      0x71a,  0x71b,  0x71c,  0x71d,  0x71f,  0x720,  0x721,  0x722,  0x723,
      0x724,  0x725,  0x726,  0x727,  0x729,  0x72b,  0x72d,  0x72e,  0x74e,
      0x74f,  0x750,  0x751,  0x752,  0x753,  0x754,  0x755,  0x756,  0x757,
      0x758,  0x75c,  0x75d,  0x75e,  0x75f,  0x760,  0x761,  0x762,  0x763,
      0x764,  0x765,  0x766,  0x850,  0x851,  0x852,  0x853,  0x855,  0x8a0,
      0x8a2,  0x8a3,  0x8a4,  0x8a5,  0x8a6,  0x8a7,  0x8a8,  0x8a9,  0x1807,
      0x1820, 0x1821, 0x1822, 0x1823, 0x1824, 0x1825, 0x1826, 0x1827, 0x1828,
      0x1829, 0x182a, 0x182b, 0x182c, 0x182d, 0x182e, 0x182f, 0x1830, 0x1831,
      0x1832, 0x1833, 0x1834, 0x1835, 0x1836, 0x1837, 0x1838, 0x1839, 0x183a,
      0x183b, 0x183c, 0x183d, 0x183e, 0x183f, 0x1840, 0x1841, 0x1842, 0x1843,
      0x1844, 0x1845, 0x1846, 0x1847, 0x1848, 0x1849, 0x184a, 0x184b, 0x184c,
      0x184d, 0x184e, 0x184f, 0x1850, 0x1851, 0x1852, 0x1853, 0x1854, 0x1855,
      0x1856, 0x1857, 0x1858, 0x1859, 0x185a, 0x185b, 0x185c, 0x185d, 0x185e,
      0x185f, 0x1860, 0x1861, 0x1862, 0x1863, 0x1864, 0x1865, 0x1866, 0x1867,
      0x1868, 0x1869, 0x186a, 0x186b, 0x186c, 0x186d, 0x186e, 0x186f, 0x1870,
      0x1871, 0x1872, 0x1873, 0x1874, 0x1875, 0x1876, 0x1877, 0x1887, 0x1888,
      0x1889, 0x188a, 0x188b, 0x188c, 0x188d, 0x188e, 0x188f, 0x1890, 0x1891,
      0x1892, 0x1893, 0x1894, 0x1895, 0x1896, 0x1897, 0x1898, 0x1899, 0x189a,
      0x189b, 0x189c, 0x189d, 0x189e, 0x189f, 0x18a0, 0x18a1, 0x18a2, 0x18a3,
      0x18a4, 0x18a5, 0x18a6, 0x18a7, 0x18a8, 0x18aa, 0xa840, 0xa841, 0xa842,
      0xa843, 0xa844, 0xa845, 0xa846, 0xa847, 0xa848, 0xa849, 0xa84a, 0xa84b,
      0xa84c, 0xa84d, 0xa84e, 0xa84f, 0xa850, 0xa851, 0xa852, 0xa853, 0xa854,
      0xa855, 0xa856, 0xa857, 0xa858, 0xa859, 0xa85a, 0xa85b, 0xa85c, 0xa85d,
      0xa85e, 0xa85f, 0xa860, 0xa861, 0xa862, 0xa863, 0xa864, 0xa865, 0xa866,
      0xa867, 0xa868, 0xa869, 0xa86a, 0xa86b, 0xa86c, 0xa86d, 0xa86e, 0xa86f,
      0xa870, 0xa871};

  for (size_t i = 0; i < label.size(); i++) {
    uint32_t c = label[i];
    if (c == 0x200c) {
      if (i > 0) {
        if (std::binary_search(std::begin(virama), std::end(virama),
                               label[i - 1])) {
          return true;
        }
      }
      if ((i == 0) || (i + 1 >= label.size())) {
        return false;
      }
      // we go backward looking for L or D
      auto is_l_or_d = [](uint32_t code) {
        return std::binary_search(std::begin(L), std::end(L), code) ||
               std::binary_search(std::begin(D), std::end(D), code);
      };
      auto is_r_or_d = [](uint32_t code) {
        return std::binary_search(std::begin(R), std::end(R), code) ||
               std::binary_search(std::begin(D), std::end(D), code);
      };
      std::u32string_view before = label.substr(0, i);
      std::u32string_view after = label.substr(i + 1);
      return (std::find_if(before.begin(), before.end(), is_l_or_d) !=
              before.end()) &&
             (std::find_if(after.begin(), after.end(), is_r_or_d) !=
              after.end());
    } else if (c == 0x200d) {
      if (i > 0) {
        if (std::binary_search(std::begin(virama), std::end(virama),
                               label[i - 1])) {
          return true;
        }
      }
      return false;
    }
  }

  // If CheckBidi, and if the domain name is a  Bidi domain name, then the label
  // must satisfy all six of the numbered conditions in [IDNA2008] RFC 5893,
  // Section 2.

  // The following rule, consisting of six conditions, applies to labels
  // in Bidi domain names.  The requirements that this rule satisfies are
  // described in Section 3.  All of the conditions must be satisfied for
  // the rule to be satisfied.
  //
  //  1.  The first character must be a character with Bidi property L, R,
  //     or AL.  If it has the R or AL property, it is an RTL label; if it
  //     has the L property, it is an LTR label.
  //
  //  2.  In an RTL label, only characters with the Bidi properties R, AL,
  //      AN, EN, ES, CS, ET, ON, BN, or NSM are allowed.
  //
  //   3.  In an RTL label, the end of the label must be a character with
  //       Bidi property R, AL, EN, or AN, followed by zero or more
  //       characters with Bidi property NSM.
  //
  //   4.  In an RTL label, if an EN is present, no AN may be present, and
  //       vice versa.
  //
  //  5.  In an LTR label, only characters with the Bidi properties L, EN,
  //       ES, CS, ET, ON, BN, or NSM are allowed.
  //
  //   6.  In an LTR label, the end of the label must be a character with
  //       Bidi property L or EN, followed by zero or more characters with
  //       Bidi property NSM.

  size_t last_non_nsm_char = find_last_not_of_nsm(label);
  if (last_non_nsm_char == std::u32string_view::npos) {
    return false;
  }

  // A "Bidi domain name" is a domain name that contains at least one RTL label.
  // The following rule, consisting of six conditions, applies to labels in Bidi
  // domain names.
  if (is_rtl_label(label)) {
    // The first character must be a character with Bidi property L, R,
    // or AL. If it has the R or AL property, it is an RTL label; if it
    // has the L property, it is an LTR label.

    if (find_direction(label[0]) == direction::L) {
      // Eval as LTR

      // In an LTR label, only characters with the Bidi properties L, EN,
      // ES, CS, ET, ON, BN, or NSM are allowed.
      for (size_t i = 0; i < last_non_nsm_char; i++) {
        const direction d = find_direction(label[i]);
        if (!(d == direction::L || d == direction::EN || d == direction::ES ||
              d == direction::CS || d == direction::ET || d == direction::ON ||
              d == direction::BN || d == direction::NSM)) {
          return false;
        }

        if ((i == last_non_nsm_char) &&
            !(d == direction::L || d == direction::EN)) {
          return false;
        }
      }

      return true;

    } else {
      // Eval as RTL

      bool has_an = false;
      bool has_en = false;
      for (size_t i = 0; i <= last_non_nsm_char; i++) {
        const direction d = find_direction(label[i]);

        // In an RTL label, if an EN is present, no AN may be present, and vice
        // versa.
        if ((d == direction::EN && ((has_en = true) && has_an)) ||
            (d == direction::AN && ((has_an = true) && has_en))) {
          return false;
        }

        if (!(d == direction::R || d == direction::AL || d == direction::AN ||
              d == direction::EN || d == direction::ES || d == direction::CS ||
              d == direction::ET || d == direction::ON || d == direction::BN ||
              d == direction::NSM)) {
          return false;
        }

        if (i == last_non_nsm_char &&
            !(d == direction::R || d == direction::AL || d == direction::AN ||
              d == direction::EN)) {
          return false;
        }
      }

      return true;
    }
  }

  return true;
}

}  // namespace ada::idna
