// Copyright 2022 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_STORAGE_BUCKET_ENCRYPTION_H
#define GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_STORAGE_BUCKET_ENCRYPTION_H

#include "google/cloud/storage/version.h"
#include <utility>

namespace google {
namespace cloud {
namespace storage {
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_BEGIN

/**
 * Describes the default customer managed encryption key for a bucket.
 *
 * Customer managed encryption keys (CMEK) are encryption keys selected by the
 * user and generated by Google Cloud Key Management Service.
 *
 * @see https://cloud.google.com/storage/docs/encryption/customer-managed-keys
 *     for a general description of CMEK in Google Cloud Storage.
 *
 * @see https://cloud.google.com/kms/ for details about the Cloud Key Management
 *     Service.
 */
struct BucketEncryption {
  std::string default_kms_key_name;
};

inline bool operator==(BucketEncryption const& lhs,
                       BucketEncryption const& rhs) {
  return lhs.default_kms_key_name == rhs.default_kms_key_name;
}

inline bool operator<(BucketEncryption const& lhs,
                      BucketEncryption const& rhs) {
  return lhs.default_kms_key_name < rhs.default_kms_key_name;
}

inline bool operator!=(BucketEncryption const& lhs,
                       BucketEncryption const& rhs) {
  return std::rel_ops::operator!=(lhs, rhs);
}

inline bool operator>(BucketEncryption const& lhs,
                      BucketEncryption const& rhs) {
  return std::rel_ops::operator>(lhs, rhs);
}

inline bool operator<=(BucketEncryption const& lhs,
                       BucketEncryption const& rhs) {
  return std::rel_ops::operator<=(lhs, rhs);
}

inline bool operator>=(BucketEncryption const& lhs,
                       BucketEncryption const& rhs) {
  return std::rel_ops::operator>=(lhs, rhs);
}

GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_END
}  // namespace storage
}  // namespace cloud
}  // namespace google

#endif  // GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_STORAGE_BUCKET_ENCRYPTION_H
