#include <DataTypes/DataTypeArray.h>
#include <DataTypes/DataTypeDate.h>
#include <DataTypes/DataTypeDateTime.h>
#include <DataTypes/DataTypeLowCardinality.h>
#include <DataTypes/DataTypeString.h>
#include <DataTypes/DataTypesNumber.h>
#include <Interpreters/CrashLog.h>
#include <base/getFQDNOrHostName.h>
#include <Common/ClickHouseRevision.h>
#include <Common/DateLUTImpl.h>
#include <Common/Stopwatch.h>
#include <Common/SymbolIndex.h>

#include <Common/config_version.h>


namespace DB
{

std::weak_ptr<CrashLog> CrashLog::crash_log;


ColumnsDescription CrashLogElement::getColumnsDescription()
{
    return ColumnsDescription
    {
        {"hostname", std::make_shared<DataTypeLowCardinality>(std::make_shared<DataTypeString>()), "The hostname where the crash occurred."},
        {"event_date", std::make_shared<DataTypeDate>(), "The date of the crash."},
        {"event_time", std::make_shared<DataTypeDateTime>(), "The time of the crash."},
        {"timestamp_ns", std::make_shared<DataTypeUInt64>(), "Timestamp of the event with nanoseconds."},
        {"signal", std::make_shared<DataTypeInt32>(), "Signal number."},
        {"thread_id", std::make_shared<DataTypeUInt64>(), "Thread ID."},
        {"query_id", std::make_shared<DataTypeString>(), "Query ID."},
        {"trace", std::make_shared<DataTypeArray>(std::make_shared<DataTypeUInt64>()), "Stack trace at the moment of crash. Each element is a virtual memory address inside ClickHouse server process."},
        {"trace_full", std::make_shared<DataTypeArray>(std::make_shared<DataTypeString>()), "Stack trace at the moment of crash. Each element contains a called method inside ClickHouse server process."},
        {"version", std::make_shared<DataTypeString>(), "ClickHouse server version."},
        {"revision", std::make_shared<DataTypeUInt32>(), "ClickHouse server revision."},
        {"build_id", std::make_shared<DataTypeString>(), "BuildID that is generated by compiler."},
    };
}

void CrashLogElement::appendToBlock(MutableColumns & columns) const
{
    size_t i = 0;

    columns[i++]->insert(getFQDNOrHostName());
    columns[i++]->insert(DateLUT::instance().toDayNum(event_time).toUnderType());
    columns[i++]->insert(event_time);
    columns[i++]->insert(timestamp_ns);
    columns[i++]->insert(signal);
    columns[i++]->insert(thread_id);
    columns[i++]->insertData(query_id.data(), query_id.size());
    columns[i++]->insert(trace);
    columns[i++]->insert(trace_full);
    columns[i++]->insert(VERSION_FULL);
    columns[i++]->insert(ClickHouseRevision::getVersionRevision());

    String build_id_hex;
#if defined(__ELF__) && !defined(OS_FREEBSD)
    build_id_hex = SymbolIndex::instance().getBuildIDHex();
#endif
    columns[i++]->insert(build_id_hex);
}

}

void collectCrashLog(Int32 signal, UInt64 thread_id, const String & query_id, const StackTrace & stack_trace)
{
    using namespace DB;

    if (auto crash_log_owned = CrashLog::crash_log.lock())
    {
        UInt64 time = clock_gettime_ns(CLOCK_REALTIME);

        size_t stack_trace_size = stack_trace.getSize();
        size_t stack_trace_offset = stack_trace.getOffset();
        size_t num_frames = stack_trace_size - stack_trace_offset;

        Array trace;
        Array trace_full;

        trace.reserve(num_frames);
        trace_full.reserve(num_frames);

        for (size_t i = stack_trace_offset; i < stack_trace_size; ++i)
            trace.push_back(reinterpret_cast<uintptr_t>(stack_trace.getFramePointers()[i]));

        stack_trace.toStringEveryLine([&trace_full](std::string_view line) { trace_full.push_back(line); });

        CrashLogElement element{static_cast<time_t>(time / 1000000000), time, signal, thread_id, query_id, trace, trace_full};
        crash_log_owned->add(std::move(element));
    }
}
