#include <Columns/ColumnVector.h>
#include <Core/callOnTypeIndex.h>
#include <DataTypes/IDataType.h>
#include <DataTypes/DataTypesNumber.h>
#include <DataTypes/DataTypeDate.h>
#include <DataTypes/DataTypeDateTime.h>
#include <DataTypes/DataTypeDateTime64.h>
#include <DataTypes/DataTypeNullable.h>
#include <Formats/FormatSettings.h>
#include <Functions/FunctionFactory.h>
#include <Functions/IFunction.h>
#include <IO/WriteBufferFromString.h>
#include <base/defines.h>
#include <set>

namespace DB
{
    namespace ErrorCodes
    {
        extern const int ILLEGAL_COLUMN;
        extern const int ILLEGAL_TYPE_OF_ARGUMENT;
        extern const int INCORRECT_DATA;
    }

    template <typename Name, typename ArgDataType, typename ConcurrencyDataType>
    class ExecutableFunctionRunningConcurrency : public IExecutableFunction
    {
    public:
        String getName() const override
        {
            return Name::name;
        }

        ColumnPtr executeImpl(const ColumnsWithTypeAndName & arguments, const DataTypePtr &, size_t input_rows_count) const override
        {
            using ColVecArg = typename ArgDataType::ColumnType;
            const ColVecArg * col_begin = checkAndGetColumn<ColVecArg>(arguments[0].column.get());
            const ColVecArg * col_end   = checkAndGetColumn<ColVecArg>(arguments[1].column.get());
            if (!col_begin || !col_end)
                throw Exception(ErrorCodes::ILLEGAL_COLUMN, "Constant columns are not supported at the moment");
            const typename ColVecArg::Container & vec_begin = col_begin->getData();
            const typename ColVecArg::Container & vec_end   = col_end->getData();

            using ColVecConc = typename ConcurrencyDataType::ColumnType;
            using FieldType = typename ConcurrencyDataType::FieldType;
            typename ColVecConc::MutablePtr col_concurrency = ColVecConc::create(input_rows_count);
            typename ColVecConc::Container & vec_concurrency = col_concurrency->getData();

            std::multiset<typename ArgDataType::FieldType> ongoing_until;
            auto begin_serializaion = arguments[0].type->getDefaultSerialization();
            auto end_serialization = arguments[1].type->getDefaultSerialization();
            for (size_t i = 0; i < input_rows_count; ++i)
            {
                const auto begin = vec_begin[i];
                const auto end   = vec_end[i];

                if (unlikely(begin > end))
                {
                    const FormatSettings default_format{};
                    WriteBufferFromOwnString buf_begin;
                    WriteBufferFromOwnString buf_end;
                    begin_serializaion->serializeTextQuoted(*(arguments[0].column), i, buf_begin, default_format);
                    end_serialization->serializeTextQuoted(*(arguments[1].column), i, buf_end, default_format);
                    throw Exception(ErrorCodes::INCORRECT_DATA, "Incorrect order of events: {} > {}", buf_begin.str(), buf_end.str());
                }

                ongoing_until.insert(end);

                // Erase all the elements from "ongoing_until" which
                // are less than or equal to "begin", i.e. durations
                // that have already ended. We consider "begin" to be
                // inclusive, and "end" to be exclusive.
                ongoing_until.erase(
                    ongoing_until.begin(), ongoing_until.upper_bound(begin));

                vec_concurrency[i] = static_cast<FieldType>(ongoing_until.size());
            }

            return col_concurrency;
        }

        bool useDefaultImplementationForConstants() const override
        {
            return true;
        }
    };

    template <typename Name, typename ArgDataType, typename ConcurrencyDataType>
    class FunctionBaseRunningConcurrency : public IFunctionBase
    {
    public:
        explicit FunctionBaseRunningConcurrency(DataTypes argument_types_, DataTypePtr return_type_)
            : argument_types(std::move(argument_types_))
            , return_type(std::move(return_type_)) {}

        String getName() const override
        {
            return Name::name;
        }

        const DataTypes & getArgumentTypes() const override
        {
            return argument_types;
        }

        const DataTypePtr & getResultType() const override
        {
            return return_type;
        }

        ExecutableFunctionPtr prepare(const ColumnsWithTypeAndName &) const override
        {
            return std::make_unique<ExecutableFunctionRunningConcurrency<Name, ArgDataType, ConcurrencyDataType>>();
        }

        bool isStateful() const override
        {
            return true;
        }

        bool isSuitableForShortCircuitArgumentsExecution(const DataTypesWithConstInfo & /*arguments*/) const override { return true; }

    private:
        DataTypes argument_types;
        DataTypePtr return_type;
    };

    template <typename Name, typename ConcurrencyDataType>
    class RunningConcurrencyOverloadResolver : public IFunctionOverloadResolver
    {
        template <typename T>
        struct TypeTag
        {
            using Type = T;
        };

        /// Call a polymorphic lambda with a type tag of src_type.
        template <typename F>
        void dispatchForSourceType(const IDataType & src_type, F && f) const
        {
            WhichDataType which(src_type);

            switch (which.idx)
            {
            case TypeIndex::Date:       f(TypeTag<DataTypeDate>());       break;
            case TypeIndex::DateTime:   f(TypeTag<DataTypeDateTime>());   break;
            case TypeIndex::DateTime64: f(TypeTag<DataTypeDateTime64>()); break;
            default:
                throw Exception(ErrorCodes::ILLEGAL_TYPE_OF_ARGUMENT, "Arguments for function {} must be Date, DateTime, or DateTime64.", getName());
            }
        }

    public:
        static constexpr auto name = Name::name;

        static FunctionOverloadResolverPtr create(ContextPtr)
        {
            return std::make_unique<RunningConcurrencyOverloadResolver<Name, ConcurrencyDataType>>();
        }

        String getName() const override
        {
            return Name::name;
        }

        FunctionBasePtr buildImpl(const ColumnsWithTypeAndName & arguments, const DataTypePtr & return_type) const override
        {
            // The type of the second argument must match with that of the first one.
            if (unlikely(!arguments[1].type->equals(*(arguments[0].type))))
            {
                throw Exception(ErrorCodes::ILLEGAL_TYPE_OF_ARGUMENT, "Function {} must be called with two arguments having the same type.", getName());
            }

            DataTypes argument_types = { arguments[0].type, arguments[1].type };
            FunctionBasePtr base;
            dispatchForSourceType(*(arguments[0].type), [&](auto arg_type_tag) // Throws when the type is inappropriate.
            {
                using Tag = decltype(arg_type_tag);
                using ArgDataType = typename Tag::Type;

                base = std::make_unique<FunctionBaseRunningConcurrency<Name, ArgDataType, ConcurrencyDataType>>(argument_types, return_type);
            });

            return base;
        }

        DataTypePtr getReturnTypeImpl(const DataTypes &) const override
        {
            return std::make_shared<ConcurrencyDataType>();
        }

        size_t getNumberOfArguments() const override
        {
            return 2;
        }

        bool isInjective(const ColumnsWithTypeAndName &) const override
        {
            return false;
        }

        bool isStateful() const override
        {
            return true;
        }

        bool useDefaultImplementationForNulls() const override
        {
            return false;
        }
    };

    struct NameRunningConcurrency
    {
        static constexpr auto name = "runningConcurrency";
    };

    REGISTER_FUNCTION(RunningConcurrency)
    {
        FunctionDocumentation::Description description_runningConcurrency = R"(
Calculates the number of concurrent events.
Each event has a start time and an end time.
The start time is included in the event, while the end time is excluded.
Columns with a start time and an end time must be of the same data type.
The function calculates the total number of active (concurrent) events for each event start time.

:::tip Requirements
Events must be ordered by the start time in ascending order.
If this requirement is violated the function raises an exception.
Every data block is processed separately.
If events from different data blocks overlap then they can not be processed correctly.
:::

:::warning Deprecated
It is advised to use [window functions](/sql-reference/window-functions) instead.
:::
)";
        FunctionDocumentation::Syntax syntax_runningConcurrency = "runningConcurrency(start, end)";
        FunctionDocumentation::Arguments arguments_runningConcurrency = {
            {"start", "A column with the start time of events.", {"Date", "DateTime", "DateTime64"}},
            {"end", "A column with the end time of events.", {"Date", "DateTime", "DateTime64"}}
        };
        FunctionDocumentation::ReturnedValue returned_value_runningConcurrency = {"Returns the number of concurrent events at each event start time.", {"UInt32"}};
        FunctionDocumentation::Examples examples_runningConcurrency = {
        {
            "Usage example",
            R"(
SELECT start, runningConcurrency(start, end) FROM example_table;
            )",
            R"(
┌──────start─┬─runningConcurrency(start, end)─┐
│ 2025-03-03 │                              1 │
│ 2025-03-06 │                              2 │
│ 2025-03-07 │                              3 │
│ 2025-03-11 │                              2 │
└────────────┴────────────────────────────────┘
            )"
        }
        };
        FunctionDocumentation::IntroducedIn introduced_in_runningConcurrency = {21, 3};
        FunctionDocumentation::Category category_runningConcurrency = FunctionDocumentation::Category::Other;
        FunctionDocumentation documentation_runningConcurrency = {description_runningConcurrency, syntax_runningConcurrency, arguments_runningConcurrency, returned_value_runningConcurrency, examples_runningConcurrency, introduced_in_runningConcurrency, category_runningConcurrency};

        factory.registerFunction<RunningConcurrencyOverloadResolver<NameRunningConcurrency, DataTypeUInt32>>(documentation_runningConcurrency);
    }
}
