#!/usr/bin/env python3
from pathlib import Path
from typing import List

VERSIONS_FILE = (
    Path(__file__).absolute().parent.parent / "list-versions" / "version_date.tsv"
)

HEADER = """<!--
the file is autogenerated by utils/security-generator/generate_security.py
-->

# ClickHouse Security Vulnerability Response Policy

## Security Change Log and Support

Details regarding security fixes are publicly reported in our [security changelog](https://clickhouse.com/docs/whats-new/security-changelog/). A summary of known security vulnerabilities is shown at the bottom of this page.

Vulnerability notifications pre-release or during embargo periods are available to open source users and support customers registered for vulnerability alerts. Refer to our [Embargo Policy](#embargo-policy) below.

The following versions of ClickHouse server are currently supported with security updates:
"""

FOOTER = """## Reporting a Vulnerability

We're extremely grateful for security researchers and users that report vulnerabilities to the ClickHouse Open Source Community. All reports are thoroughly investigated by developers.

To report a potential vulnerability in ClickHouse please send the details about it through our public bug bounty program hosted by [Bugcrowd](https://bugcrowd.com/clickhouse) and be rewarded for it as per the program scope and rules of engagement.

### When Should I Report a Vulnerability?

- You think you discovered a potential security vulnerability in ClickHouse
- You are unsure how a vulnerability affects ClickHouse

### When Should I NOT Report a Vulnerability?

- You need help tuning ClickHouse components for security
- You need help applying security related updates
- Your issue is not security related

## Security Vulnerability Response

Each report is acknowledged and analyzed by ClickHouse maintainers within 5 working days.
As the security issue moves from triage, to identified fix, to release planning we will keep the reporter updated.

## Public Disclosure Timing

A public disclosure date is negotiated by the ClickHouse maintainers and the bug submitter. We prefer to fully disclose the bug as soon as possible once a user mitigation is available. It is reasonable to delay disclosure when the bug or the fix is not yet fully understood, the solution is not well-tested, or for vendor coordination. The timeframe for disclosure is from immediate (especially if it's already publicly known) to 90 days. For a vulnerability with a straightforward mitigation, we expect the report date to disclosure date to be on the order of 7 days.

## Embargo Policy

Open source users and support customers may subscribe to receive alerts during the embargo period by visiting [https://trust.clickhouse.com/?product=clickhouseoss](https://trust.clickhouse.com/?product=clickhouseoss), requesting access and subscribing for alerts. Subscribers agree not to make these notifications public, issue communications, share this information with others, or issue public patches before the disclosure date. Accidental disclosures must be reported immediately to trust@clickhouse.com. Failure to follow this policy or repeated leaks may result in removal from the subscriber list.

Participation criteria:
1. Be a current open source user or support customer with a valid corporate email domain (no @gmail.com, @azure.com, etc.).
1. Sign up to the ClickHouse OSS Trust Center at [https://trust.clickhouse.com](https://trust.clickhouse.com).
1. Accept the ClickHouse Security Vulnerability Response Policy as outlined above.
1. Subscribe to ClickHouse OSS Trust Center alerts.

Removal criteria:
1. Members may be removed for failure to follow this policy or repeated leaks.
1. Members may be removed for bounced messages (mail delivery failure).
1. Members may unsubscribe at any time.

Notification process:
ClickHouse will post notifications within our OSS Trust Center and notify subscribers. Subscribers must log in to the Trust Center to download the notification. The notification will include the timeframe for public disclosure.
"""


def generate_supported_versions() -> str:
    with open(VERSIONS_FILE, "r", encoding="utf-8") as fd:
        versions = [line.split(maxsplit=1)[0][1:] for line in fd.readlines()]

    supported_year = 0  # set automatically when all supported versions are filled
    # 3 regular versions
    regular = []  # type: List[str]
    max_regular = 3
    # 2 LTS versions, one of them could be in regular
    lts = []  # type: List[str]
    max_lts = 2
    # The rest are unsupported
    unsupported = []  # type: List[str]
    table = [
        "| Version | Supported |",
        "|:-|:-|",
    ]
    for version in versions:
        year = int(version.split(".")[0])
        month = int(version.split(".")[1])
        version = f"{year}.{month}"
        to_append = ""
        if version in regular or version in lts:
            continue
        if len(regular) < max_regular:
            regular.append(version)
            to_append = f"| {version} | ✔️ |"
        if len(lts) < max_lts and month in [3, 8]:
            lts.append(version)
            to_append = f"| {version} | ✔️ |"
        if to_append:
            if len(regular) == max_regular and len(lts) == max_lts:
                supported_year = year
            table.append(to_append)
            continue
        if year < supported_year:
            # The whole year is unsupported
            version = f"{year}.*"
        if not version in unsupported:
            unsupported.append(version)
            table.append(f"| {version} | ❌ |")

    return "\n".join(table) + "\n"


def main() -> None:
    print(HEADER)
    print(generate_supported_versions())
    print(FOOTER)


if __name__ == "__main__":
    main()
