#include <Columns/ColumnFixedString.h>
#include <DataTypes/DataTypeFixedString.h>
#include <Functions/FunctionFactory.h>
#include <Functions/FunctionHelpers.h>
#include <Functions/IFunction.h>
#include <Functions/PerformanceAdaptors.h>
#include <Functions/FunctionsRandom.h>
#include <pcg_random.hpp>
#include <Common/randomSeed.h>
#include <base/arithmeticOverflow.h>
#include <base/unaligned.h>

#include <base/defines.h>

namespace DB
{
namespace ErrorCodes
{
    extern const int ILLEGAL_TYPE_OF_ARGUMENT;
    extern const int ILLEGAL_COLUMN;
    extern const int DECIMAL_OVERFLOW;
}

namespace
{

/* Generate random fixed string with fully random bytes (including zero). */
template <typename RandImpl>
class FunctionRandomFixedStringImpl : public IFunction
{
public:
    static constexpr auto name = "randomFixedString";

    String getName() const override { return name; }

    bool isVariadic() const override { return false; }

    bool isSuitableForShortCircuitArgumentsExecution(const DataTypesWithConstInfo & /*arguments*/) const override { return false; }

    size_t getNumberOfArguments() const override { return 1; }

    DataTypePtr getReturnTypeImpl(const ColumnsWithTypeAndName & arguments) const override
    {
        if (!isUInt(arguments[0].type))
            throw Exception(ErrorCodes::ILLEGAL_TYPE_OF_ARGUMENT, "First argument for function {} must be unsigned integer", getName());

        if (!arguments[0].column || !isColumnConst(*arguments[0].column))
            throw Exception(ErrorCodes::ILLEGAL_COLUMN, "First argument for function {} must be constant", getName());

        const size_t n = assert_cast<const ColumnConst &>(*arguments[0].column).getValue<UInt64>();
        return std::make_shared<DataTypeFixedString>(n);
    }

    bool isDeterministic() const override { return false; }
    bool isDeterministicInScopeOfQuery() const override { return false; }

    ColumnPtr executeImpl(const ColumnsWithTypeAndName & arguments, const DataTypePtr &, size_t input_rows_count) const override
    {
        const size_t n = assert_cast<const ColumnConst &>(*arguments[0].column).getValue<UInt64>();

        auto col_to = ColumnFixedString::create(n);
        ColumnFixedString::Chars & data_to = col_to->getChars();

        if (input_rows_count == 0)
            return col_to;

        size_t total_size;
        if (common::mulOverflow(input_rows_count, n, total_size))
            throw Exception(ErrorCodes::DECIMAL_OVERFLOW, "Decimal math overflow");

        /// Fill random bytes.
        data_to.resize(total_size);
        RandImpl::execute(reinterpret_cast<char *>(data_to.data()), total_size);

        return col_to;
    }
};

class FunctionRandomFixedString : public FunctionRandomFixedStringImpl<TargetSpecific::Default::RandImpl>
{
public:
    explicit FunctionRandomFixedString(ContextPtr context) : selector(context)
    {
        selector.registerImplementation<TargetArch::Default,
            FunctionRandomFixedStringImpl<TargetSpecific::Default::RandImpl>>();

    #if USE_MULTITARGET_CODE
        selector.registerImplementation<TargetArch::AVX2,
            FunctionRandomFixedStringImpl<TargetSpecific::AVX2::RandImpl>>();
    #endif
    }

    ColumnPtr executeImpl(const ColumnsWithTypeAndName & arguments, const DataTypePtr & result_type, size_t input_rows_count) const override
    {
        return selector.selectAndExecute(arguments, result_type, input_rows_count);
    }

    static FunctionPtr create(ContextPtr context)
    {
        return std::make_shared<FunctionRandomFixedString>(context);
    }

private:
    ImplementationSelector<IFunction> selector;
};

}

REGISTER_FUNCTION(RandomFixedString)
{
    FunctionDocumentation::Description description = R"(
Generates a random fixed-size string with the specified number of character.
The returned characters are not necessarily ASCII characters, i.e. they may not be printable.
    )";
    FunctionDocumentation::Syntax syntax = "randomFixedString(length)";
    FunctionDocumentation::Arguments arguments = {
        {"length", "Length of the string in bytes.", {"UInt*"}}
    };
    FunctionDocumentation::ReturnedValue returned_value = {"Returns a string filled with random bytes.", {"FixedString"}};
    FunctionDocumentation::Examples examples = {
        {"Usage example", "SELECT randomFixedString(13) AS rnd, toTypeName(rnd)", R"(
┌─rnd──────┬─toTypeName(randomFixedString(13))─┐
│ j▒h㋖HɨZ'▒ │ FixedString(13)                 │
└──────────┴───────────────────────────────────┘
        )"}
    };
    FunctionDocumentation::IntroducedIn introduced_in = {20, 5};
    FunctionDocumentation::Category category = FunctionDocumentation::Category::RandomNumber;
    FunctionDocumentation documentation = {description, syntax, arguments, returned_value, examples, introduced_in, category};

    factory.registerFunction<FunctionRandomFixedString>(documentation);
}

}
