#include "config.h"

#if USE_H3

#include <Columns/ColumnArray.h>
#include <Columns/ColumnsNumber.h>
#include <DataTypes/DataTypeArray.h>
#include <DataTypes/DataTypesNumber.h>
#include <Functions/FunctionFactory.h>
#include <Functions/IFunction.h>
#include <Common/typeid_cast.h>
#include <IO/WriteHelpers.h>
#include <base/range.h>
#include <constants.h>
#include <h3api.h>


namespace DB
{
namespace ErrorCodes
{
    extern const int ILLEGAL_TYPE_OF_ARGUMENT;
    extern const int ILLEGAL_COLUMN;
}

namespace
{

class FunctionH3GetDestinationIndexFromUnidirectionalEdge : public IFunction
{
public:
    static constexpr auto name = "h3GetDestinationIndexFromUnidirectionalEdge";

    static FunctionPtr create(ContextPtr) { return std::make_shared<FunctionH3GetDestinationIndexFromUnidirectionalEdge>(); }

    std::string getName() const override { return name; }

    size_t getNumberOfArguments() const override { return 1; }
    bool useDefaultImplementationForConstants() const override { return true; }
    bool isSuitableForShortCircuitArgumentsExecution(const DataTypesWithConstInfo & /*arguments*/) const override { return true; }

    DataTypePtr getReturnTypeImpl(const DataTypes & arguments) const override
    {
        const auto * arg = arguments[0].get();
        if (!WhichDataType(arg).isUInt64())
            throw Exception(
                ErrorCodes::ILLEGAL_TYPE_OF_ARGUMENT,
                "Illegal type {} of argument {} of function {}. Must be UInt64",
                arg->getName(), 1, getName());

        return std::make_shared<DataTypeUInt64>();
    }

    DataTypePtr getReturnTypeForDefaultImplementationForDynamic() const override
    {
        return std::make_shared<DataTypeUInt64>();
    }

    ColumnPtr executeImpl(const ColumnsWithTypeAndName & arguments, const DataTypePtr &, size_t input_rows_count) const override
    {
        auto non_const_arguments = arguments;
        for (auto & argument : non_const_arguments)
            argument.column = argument.column->convertToFullColumnIfConst();

        const auto * col_hindex_edge = checkAndGetColumn<ColumnUInt64>(non_const_arguments[0].column.get());
        if (!col_hindex_edge)
            throw Exception(
                ErrorCodes::ILLEGAL_COLUMN,
                "Illegal type {} of argument {} of function {}. Must be UInt64.",
                arguments[0].type->getName(),
                1,
                getName());

        const auto & data_hindex_edge = col_hindex_edge->getData();

        auto dst = ColumnVector<UInt64>::create();
        auto & dst_data = dst->getData();
        dst_data.resize(input_rows_count);

        for (size_t row = 0; row < input_rows_count; ++row)
        {
            const UInt64 edge = data_hindex_edge[row];
            const UInt64 res = getDirectedEdgeDestination(edge);
            dst_data[row] = res;
        }

        return dst;
    }
};

}

REGISTER_FUNCTION(H3GetDestinationIndexFromUnidirectionalEdge)
{
    FunctionDocumentation::Description description = R"(
Returns the destination hexagon index from the unidirectional edge [H3](#h3-index).
    )";
    FunctionDocumentation::Syntax syntax = "h3GetDestinationIndexFromUnidirectionalEdge(edge)";
    FunctionDocumentation::Arguments arguments = {
        {"edge", "Hexagon index number that represents a unidirectional edge.", {"UInt64"}}
    };
    FunctionDocumentation::ReturnedValue returned_value = {
        "Returns the destination hexagon index from the unidirectional edge.",
        {"UInt64"}
    };
    FunctionDocumentation::Examples examples = {
        {
            "Get destination index from a unidirectional edge",
            "SELECT h3GetDestinationIndexFromUnidirectionalEdge(1248204388774707197) AS destination",
            R"(
┌────────destination─┐
│ 599686043507097597 │
└────────────────────┘
            )"
        }
    };
    FunctionDocumentation::IntroducedIn introduced_in = {22, 6};
    FunctionDocumentation::Category category = FunctionDocumentation::Category::Geo;
    FunctionDocumentation documentation = {description, syntax, arguments, returned_value, examples, introduced_in, category};
    factory.registerFunction<FunctionH3GetDestinationIndexFromUnidirectionalEdge>(documentation);
}

}

#endif
