---
description: 'Documentation for H3'
sidebar_label: 'H3 Indexes'
slug: /sql-reference/functions/geo/h3
title: 'Functions for Working with H3 Indexes'
doc_type: 'reference'
---

## H3 Index {#h3-index}

[H3](https://h3geo.org/) is a geographical indexing system where the Earth's surface is divided into a grid of even hexagonal cells. This system is hierarchical, i. e. each hexagon on the top level ("parent") can be split into seven even but smaller ones ("children"), and so on.

The level of the hierarchy is called `resolution` and can receive a value from `0` till `15`, where `0` is the `base` level with the largest and coarsest cells.

A latitude and longitude pair can be transformed to a 64-bit H3 index, identifying a grid cell.

The H3 index is used primarily for bucketing locations and other geospatial manipulations.

The full description of the H3 system is available at [the Uber Engineering site](https://www.uber.com/blog/h3/).

## h3IsValid {#h3isvalid}

Verifies whether the number is a valid [H3](#h3-index) index.

**Syntax**

```sql
h3IsValid(h3index)
```

**Parameter**

- `h3index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned values**

- 1 — The number is a valid H3 index. [UInt8](../../data-types/int-uint.md).
- 0 — The number is not a valid H3 index. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3IsValid(630814730351855103) AS h3IsValid;
```

Result:

```text
┌─h3IsValid─┐
│         1 │
└───────────┘
```

## h3GetResolution {#h3getresolution}

Defines the resolution of the given [H3](#h3-index) index.

**Syntax**

```sql
h3GetResolution(h3index)
```

**Parameter**

- `h3index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned values**

- Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).
- If the index is not valid, the function returns a random value. Use [h3IsValid](#h3isvalid) to verify the index. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3GetResolution(639821929606596015) AS resolution;
```

Result:

```text
┌─resolution─┐
│         14 │
└────────────┘
```

## h3EdgeAngle {#h3edgeangle}

Calculates the average length of an [H3](#h3-index) hexagon edge in grades.

**Syntax**

```sql
h3EdgeAngle(resolution)
```

**Parameter**

- `resolution` — Index resolution. [UInt8](../../data-types/int-uint.md). Range: `[0, 15]`.

**Returned values**

- The average length of an [H3](#h3-index) hexagon edge in grades. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3EdgeAngle(10) AS edgeAngle;
```

Result:

```text
┌───────h3EdgeAngle(10)─┐
│ 0.0005927224846720883 │
└───────────────────────┘
```

## h3EdgeLengthM {#h3edgelengthm}

Calculates the average length of an [H3](#h3-index) hexagon edge in meters.

**Syntax**

```sql
h3EdgeLengthM(resolution)
```

**Parameter**

- `resolution` — Index resolution. [UInt8](../../data-types/int-uint.md). Range: `[0, 15]`.

**Returned values**

- The average edge length of an [H3](#h3-index) hexagon in meters. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3EdgeLengthM(15) AS edgeLengthM;
```

Result:

```text
┌─edgeLengthM─┐
│ 0.509713273 │
└─────────────┘
```

## h3EdgeLengthKm {#h3edgelengthkm}

Calculates the average length of an [H3](#h3-index) hexagon edge in kilometers.

**Syntax**

```sql
h3EdgeLengthKm(resolution)
```

**Parameter**

- `resolution` — Index resolution. [UInt8](../../data-types/int-uint.md). Range: `[0, 15]`.

**Returned values**

- The average length of an [H3](#h3-index) hexagon edge in kilometers. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3EdgeLengthKm(15) AS edgeLengthKm;
```

Result:

```text
┌─edgeLengthKm─┐
│  0.000509713 │
└──────────────┘
```

## geoToH3 {#geotoh3}

Returns [H3](#h3-index) point index `(lat, lon)` with specified resolution.

**Syntax**

```sql
geoToH3(lat, lon, resolution)
```

**Arguments**

- `lat` — Latitude. [Float64](../../data-types/float.md).
- `lon` — Longitude. [Float64](../../data-types/float.md).
- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned values**

- Hexagon index number. [UInt64](../../data-types/int-uint.md).
- 0 in case of error. [UInt64](../../data-types/int-uint.md).

Note: In ClickHouse v25.4 or older, `geoToH3()` takes values in order `(lon, lat)`. As per ClickHouse v25.5, the input values are in order `(lat, lon)`. The previous behaviour can be restored using setting `geotoh3_argument_order = 'lon_lat'`.

**Example**

Query:

```sql
SELECT geoToH3(55.71290588, 37.79506683, 15) AS h3Index;
```

Result:

```text
┌────────────h3Index─┐
│ 644325524701193974 │
└────────────────────┘
```

## h3ToGeo {#h3togeo}

Returns the centroid latitude and longitude corresponding to the provided [H3](#h3-index) index.

**Syntax**

```sql
h3ToGeo(h3Index)
```

**Arguments**

- `h3Index` — H3 Index. [UInt64](../../data-types/int-uint.md).

**Returned values**

- A tuple consisting of two values: `tuple(lat,lon)`. `lat` — Latitude. [Float64](../../data-types/float.md). `lon` — Longitude. [Float64](../../data-types/float.md).

Note: In ClickHouse v24.12 or older, `h3ToGeo()` returns values in order `(lon, lat)`. As per ClickHouse v25.1, the returned values are in order `(lat, lon)`. The previous behaviour can be restored using setting `h3togeo_lon_lat_result_order = true`.

**Example**

Query:

```sql
SELECT h3ToGeo(644325524701193974) AS coordinates;
```

Result:

```text
┌─coordinates───────────────────────────┐
│ (55.71290243145668,37.79506616830252) │
└───────────────────────────────────────┘
```

## h3ToGeoBoundary {#h3togeoboundary}

Returns array of pairs `(lat, lon)`, which corresponds to the boundary of the provided H3 index.

**Syntax**

```sql
h3ToGeoBoundary(h3Index)
```

**Arguments**

- `h3Index` — H3 Index. [UInt64](../../data-types/int-uint.md).

**Returned values**

- Array of pairs '(lat, lon)'. [Array](../../data-types/array.md)([Float64](../../data-types/float.md), [Float64](../../data-types/float.md)).

**Example**

Query:

```sql
SELECT h3ToGeoBoundary(644325524701193974) AS coordinates;
```

Result:

```text
┌─h3ToGeoBoundary(599686042433355775)────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [(37.2713558667319,-121.91508032705622),(37.353926450852256,-121.8622232890249),(37.42834118609435,-121.92354999630156),(37.42012867767779,-122.03773496427027),(37.33755608435299,-122.090428929044),(37.26319797461824,-122.02910130919001)] │
└────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3kRing {#h3kring}

 Lists all the [H3](#h3-index) hexagons in the raduis of `k` from the given hexagon in random order.

**Syntax**

```sql
h3kRing(h3index, k)
```

**Arguments**

- `h3index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).
- `k` — Radius. [integer](../../data-types/int-uint.md)

**Returned values**

- Array of H3 indexes. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
SELECT arrayJoin(h3kRing(644325529233966508, 1)) AS h3index;
```

Result:

```text
┌────────────h3index─┐
│ 644325529233966508 │
│ 644325529233966497 │
│ 644325529233966510 │
│ 644325529233966504 │
│ 644325529233966509 │
│ 644325529233966355 │
│ 644325529233966354 │
└────────────────────┘
```

## h3GetBaseCell {#h3getbasecell}

Returns the base cell number of the [H3](#h3-index) index.

**Syntax**

```sql
h3GetBaseCell(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Hexagon base cell number. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3GetBaseCell(612916788725809151) AS basecell;
```

Result:

```text
┌─basecell─┐
│       12 │
└──────────┘
```

## h3HexAreaM2 {#h3hexaream2}

Returns average hexagon area in square meters at the given resolution.

**Syntax**

```sql
h3HexAreaM2(resolution)
```

**Parameter**

- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned value**

- Area in square meters. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3HexAreaM2(13) AS area;
```

Result:

```text
┌─area─┐
│ 43.9 │
└──────┘
```

## h3HexAreaKm2 {#h3hexareakm2}

Returns average hexagon area in square kilometers at the given resolution.

**Syntax**

```sql
h3HexAreaKm2(resolution)
```

**Parameter**

- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned value**

- Area in square kilometers. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3HexAreaKm2(13) AS area;
```

Result:

```text
┌──────area─┐
│ 0.0000439 │
└───────────┘
```

## h3IndexesAreNeighbors {#h3indexesareneighbors}

Returns whether or not the provided [H3](#h3-index) indexes are neighbors.

**Syntax**

```sql
h3IndexesAreNeighbors(index1, index2)
```

**Arguments**

- `index1` — Hexagon index number. [UInt64](../../data-types/int-uint.md).
- `index2` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- `1` — Indexes are neighbours. [UInt8](../../data-types/int-uint.md).
- `0` — Indexes are not neighbours. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3IndexesAreNeighbors(617420388351344639, 617420388352655359) AS n;
```

Result:

```text
┌─n─┐
│ 1 │
└───┘
```

## h3ToChildren {#h3tochildren}

Returns an array of child indexes for the given [H3](#h3-index) index.

**Syntax**

```sql
h3ToChildren(index, resolution)
```

**Arguments**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).
- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned values**

- Array of the child H3-indexes. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
SELECT h3ToChildren(599405990164561919, 6) AS children;
```

Result:

```text
┌─children───────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [603909588852408319,603909588986626047,603909589120843775,603909589255061503,603909589389279231,603909589523496959,603909589657714687] │
└────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3ToParent {#h3toparent}

Returns the parent (coarser) index containing the given [H3](#h3-index) index.

**Syntax**

```sql
h3ToParent(index, resolution)
```

**Arguments**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).
- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned value**

- Parent H3 index. [UInt64](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3ToParent(599405990164561919, 3) AS parent;
```

Result:

```text
┌─────────────parent─┐
│ 590398848891879423 │
└────────────────────┘
```

## h3ToString {#h3tostring}

Converts the `H3Index` representation of the index to the string representation.

```sql
h3ToString(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- String representation of the H3 index. [String](../../data-types/string.md).

**Example**

Query:

```sql
SELECT h3ToString(617420388352917503) AS h3_string;
```

Result:

```text
┌─h3_string───────┐
│ 89184926cdbffff │
└─────────────────┘
```

## stringToH3 {#stringtoh3}

Converts the string representation to the `H3Index` (UInt64) representation.

**Syntax**

```sql
stringToH3(index_str)
```

**Parameter**

- `index_str` — String representation of the H3 index. [String](../../data-types/string.md).

**Returned value**

- Hexagon index number. Returns 0 on error. [UInt64](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT stringToH3('89184926cc3ffff') AS index;
```

Result:

```text
┌──────────────index─┐
│ 617420388351344639 │
└────────────────────┘
```

## h3GetResolution {#h3getresolution-1}

Returns the resolution of the [H3](#h3-index) index.

**Syntax**

```sql
h3GetResolution(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3GetResolution(617420388352917503) AS res;
```

Result:

```text
┌─res─┐
│   9 │
└─────┘
```

## h3IsResClassIII {#h3isresclassiii}

Returns whether [H3](#h3-index) index has a resolution with Class III orientation.

**Syntax**

```sql
h3IsResClassIII(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- `1` — Index has a resolution with Class III orientation. [UInt8](../../data-types/int-uint.md).
- `0` — Index doesn't have a resolution with Class III orientation. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3IsResClassIII(617420388352917503) AS res;
```

Result:

```text
┌─res─┐
│   1 │
└─────┘
```

## h3IsPentagon {#h3ispentagon}

Returns whether this [H3](#h3-index) index represents a pentagonal cell.

**Syntax**

```sql
h3IsPentagon(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- `1` — Index represents a pentagonal cell. [UInt8](../../data-types/int-uint.md).
- `0` — Index doesn't represent a pentagonal cell. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3IsPentagon(644721767722457330) AS pentagon;
```

Result:

```text
┌─pentagon─┐
│        0 │
└──────────┘
```

## h3GetFaces {#h3getfaces}

Returns icosahedron faces intersected by a given [H3](#h3-index) index.

**Syntax**

```sql
h3GetFaces(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned values**

- Array containing icosahedron faces intersected by a given H3 index. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
SELECT h3GetFaces(599686042433355775) AS faces;
```

Result:

```text
┌─faces─┐
│ [7]   │
└───────┘
```

## h3CellAreaM2 {#h3cellaream2}

Returns the exact area of a specific cell in square meters corresponding to the given input H3 index.

**Syntax**

```sql
h3CellAreaM2(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Cell area in square meters. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3CellAreaM2(579205133326352383) AS area;
```

Result:

```text
┌───────────────area─┐
│ 4106166334463.9233 │
└────────────────────┘
```

## h3CellAreaRads2 {#h3cellarearads2}

Returns the exact area of a specific cell in square radians corresponding to the given input H3 index.

**Syntax**

```sql
h3CellAreaRads2(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Cell area in square radians. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3CellAreaRads2(579205133326352383) AS area;
```

Result:

```text
┌────────────────area─┐
│ 0.10116268528089567 │
└─────────────────────┘
```

## h3ToCenterChild {#h3tocenterchild}

Returns the center child (finer) [H3](#h3-index) index contained by given [H3](#h3-index) at the given resolution.

**Syntax**

```sql
h3ToCenterChild(index, resolution)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).
- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned values**

- [H3](#h3-index) index of the center child contained by given [H3](#h3-index) at the given resolution. [UInt64](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3ToCenterChild(577023702256844799,1) AS centerToChild;
```

Result:

```text
┌──────centerToChild─┐
│ 581496515558637567 │
└────────────────────┘
```

## h3ExactEdgeLengthM {#h3exactedgelengthm}

Returns the exact edge length of the unidirectional edge represented by the input h3 index in meters.

**Syntax**

```sql
h3ExactEdgeLengthM(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Exact edge length in meters. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3ExactEdgeLengthM(1310277011704381439) AS exactEdgeLengthM;;
```

Result:

```text
┌───exactEdgeLengthM─┐
│ 195449.63163407316 │
└────────────────────┘
```

## h3ExactEdgeLengthKm {#h3exactedgelengthkm}

Returns the exact edge length of the unidirectional edge represented by the input h3 index in kilometers.

**Syntax**

```sql
h3ExactEdgeLengthKm(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Exact edge length in kilometers. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3ExactEdgeLengthKm(1310277011704381439) AS exactEdgeLengthKm;;
```

Result:

```text
┌──exactEdgeLengthKm─┐
│ 195.44963163407317 │
└────────────────────┘
```

## h3ExactEdgeLengthRads {#h3exactedgelengthrads}

Returns the exact edge length of the unidirectional edge represented by the input h3 index in radians.

**Syntax**

```sql
h3ExactEdgeLengthRads(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Exact edge length in radians. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3ExactEdgeLengthRads(1310277011704381439) AS exactEdgeLengthRads;;
```

Result:

```text
┌──exactEdgeLengthRads─┐
│ 0.030677980118976447 │
└──────────────────────┘
```

## h3NumHexagons {#h3numhexagons}

Returns the number of unique H3 indices at the given resolution.

**Syntax**

```sql
h3NumHexagons(resolution)
```

**Parameter**

- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned value**

- Number of H3 indices. [Int64](../../data-types/int-uint.md).

**Example**

Query:

```sql
SELECT h3NumHexagons(3) AS numHexagons;
```

Result:

```text
┌─numHexagons─┐
│       41162 │
└─────────────┘
```

## h3PointDistM {#h3pointdistm}

Returns the "great circle" or "haversine" distance between pairs of GeoCoord points (latitude/longitude) pairs in meters.

**Syntax**

```sql
h3PointDistM(lat1, lon1, lat2, lon2)
```

**Arguments**

- `lat1`, `lon1` — Latitude and Longitude of point1 in degrees. [Float64](../../data-types/float.md).
- `lat2`, `lon2` — Latitude and Longitude of point2 in degrees. [Float64](../../data-types/float.md).

**Returned values**

- Haversine or great circle distance in meters.[Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3PointDistM(-10.0 ,0.0, 10.0, 0.0) AS h3PointDistM;
```

Result:

```text
┌──────h3PointDistM─┐
│ 2223901.039504589 │
└───────────────────┘
```

## h3PointDistKm {#h3pointdistkm}

Returns the "great circle" or "haversine" distance between pairs of GeoCoord points (latitude/longitude) pairs in kilometers.

**Syntax**

```sql
h3PointDistKm(lat1, lon1, lat2, lon2)
```

**Arguments**

- `lat1`, `lon1` — Latitude and Longitude of point1 in degrees. [Float64](../../data-types/float.md).
- `lat2`, `lon2` — Latitude and Longitude of point2 in degrees. [Float64](../../data-types/float.md).

**Returned values**

- Haversine or great circle distance in kilometers. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3PointDistKm(-10.0 ,0.0, 10.0, 0.0) AS h3PointDistKm;
```

Result:

```text
┌─────h3PointDistKm─┐
│ 2223.901039504589 │
└───────────────────┘
```

## h3PointDistRads {#h3pointdistrads}

Returns the "great circle" or "haversine" distance between pairs of GeoCoord points (latitude/longitude) pairs in radians.

**Syntax**

```sql
h3PointDistRads(lat1, lon1, lat2, lon2)
```

**Arguments**

- `lat1`, `lon1` — Latitude and Longitude of point1 in degrees. [Float64](../../data-types/float.md).
- `lat2`, `lon2` — Latitude and Longitude of point2 in degrees. [Float64](../../data-types/float.md).

**Returned values**

- Haversine or great circle distance in radians. [Float64](../../data-types/float.md).

**Example**

Query:

```sql
SELECT h3PointDistRads(-10.0 ,0.0, 10.0, 0.0) AS h3PointDistRads;
```

Result:

```text
┌────h3PointDistRads─┐
│ 0.3490658503988659 │
└────────────────────┘
```

## h3GetRes0Indexes {#h3getres0indexes}

Returns an array of all the resolution 0 H3 indexes.

**Syntax**

```sql
h3GetRes0Indexes()
```

**Returned values**

- Array of all the resolution 0 H3 indexes. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
SELECT h3GetRes0Indexes AS indexes ;
```

Result:

```text
┌─indexes─────────────────────────────────────┐
│ [576495936675512319,576531121047601151,....]│
└─────────────────────────────────────────────┘
```

## h3GetPentagonIndexes {#h3getpentagonindexes}

Returns all the pentagon H3 indexes at the specified resolution.

**Syntax**

```sql
h3GetPentagonIndexes(resolution)
```

**Parameter**

- `resolution` — Index resolution. Range: `[0, 15]`. [UInt8](../../data-types/int-uint.md).

**Returned value**

- Array of all pentagon H3 indexes. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
SELECT h3GetPentagonIndexes(3) AS indexes;
```

Result:

```text
┌─indexes────────────────────────────────────────────────────────┐
│ [590112357393367039,590464201114255359,590816044835143679,...] │
└────────────────────────────────────────────────────────────────┘
```

## h3Line {#h3line}

Returns the line of indices between the two indices that are provided.

**Syntax**

```sql
h3Line(start,end)
```

**Parameter**

- `start` — Hexagon index number that represents a starting point. [UInt64](../../data-types/int-uint.md).
- `end` — Hexagon index number that represents an ending point. [UInt64](../../data-types/int-uint.md).

**Returned value**

Array of h3 indexes representing the line of indices between the two provided indices. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
 SELECT h3Line(590080540275638271,590103561300344831) AS indexes;
```

Result:

```text
┌─indexes────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [590080540275638271,590080471556161535,590080883873021951,590106516237844479,590104385934065663,590103630019821567,590103561300344831] │
└────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3Distance {#h3distance}

Returns the distance in grid cells between the two indices that are provided.

**Syntax**

```sql
h3Distance(start,end)
```

**Parameter**

- `start` — Hexagon index number that represents a starting point. [UInt64](../../data-types/int-uint.md).
- `end` — Hexagon index number that represents an ending point. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Number of grid cells. [Int64](../../data-types/int-uint.md).

Returns a negative number if finding the distance fails.

**Example**

Query:

```sql
 SELECT h3Distance(590080540275638271,590103561300344831) AS distance;
```

Result:

```text
┌─distance─┐
│        7 │
└──────────┘
```

## h3HexRing {#h3hexring}

Returns the indexes of the hexagonal ring centered at the provided origin h3Index and length k.

Returns 0 if no pentagonal distortion was encountered.

**Syntax**

```sql
h3HexRing(index, k)
```

**Parameter**

- `index` — Hexagon index number that represents the origin. [UInt64](../../data-types/int-uint.md).
- `k` — Distance. [UInt64](../../data-types/int-uint.md).

**Returned values**

- Array of H3 indexes. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
 SELECT h3HexRing(590080540275638271, toUInt16(1)) AS hexRing;
```

Result:

```text
┌─hexRing─────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [590080815153545215,590080471556161535,590080677714591743,590077585338138623,590077447899185151,590079509483487231] │
└─────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3GetUnidirectionalEdge {#h3getunidirectionaledge}

Returns a unidirectional edge H3 index based on the provided origin and destination and returns 0 on error.

**Syntax**

```sql
h3GetUnidirectionalEdge(originIndex, destinationIndex)
```

**Parameter**

- `originIndex` — Origin Hexagon index number. [UInt64](../../data-types/int-uint.md).
- `destinationIndex` — Destination Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Unidirectional Edge Hexagon Index number. [UInt64](../../data-types/int-uint.md).

**Example**

Query:

```sql
 SELECT h3GetUnidirectionalEdge(599686042433355775, 599686043507097599) AS edge;
```

Result:

```text
┌────────────────edge─┐
│ 1248204388774707199 │
└─────────────────────┘
```

## h3UnidirectionalEdgeIsValid {#h3unidirectionaledgeisvalid}

Determines if the provided H3Index is a valid unidirectional edge index. Returns 1 if it's a unidirectional edge and 0 otherwise.

**Syntax**

```sql
h3UnidirectionalEdgeisValid(index)
```

**Parameter**

- `index` — Hexagon index number. [UInt64](../../data-types/int-uint.md).

**Returned value**

- 1 — The H3 index is a valid unidirectional edge. [UInt8](../../data-types/int-uint.md).
- 0 — The H3 index is not a valid unidirectional edge. [UInt8](../../data-types/int-uint.md).

**Example**

Query:

```sql
 SELECT h3UnidirectionalEdgeIsValid(1248204388774707199) AS validOrNot;
```

Result:

```text
┌─validOrNot─┐
│          1 │
└────────────┘
```

## h3GetOriginIndexFromUnidirectionalEdge {#h3getoriginindexfromunidirectionaledge}

Returns the origin hexagon index from the unidirectional edge H3Index.

**Syntax**

```sql
h3GetOriginIndexFromUnidirectionalEdge(edge)
```

**Parameter**

- `edge` — Hexagon index number that represents a unidirectional edge. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Origin Hexagon Index number. [UInt64](../../data-types/int-uint.md).

**Example**

Query:

```sql
 SELECT h3GetOriginIndexFromUnidirectionalEdge(1248204388774707197) AS origin;
```

Result:

```text
┌─────────────origin─┐
│ 599686042433355773 │
└────────────────────┘
```

## h3GetDestinationIndexFromUnidirectionalEdge {#h3getdestinationindexfromunidirectionaledge}

Returns the destination hexagon index from the unidirectional edge H3Index.

**Syntax**

```sql
h3GetDestinationIndexFromUnidirectionalEdge(edge)
```

**Parameter**

- `edge` — Hexagon index number that represents a unidirectional edge. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Destination Hexagon Index number. [UInt64](../../data-types/int-uint.md).

**Example**

Query:

```sql
 SELECT h3GetDestinationIndexFromUnidirectionalEdge(1248204388774707197) AS destination;
```

Result:

```text
┌────────destination─┐
│ 599686043507097597 │
└────────────────────┘
```

## h3GetIndexesFromUnidirectionalEdge {#h3getindexesfromunidirectionaledge}

Returns the origin and destination hexagon indexes from the given unidirectional edge H3Index.

**Syntax**

```sql
h3GetIndexesFromUnidirectionalEdge(edge)
```

**Parameter**

- `edge` — Hexagon index number that represents a unidirectional edge. [UInt64](../../data-types/int-uint.md).

**Returned value**

A tuple consisting of two values `tuple(origin,destination)`:

- `origin` — Origin Hexagon index number. [UInt64](../../data-types/int-uint.md).
- `destination` — Destination Hexagon index number. [UInt64](../../data-types/int-uint.md).

Returns `(0,0)` if the provided input is not valid.

**Example**

Query:

```sql
 SELECT h3GetIndexesFromUnidirectionalEdge(1248204388774707199) AS indexes;
```

Result:

```text
┌─indexes─────────────────────────────────┐
│ (599686042433355775,599686043507097599) │
└─────────────────────────────────────────┘
```

## h3GetUnidirectionalEdgesFromHexagon {#h3getunidirectionaledgesfromhexagon}

Provides all of the unidirectional edges from the provided H3Index.

**Syntax**

```sql
h3GetUnidirectionalEdgesFromHexagon(index)
```

**Parameter**

- `index` — Hexagon index number that represents a unidirectional edge. [UInt64](../../data-types/int-uint.md).

**Returned value**

Array of h3 indexes representing each unidirectional edge. [Array](../../data-types/array.md)([UInt64](../../data-types/int-uint.md)).

**Example**

Query:

```sql
 SELECT h3GetUnidirectionalEdgesFromHexagon(1248204388774707199) AS edges;
```

Result:

```text
┌─edges─────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [1248204388774707199,1320261982812635135,1392319576850563071,1464377170888491007,1536434764926418943,1608492358964346879] │
└───────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3GetUnidirectionalEdgeBoundary {#h3getunidirectionaledgeboundary}

Returns the coordinates defining the unidirectional edge.

**Syntax**

```sql
h3GetUnidirectionalEdgeBoundary(index)
```

**Parameter**

- `index` — Hexagon index number that represents a unidirectional edge. [UInt64](../../data-types/int-uint.md).

**Returned value**

- Array of pairs '(lon, lat)'. [Array](../../data-types/array.md)([Float64](../../data-types/float.md), [Float64](../../data-types/float.md)).

**Example**

Query:

```sql
 SELECT h3GetUnidirectionalEdgeBoundary(1248204388774707199) AS boundary;
```

Result:

```text
┌─boundary────────────────────────────────────────────────────────────────────────┐
│ [(37.42012867767779,-122.03773496427027),(37.33755608435299,-122.090428929044)] │
└─────────────────────────────────────────────────────────────────────────────────┘
```
