import re
import sys

from praktika.info import Info

from ci.jobs.scripts.workflow_hooks.pr_description import Labels

autogenerated_doc_files = [
    "src/Storages/MergeTree/MergeTreeSettings.cpp",
    "src/Core/Settings.cpp",
]


def belongs_to_autogenerated_category(filename):
    """
    Check if a C++ file contains FunctionDocumentation::Category::x where x is one of the
    autogenerated function categories that are now completely automatically generated
    """

    autogenerated_function_categories = [
        "Array",
        "Arithmetic",
        "Conditional",
        "Bitmap",
        "Bit",
        "Comparison",
        "Conditional",
        "Distance",
        "DateAndTime",
        "Null",
    ]

    try:
        with open(filename, "r", encoding="utf-8") as file:
            content = file.read()

        category_pattern = (
            r"FunctionDocumentation::Category::("
            + "|".join(autogenerated_function_categories)
            + r")"
        )
        matches = re.findall(category_pattern, content)

        if matches:
            print(
                f"Found autogenerated function categories in {filename}: {set(matches)}"
            )
            return True
        else:
            print(f"No autogenerated function categories found in {filename}")
            return False

    except Exception as e:
        print(f"Error reading file {filename}: {e}")
        return False


def check_docs():
    info = Info()
    if Labels.PR_FEATURE in info.pr_labels:
        changed_files = info.get_kv_data("changed_files")

        has_doc_changes = any(
            file.startswith("docs/") or file in autogenerated_doc_files
            for file in changed_files
        )

        has_source_code_docs_registered = any(
            belongs_to_autogenerated_category(file)
            for file in changed_files
            if file.endswith(".cpp")
        )

        if not has_doc_changes and not has_source_code_docs_registered:
            print(
                "No changes in docs for new feature and no autogenerated function categories found"
            )
            return False
    return True


if __name__ == "__main__":
    if not check_docs():
        sys.exit(1)
