///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoLogThread.cc  
// ----------------                                                     
// Cego log thread class implementation
//                                                         
// Design and Implementation by Bjoern Lemke
//               
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoLogThread
//
// Description: Logging Thread imlementation to handle db log messages
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/Host.h>
#include <lfcbase/Sleeper.h>

// CEGO INCLUDES
#include "CegoLogThread.h"
#include "CegoLogThreadPool.h"
#include "CegoDefs.h"
#include "CegoXMLdef.h"

CegoLogThread::CegoLogThread() : Thread()
{
}

CegoLogThread::CegoLogThread(CegoLogThreadPool *pPool, CegoDatabaseManager *pDBMng) : Thread()
{
    _pDBMng = pDBMng;
    _pPool = pPool;
    _modId = pDBMng->getModId("CegoLogThread");
}

CegoLogThread::~CegoLogThread()  
{
}

const Chain& CegoLogThread::lastAction() const
{
    return _lastAction;
}

void* CegoLogThread::job(void* arg)
{
    _idx = *(long*)arg;

    _pTim = new NanoTimer();
    
    _pPool->setTid(_idx, getTid());

    unsigned queueDelay = _pDBMng->getQueueDelay();
    
    while ( ! _pPool->isTerminated() )
    {	
	try
	{
	    _pTim->reset();
	    _pTim->start();

	    _pRequest = _pPool->nextRequest();
	    
	    if ( _pRequest ) 
	    {		
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": Serving service request"));
		_pPool->setState(_idx, CegoLogThreadPool::CONNECTED);

		_pDBMng->increaseActiveLogThread();

		CegoLogHandler *pLH = 0;

		try
		{
		    pLH = new CegoLogHandler(_pDBMng, _pRequest);		    
		    serveSession(pLH);
		    delete pLH;
		}
		catch ( Exception e)
		{	
		    if ( pLH )
			delete pLH;

		    Chain msg;
		    Chain module;
		    unsigned line;
		    
		    Chain exep;
		    while ( e.pop(module, line, msg) )
		    {
			exep += Chain("\n\t") + module + Chain("(") +  Chain(line) + Chain(") : ") + msg;
		    }

		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": Log session aborted : ") + exep);
		}

		_pDBMng->decreaseActiveLogThread();
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": service request finished"));
		_pPool->setState(_idx, CegoLogThreadPool::READY);

		delete _pRequest;
	    }
	    else
	    {
		Sleeper s;
		s.microSleep(queueDelay);
	    }

	    _pTim->stop();
	    _pPool->addThreadIdle(_idx, _pTim->getSum());       
	}
	catch ( Exception e)
	{
	    Chain msg;
	    e.pop(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": ") + msg);
	}
    }
    return 0;
}

void CegoLogThread::serveSession(CegoLogHandler *pLH)
{   
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": serving session"));
    
    Chain tableSet;
    Chain logUser;
    Chain logPwd;
    
    if ( pLH->acceptLogSession(tableSet, logUser, logPwd) )
    {
	Chain msg;
	bool isTrace;
	if ( _pDBMng->checkUser(logUser, logPwd, msg, isTrace) == false )
	{
	    throw Exception(EXLOC, msg); 
	}

	char* logEntry;
	unsigned len;
	unsigned logCount=0;

	File* pLogFile = 0;

	unsigned offset=0;

	_pPool->setState(_idx, CegoLogThreadPool::BUSY);
	      
	Chain dbHost;
	_pDBMng->getDBHost(dbHost);
	_pDBMng->setSecondary(tableSet, dbHost);
	
	_lastAction = Chain("Recovering tableset ") + tableSet;
	
	_pTim->stop();
	_pPool->addThreadIdle(_idx, _pTim->getSum());

	while ( pLH->receiveLogEntry(logEntry, len) )
	{
	    CegoLogRecord lr;
	    lr.decodeLSN(logEntry);
	    lr.decodeLogAction(logEntry);
	    
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": receiving lsn ") + Chain(lr.getLSN()));

	    if ( logCount % LOGMNG_MAXLOGCOUNT == 0 || lr.getAction() == CegoLogRecord::LOGREC_SYNC )
	    {		
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": switching logfiles on lsn ") + Chain(lr.getLSN()));
		if ( pLogFile )
		{
		    pLogFile->seek(0);
		    pLogFile->writeByte((char*)&offset, sizeof(unsigned));
		    pLogFile->trunc(offset);
		    
		    pLogFile->close();
		    delete pLogFile;
		}
		
		ListT<Chain> archPathList;
		ListT<Chain> archIdList;
		_pDBMng->getArchLogInfo(tableSet, archIdList, archPathList);
		
		Chain *pArchPath = archPathList.First();
		    
		if ( pArchPath )
		{
		    Chain logFile = *pArchPath + Chain("/") + _pDBMng->getArchiveLogName(tableSet, lr.getLSN());

		    _pDBMng->log(_modId, Logger::NOTICE, Chain("Thread ") + Chain(_idx) + Chain(": Writing to next logfile ") + logFile);

		    pLogFile=new File(logFile);
		    pLogFile->open(File::WRITE);
		    offset=0;
		    pLogFile->writeByte((char*)&offset, sizeof(unsigned));

		    logCount=0;
		    pLH->sendAck();
		}
		else
		{
		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": cannot get archive path"));
		    pLH->sendNack();
		    _pPool->setState(_idx, CegoLogThreadPool::CONNECTED);
		    return;
		}
	    }
	    else
	    {
		pLH->sendAck();
	    }
	    
	    pLogFile->writeByte((char*)&len, sizeof(unsigned));
	    pLogFile->writeByte(logEntry, len);
	    logCount++;
	    offset = offset + len + sizeof(unsigned);

	    _pPool->incNumRequest(_idx);
	}
	
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": Closing logfile ") + pLogFile->getFileName());

	if ( pLogFile )
	{
	    pLogFile->seek(0);
	    pLogFile->writeByte((char*)&offset, sizeof(unsigned));
	    pLogFile->trunc(offset);
	    
	    pLogFile->close();
	    delete pLogFile;
	}

	_pTim->reset();
	_pTim->start();
	
	_pPool->setState(_idx, CegoLogThreadPool::CONNECTED);	
    }

    _pTim->stop();
    _pPool->addThreadIdle(_idx, _pTim->getSum());

    _pTim->reset();
    _pTim->start();
}
