///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoDistManager.cc
// ------------------
// Cego distributed table manager class implementation
//                                                         
// Design and Implementation by Bjoern Lemke               
//
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoDistManager
//
// Description: Distributed table manager
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>

// CEGO INCLUDES
#include "CegoDistManager.h"
#include "CegoAVLIndexCursor.h"
#include "CegoAVLIndexManager.h"
#include "CegoBTreeManager.h"
#include "CegoXMLdef.h"
#include "CegoTypeConverter.h"
#include "CegoRecoveryManager.h"
#include "CegoAction.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoDistManager::ActiveTSUser::ActiveTSUser()
{
}
 
CegoDistManager::ActiveTSUser::ActiveTSUser(const Chain& tableSet)
{
    _tableSet = tableSet;
}

CegoDistManager::ActiveTSUser::ActiveTSUser (const Chain& tableSet, const Chain& userName, const Chain& passwd)
{
    _tableSet = tableSet;
    _userName = userName;
    _passwd = passwd;    
}

CegoDistManager::ActiveTSUser::~ActiveTSUser()
{
}
	
const Chain& CegoDistManager::ActiveTSUser::getUserName() const
{
    return _userName;
}

const Chain& CegoDistManager::ActiveTSUser::getPasswd() const
{
    return _passwd;
}

CegoDistManager::ActiveTSUser& CegoDistManager::ActiveTSUser::operator = (const ActiveTSUser& u)
{
    _tableSet = u._tableSet;
    _userName = u._userName;
    _passwd = u._passwd;
    return (*this);   
}

bool CegoDistManager::ActiveTSUser::operator == (const CegoDistManager::ActiveTSUser& u) const
{
    if ( _tableSet == u._tableSet)
	return true;
    return false;
}

CegoDistManager::CegoDistManager(CegoDatabaseManager *pDBMng) : CegoTableManager(pDBMng)
{
    _modId = pDBMng->getModId("CegoDistManager");
    _pRepHash = new HashT<CegoReplacer>(REPLACER_MAXNUM, REPLACER_HASHRANGE);
    _authEnabled = true;
    _numAllocated = 0;
    _threadId = 0;
    _pPA = 0; 
}

CegoDistManager::~CegoDistManager()
{
    if ( _pPA )
	delete _pPA;

    if ( _pRepHash )
    {
	CegoReplacer *pRep = _pRepHash->First();
	while ( pRep )
	{
	    pRep->clean();
	    pRep = _pRepHash->Next();
	}
	if ( _pRepHash )
	    delete _pRepHash;
    }
}

void CegoDistManager::createTableSet(const Chain& tableSet)
{
    createBasicTableSet(tableSet);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);    

    startTableSet(tableSet, dbHost, false, false, false, false);
    
    _pDBMng->log(_modId, Logger::NOTICE, Chain("Creating system space ..."));
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
    createSystemSpace(tabSetId);

    stopTableSet(tableSet, false);    
}

void CegoDistManager::startDistTableSet(const Chain& tableSet, const Chain& secondary, bool cleanIt, bool autoCorrect, bool doTempReset, bool cpDump)
{
    startTableSet(tableSet, secondary, cleanIt, cpDump, autoCorrect, doTempReset);
}

void CegoDistManager::registerObjects(const Chain& tableSet)
{
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
    
    ListT<Chain> tabList;
    getObjectList(tabSetId, CegoObject::TABLE, tabList);

    Chain *pTableName = tabList.First();
    while ( pTableName )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding table ") + *pTableName + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pTableName, CegoObject::TABLE);
	pTableName = tabList.Next();
    }

    ListT<Chain> viewList;
    getObjectList(tabSetId, CegoObject::VIEW, viewList);
    Chain *pViewName = viewList.First();
    while ( pViewName )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding view ") + *pViewName + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pViewName, CegoObject::VIEW);
	pViewName = viewList.Next();
    }
    
    ListT<Chain> idxList;
    getObjectList(tabSetId, CegoObject::AVLTREE, idxList);
    Chain *pIdxName = idxList.First();
    while ( pIdxName )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding index ") + *pIdxName + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pIdxName, CegoObject::AVLTREE);
	pIdxName = idxList.Next();
    }

    ListT<Chain> btreeList;
    getObjectList(tabSetId, CegoObject::BTREE, btreeList);
    Chain *pBTreeName = btreeList.First();
    while ( pBTreeName )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding btree ") + *pBTreeName + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pBTreeName, CegoObject::BTREE);
	pBTreeName = btreeList.Next();
    }

    ListT<Chain> procList;
    getObjectList(tabSetId, CegoObject::PROCEDURE, procList);
    Chain *pProcName = procList.First();
    while ( pProcName )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding procedure ") + *pProcName + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pProcName, CegoObject::PROCEDURE);
	pProcName = procList.Next();
    }

    ListT<Chain> fkeyList;
    getObjectList(tabSetId, CegoObject::FKEY, fkeyList);
    Chain *pFKey = fkeyList.First();
    while ( pFKey )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding foreign key ") + *pFKey + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pFKey, CegoObject::FKEY);
	pFKey = fkeyList.Next();
    }

    ListT<Chain> checkList;
    getObjectList(tabSetId, CegoObject::CHECK, checkList);
    Chain *pCheck = checkList.First();
    while ( pCheck )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding check ") + *pCheck + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pCheck, CegoObject::CHECK);
	pCheck = checkList.Next();
    }

    ListT<Chain> triggerList;
    getObjectList(tabSetId, CegoObject::TRIGGER, triggerList);
    Chain *pTrigger = triggerList.First();
    while ( pTrigger )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding trigger ") + *pTrigger + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pTrigger, CegoObject::TRIGGER);
	pTrigger = triggerList.Next();
    }

    ListT<Chain> aliasList;
    getObjectList(tabSetId, CegoObject::ALIAS, aliasList);
    Chain *pAlias = aliasList.First();
    while ( pAlias )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Adding alias ") + *pAlias + Chain(" to tableset ") + tableSet);
#endif
	_pDBMng->addObject(tabSetId, *pAlias, CegoObject::ALIAS);
	pAlias = aliasList.Next();
    }
}

void CegoDistManager::stopDistTableSet(const Chain& tableSet, bool archComplete)
{    
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
    
    _pDBMng->removeAllObjects(tabSetId);
    _pDBMng->releaseTableCache(tableSet);
    _pDBMng->releaseQueryCache(tableSet);

    stopTableSet(tableSet, archComplete);
}

void CegoDistManager::startTableSet(const Chain& tableSet, const Chain& secondary, bool cleanIt, bool cpDump, bool doAutoCorrect, bool doTempReset)
{
    _pDBMng->log(_modId, Logger::NOTICE, Chain("Starting tableset ") + tableSet + Chain(" ..."));
    
    _pDBMng->setCheckPointDump(tableSet, cpDump);
    
    // check, if we have a real problem
    Chain runState = _pDBMng->getTableSetRunState(tableSet);
    if ( runState == Chain(XML_CHECKPOINT_VALUE) )
    {
	throw Exception(EXLOC, Chain("Checkpoint crash detected, tableset might be inconsistent")); 	
    }

    Chain tsTicketName = _pDBMng->getTSTicket(tableSet);
    
    File tsTicket(tsTicketName);
    if ( tsTicket.exists()) 
	throw Exception(EXLOC, Chain("Backup tableset ticket exists"));
    
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
    regDataFiles(tableSet);
    
    Chain dbHost;
    _pDBMng->getDBHost(dbHost);
    
    if ( secondary != dbHost )
    {
	unsigned logPort;
	_pDBMng->getLogPort(logPort);

	Chain logUser = _pDBMng->getTSLogUser(tableSet);
	Chain logPwd = _pDBMng->getUserPwd(logUser);	

	_pDBMng->allocateLogConnection(tabSetId, tableSet, secondary, logPort, logUser, logPwd);
	long cplsn = _pDBMng->getCommittedLSN(tableSet);

	_pDBMng->setCurrentLSN(tabSetId, cplsn);
	_pDBMng->startLog(tabSetId);
    }
    else
    {
	// we have to set tableset state for recovery
	_pDBMng->setTableSetRunState(tableSet, XML_RECOVERY_VALUE);

        // set active log file
        _pDBMng->setActiveLogFile(tableSet);

        // get max written lsn
        long maxlsn = _pDBMng->getMaxLSN(tabSetId);

        _pDBMng->log(_modId, Logger::NOTICE, Chain("Setting lsn to ") + Chain(maxlsn));
        _pDBMng->setCurrentLSN(tabSetId, maxlsn);

        registerObjects(tableSet);

        // get checkpoint lsn
        long cplsn = _pDBMng->getCommittedLSN(tableSet);

	if ( cplsn == maxlsn || ( cplsn == 0 && maxlsn == 0 ) )
	{
	    _pDBMng->log(_modId, Logger::NOTICE, Chain("Tableset ") + tableSet + Chain(" in sync at lsn=") + Chain(cplsn) + Chain(", no recovery required"));
	    finishOpenTransaction(tabSetId);
	}
	else if ( cplsn < maxlsn )
	{	    
	    long minlsn = _pDBMng->getMinLSN(tabSetId);
	    
	    if ( cplsn + 1 < minlsn )
	    {
		throw Exception(EXLOC, Chain("Checkpoint lsn=") + Chain(cplsn) + Chain(" is too small ( minlsn=") + Chain(minlsn) + Chain(")")); 
	    }
	    _pDBMng->log(_modId, Logger::NOTICE, Chain("Tableset ") + tableSet + Chain(" not in sync, recovery required ( Commited lsn=") + Chain(cplsn) + Chain(")"));

	    _pDBMng->setCurrentLSN(tabSetId, cplsn);
	    
	    unsigned long long lsn;
	    long long ts = 0;

	    _pDBMng->log(_modId, Logger::NOTICE, Chain("Starting recovery ... "));    

	    setIgnoreInvalid(true);

	    try
	    {
		// we have to set tableset state for recovery, so occupied redo logs are archived via log manager
		// _pDBMng->setTableSetRunState(tableSet, XML_RECOVERY_VALUE);
		
		CegoRecoveryManager recoveryMng(this, CegoRecoveryManager::LOCAL);
		recoveryMng.recoverCurrentTransactionLog(tabSetId, 0, lsn, ts);

		// increase lsn to next
		_pDBMng->setCurrentLSN(tabSetId, lsn);
				
		_pDBMng->log(_modId, Logger::NOTICE, Chain("Recovery finished"));    
		
		_pDBMng->log(_modId, Logger::NOTICE, Chain("Closing open transactions ... "));    
		finishOpenTransaction(tabSetId);
		_pDBMng->log(_modId, Logger::NOTICE, Chain("Transactions closed"));    
		
		setIgnoreInvalid(false);
		
		// set lsn to last lsn in log
		_pDBMng->writeCheckPoint(tableSet, true, false, _pLockHandle);
	    }
	    catch ( Exception e)
	    {
	    	_pDBMng->setTableSetRunState(tableSet, XML_OFFLINE_VALUE);
		throw Exception(EXLOC, Chain("Cannot start tableset ") + tableSet, e);
	    }
	}
	else
	{
	    throw Exception(EXLOC, "Checkpoint lsn is too high ( maxlsn = " + Chain(maxlsn) + ", lsn=" + Chain(cplsn) + Chain(")")); 
	}

	// do autocorrect here

	if ( doAutoCorrect )
	{	    	   
	    try
	    {	
		bool isAutoCorrect = _pDBMng->getAutoCorrect(tableSet);
		
		if ( isAutoCorrect )
		{
		    _pDBMng->log(_modId, Logger::NOTICE, Chain("Perform autocorrect for tableset ") + tableSet + Chain(" ..."));		    
		    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
		    correctTableSet(tabSetId, false);
		}
	    }
	    catch ( Exception e)
	    {
		// _pDBMng->log(_modId, Logger::LOGERR, msg);
		stopTableSet(tableSet, false);
		throw Exception(EXLOC, Chain("Cannot correct tableset ") + tableSet, e);
	    }
	}

	if ( doTempReset )
	{
	    _pDBMng->log(_modId, Logger::NOTICE, Chain("Resetting temp space for tableset ") + tableSet + Chain(" ..."));
	    resetTemp(tableSet);
	}
	
	// setting current lsn, 
	// since the cleanTableSet procedure might fix buffers and might write a checkpoint,
	// this must be done BEFORE the cleanIt option is performced.
	// Fixed with version 2.47.11

	// Fix in version 2.47.14 : the current lsn has to be setup before registerObjects
	// since also this method might trigger checkpoints
	
	// _pDBMng->log(_modId, Logger::NOTICE, Chain("Setting lsn to ") + Chain(maxlsn));
	// _pDBMng->setCurrentLSN(tabSetId, maxlsn);

	if ( cleanIt )
	{
	    _pDBMng->log(_modId, Logger::NOTICE, Chain("Cleaning tableset ") + tableSet + Chain(" ..."));
	    unsigned cleanCount = cleanTableSet(tabSetId);
	    _pDBMng->log(_modId, Logger::NOTICE, Chain(cleanCount) + Chain(" pages cleaned"));
	}
	
	// now starting log
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Starting log for tableset ") + tableSet + Chain(" ..."));
	_pDBMng->startLog(tabSetId);
    }
    
    _pDBMng->allocateTableCache(tableSet);
    _pDBMng->allocateQueryCache(tableSet);

    _pDBMng->setTableSetRunState(tableSet, XML_ONLINE_VALUE); 
    _pDBMng->setTableSetSyncState(tableSet, XML_SYNCHED_VALUE); 
}

bool CegoDistManager::distObjectExists(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType objType)
{
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    Chain hostName = _pDBMng->getPrimary(tabSetId);
    
    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	if ( _pDBMng->objectExists(tabSetId, objName, objType) )
	    return true;
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	Chain msg;
	CegoDbHandler::ResultType res;
    
	res = pSH->reqObjectInfoOp(tabSetId, objName, objType);

	_pDBMng->releaseSession(pSH);
	
	if ( res == CegoDbHandler::DB_INFO )	    
	    return true;	
    }
    return false;
}

void CegoDistManager::getDistObject(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type, CegoDecodableObject& oe)
{
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, objName, type, CegoXMLSpace::READ, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + objName;
	throw Exception(EXLOC, msg); 
    }

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Getting object ") + objName + Chain(" of type ") + CegoTypeConverter::getObjectTypeString(type));
#endif

    Chain hostName = _pDBMng->getPrimary(tabSetId);
    
    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	getLocalObject(tabSetId, objName, type, oe);
	oe.setLocal(true);
	oe.setTableSet(tableSet);
    }
    else
    {	
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	Chain msg;
	CegoDbHandler::ResultType res;
    
	res = pSH->reqObjectInfoOp(tabSetId, objName, type);
	
	if ( res == CegoDbHandler::DB_INFO )
	{
	    // oe.putElement( pSH->getObjElement() );
	    oe.setLocal(false);
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::getLocalObject(unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type, CegoDecodableObject& oe)
{
    getObject(tabSetId, objName, type, oe);
}

void CegoDistManager::truncateDistTable(const Chain& tableName, const Chain& tableSet)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Truncating table ") + tableName);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
    
    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, tableName, CegoObject::TABLE, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for table ") + tableName;
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(tabSetId);
    
    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	truncateTable(tabSetId, tableName);
	_pDBMng->cleanCache(tabSetId, CegoObject::TABLE, tableName);	
    }
    else
    {	
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);
	
	Chain msg;
	CegoDbHandler::ResultType res = CegoDbHandler::DB_ERROR;

	// TODO
	// res = pSH->reqTruncateTableOp(tableSet, objName, type);

	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::dropDistObject(const Chain& objName, const Chain& tableSet, CegoObject::ObjectType type)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Dropping object ") + objName);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, objName, type, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + objName;
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(tabSetId);
    
    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	dropObjectSynced(tabSetId, objName, type);
	
	if ( type == CegoObject::TABLE || type == CegoObject::VIEW || type == CegoObject::PROCEDURE )
	{
	    _pDBMng->cleanCache(tabSetId, type, objName);
	}
    }
    else
    {	
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);
	
	Chain msg;
	CegoDbHandler::ResultType res;

	res = pSH->reqDropObjectOp(tableSet, objName, type);

	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::createDistDataTable(const Chain& tableSet, const Chain& tableName, CegoObject::ObjectType type, ListT<CegoField>& fldList, ListT<CegoField>& idxList, bool useColumnId)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global table ") + tableName + Chain(" in tableset ") + tableSet);
#endif

    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, tableName, type, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + tableName;
	throw Exception(EXLOC, msg); 
    }

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {	
	createLocalDataTable(tabSetId, tableName, CegoObject::TABLE, fldList, idxList, useColumnId);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;
	res = pSH->reqCreateTableOp(tableSet, tableName, type,  fldList, idxList);

	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::createLocalDataTable(unsigned tabSetId, const Chain& tableName, CegoObject::ObjectType type, ListT<CegoField>& fldList, ListT<CegoField>& idxList, bool useColumnId)
{    
    createDataTable(tabSetId, tableName, type, fldList, useColumnId);
    _pDBMng->addObject(tabSetId, tableName, CegoObject::TABLE);
    
    if ( ! idxList.isEmpty() )
    {
	Chain idxName = tableName + Chain(TABMNG_PBTREE_SUFFIX);
	
	try
	{	    
	    createPrimaryIndexTable(tabSetId, idxName, tableName, idxList);
	}
	catch ( Exception e )
	{	    
	    removeObject(tabSetId, tableName, CegoObject::TABLE);
	    _pDBMng->removeObject(tabSetId, tableName, CegoObject::TABLE);
	    throw Exception(EXLOC, Chain("Cannot create primary index table"), e);
	}
	_pDBMng->addObject(tabSetId, idxName, CegoObject::BTREE);
    }
}

void CegoDistManager::createDistIndexTable( const Chain& tableSet, const Chain& indexName, const Chain& tableName, ListT<CegoField>& idxList, CegoObject::ObjectType type, bool isCached)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global index ") + indexName + Chain(" in tableset ") + tableSet);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, indexName, CegoObject::AVLTREE, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for index ") + indexName;
	throw Exception(EXLOC, msg); 
    }

    Chain hostName =_pDBMng->getPrimary(tableSet);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	createIndexTableSynced(tabSetId, indexName, tableName, type, idxList, isCached);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	pSH->reqCreateIndexOp(tableSet, indexName, tableName, type, idxList);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::createDistForeignKey( const Chain& tableSet, const Chain& fkey, const Chain& tableName, const ListT<CegoField>& keyList, const Chain& refTable, const ListT<CegoField>& refList)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global foreign key ") + fkey + Chain(" in tableset ") + tableSet);
#endif

    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, tableName, CegoObject::TABLE, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + tableName;
	throw Exception(EXLOC, msg); 
    }

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	createLocalForeignKey(tabSetId, fkey, tableName, keyList, refTable, refList);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	pSH->reqCreateFKeyOp(tabSetId, fkey, tableName, keyList, refTable, refList);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::createLocalForeignKey(unsigned tabSetId, const Chain& fkey, const Chain& tableName, const ListT<CegoField>& keyList, const Chain& refTable, const ListT<CegoField>& refList)
{
    createForeignKey(tabSetId, fkey, tableName, keyList, refTable, refList);

    _pDBMng->addObject(tabSetId, fkey, CegoObject::FKEY);
}

void CegoDistManager::createDistCheck( const Chain& tableSet, const Chain& checkName, const Chain& tableName, CegoPredicate *pPredDesc)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global check ") + checkName + Chain(" in tableset ") + tableSet);
#endif
    
    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, tableName, CegoObject::TABLE, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + tableName;
	throw Exception(EXLOC, msg); 
    }

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	createLocalCheck(tabSetId, checkName, tableName, pPredDesc);
    }
    else
    {	
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	// no more supported
	// pSH->reqCreateCheckOp(tableSet, checkName, tableName, pPredDesc);
	_pDBMng->releaseSession(pSH);
    }    
}

void CegoDistManager::createLocalCheck(unsigned tabSetId, const Chain& checkName, const Chain& tableName, CegoPredicate *pPredDesc)
{
    createCheck(tabSetId, checkName, tableName, pPredDesc);    
    _pDBMng->addObject(tabSetId, checkName, CegoObject::CHECK);
}

void CegoDistManager::reorgDistObject(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Reorg global object ") + objName + Chain(" in tableset ") + tableSet);
#endif

    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, objName, type, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + objName;
	throw Exception(EXLOC, msg); 
    }

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	reorgObjectSynced(tabSetId, objName, type);	
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;
	res = pSH->reqReorgObjectOp(tableSet, objName, type);

	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::startDistTransaction( const Chain& tableSet)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Starting global transaction in tableset ") + tableSet);
#endif

    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	beginTransaction(tabSetId, true);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;
	res = pSH->reqStartTransactionOp(tableSet);

	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

long CegoDistManager::endDistTransaction( const Chain& tableSet)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Ending global transaction in tableset ") + tableSet);
#endif

    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	return commitTransactionSynced(tabSetId);  
    }
    else
    {
	Chain user;
	Chain password;

	long numCommitOp=0;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;
	res = pSH->reqCommitTransactionOp(tableSet);

	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}

	// TODO : numCommitOp still not set up for distributed queries
	return numCommitOp;
    }
}

long CegoDistManager::rollbackDistTransaction(const Chain& tableSet)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Rollbacking global transaction in tableset ") + tableSet);
#endif

    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {	
	return rollbackTransactionSynced(tabSetId);
    }
    else
    {
	Chain user;
	Chain password;

	long numRollbackOp=0;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;
	res = pSH->reqRollbackTransactionOp(tableSet);

	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}

	// TODO : numRollbackOp still not set up for distributed queries
	return numRollbackOp;
    }
}

unsigned long long CegoDistManager::getDistTid( const Chain& tableSet)
{
    Chain hostName = _pDBMng->getPrimary(tableSet);
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	return getLocalTid(tabSetId);
    }
    else
    {	
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	Chain msg;
	CegoDbHandler::ResultType res;
    
	unsigned long long tid = 0;
	
	res = pSH->reqGetTidOp(tableSet);
	
	if ( res == CegoDbHandler::DB_INFO )
	{
	    Chain tidString;
	    pSH->getArgValue(XML_TID_ATTR, tidString);
	    
	    tid = tidString.asUnsignedLongLong();

	    _pDBMng->releaseSession(pSH);	    
	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}	
	return tid;
    }
}

unsigned long long CegoDistManager::getLocalTid(unsigned tabSetId)
{
    return getTID(tabSetId);
}

void CegoDistManager::setAllocatedSortArea(unsigned long long n)
{
    _numAllocated = n;
}

unsigned long long CegoDistManager::getAllocatedSortArea() const
{
    return _numAllocated;
}

void CegoDistManager::insertDistDataTable(CegoTableObject& oe, ListT<CegoField>& fvl)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Inserting into global table ") + oe.getName());
#endif

    if ( _authEnabled == true && _pDBMng->verifyAccess(oe.getTabSetId(), oe.getName(), oe.getType(), CegoXMLSpace::WRITE, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + oe.getName();
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(oe.getTabSetId());    

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	insertLocalDataTable(oe, fvl);
    }
    else
    {	
	Chain tableSet =  _pDBMng->getTabSetName(oe.getTabSetId());

	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);
    
	Chain msg;
	CegoDbHandler::ResultType res;
	res = pSH->reqInsertOp(tableSet, oe.getName(), fvl);
	
	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);   
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::insertLocalDataTable(CegoTableObject& oe, ListT<CegoField>& fvl)
{    
    _pDBMng->useObject(oe.getTabSetId(), oe.getName(), oe.getType(), CegoDatabaseManager::SHARED,  this);
    
    try
    {
	ListT<CegoTableObject> idxList;
	ListT<CegoBTreeObject> btreeList;
	ListT<CegoKeyObject> keyList;
	ListT<CegoCheckObject> checkList;
	ListT<CegoTriggerObject> triggerList;
	ListT<CegoAliasObject> aliasList;
	unsigned numInvalid;
	
	bool doAppend;
	bool doLogging = true;
	
	getObjectListByTable(oe.getTabSetId(), oe.getName(), idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
	if ( numInvalid > 0 )
	{	
	    // for ongoing transactions, we don't support online index build up
	    if ( getTID(oe.getTabSetId() != 0 ))
	    {
		throw Exception(EXLOC, Chain("Invalid index detected, must be valid for transactions"));
	    }	
	    doAppend = true;
	}
	else
	{
	    doAppend = _doAppend;
	}

	CegoTriggerObject *pTO = triggerList.First();
	while ( pTO )
	{
	    if ( pTO->isOnInsert() && pTO->isBefore() )
	    {
		CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());
		pTrigger->execute(&fvl);		
	    }
	    pTO = triggerList.Next();
	}
	
	CegoDataPointer sysEntry;
	Chain virginIndex;
	CegoDataPointer dp;
	insertDataTable(oe, fvl, idxList, btreeList, keyList, checkList, sysEntry, virginIndex, dp, doLogging, doAppend, true, true);
	
	pTO = triggerList.First();
	while ( pTO )
	{
	    if ( pTO->isOnInsert() && pTO->isBefore() == false )
	    {
		CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());
		pTrigger->execute(&fvl);		
	    }
	    pTO = triggerList.Next();
	}
    }
    catch ( Exception e )
    {
	_pDBMng->unuseObject(oe.getTabSetId(), oe.getName(), oe.getType());
	throw Exception(EXLOC, Chain("Cannot insert data"), e);
    }
    _pDBMng->unuseObject(oe.getTabSetId(), oe.getName(), oe.getType());
}

void CegoDistManager::insertDistDataTable(CegoTableObject& oe, ListT< ListT<CegoField> >& fva)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Inserting into global table ") + oe.getName());
#endif

    if ( _authEnabled == true && _pDBMng->verifyAccess(oe.getTabSetId(), oe.getName(), oe.getType(), CegoXMLSpace::WRITE, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + oe.getName();
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(oe.getTabSetId());    

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	insertLocalDataTable(oe, fva);
    }
    else
    {	
	Chain tableSet =  _pDBMng->getTabSetName(oe.getTabSetId());

	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);
    
	Chain msg;
	CegoDbHandler::ResultType res = CegoDbHandler::DB_ERROR;
	// TODO
	// res = pSH->reqInsertOp(tableSet, oe.getName(), fvl);
	
	if ( res == CegoDbHandler::DB_OK )
	{
	    _pDBMng->releaseSession(pSH);   
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::insertLocalDataTable(CegoTableObject& oe, ListT< ListT<CegoField> >& fva )
{    
    _pDBMng->useObject(oe.getTabSetId(), oe.getName(), oe.getType(), CegoDatabaseManager::SHARED,  this);
    
    try
    {		
	ListT<CegoTableObject> idxList;
	ListT<CegoBTreeObject> btreeList;
	ListT<CegoKeyObject> keyList;
	ListT<CegoCheckObject> checkList;
	ListT<CegoTriggerObject> triggerList;
	ListT<CegoAliasObject> aliasList;
	unsigned numInvalid;
	
	bool doAppend;
	bool doLogging = true;
	
	getObjectListByTable(oe.getTabSetId(), oe.getName(), idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
	if ( numInvalid > 0 )
	{	
	    // for ongoing transactions, we don't support online index build up
	    if ( getTID(oe.getTabSetId() != 0 ))
	    {
		throw Exception(EXLOC, Chain("Invalid index detected, must be valid for transactions"));
	    }	
	    doAppend = true;
	}
	else
	{
	    doAppend = _doAppend;
	}

	ListT<CegoField> *pFVL = fva.First();
	unsigned numEntry = fva.Size();
	unsigned entryCount=0;
	
	while ( pFVL )
	{
	    entryCount++;
		    
	    CegoTriggerObject *pTO = triggerList.First();
	    while ( pTO )
	    {
		if ( pTO->isOnInsert() && pTO->isBefore() )
		{
		    CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());
		    pTrigger->execute(pFVL);		
		}
		pTO = triggerList.Next();
	    }
	    
	    CegoDataPointer sysEntry;
	    Chain virginIndex;
	    CegoDataPointer dp;

	    // for performance reasons, just for last insert call, we do flush log ( entryCount == numEntry )
	    insertDataTable(oe, *pFVL, idxList, btreeList, keyList, checkList, sysEntry, virginIndex, dp, doLogging, doAppend, true, entryCount == numEntry);
	    
	    pTO = triggerList.First();
	    while ( pTO )
	    {
		if ( pTO->isOnInsert() && pTO->isBefore() == false )
		{
		    CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());
		    pTrigger->execute(pFVL);		
		}
		pTO = triggerList.Next();
	    }

	    pFVL = fva.Next();
	}
    }
    catch ( Exception e )
    {
	_pDBMng->unuseObject(oe.getTabSetId(), oe.getName(), oe.getType());
	throw Exception(EXLOC, Chain("Cannot insert data"), e);
    }
    _pDBMng->unuseObject(oe.getTabSetId(), oe.getName(), oe.getType());
}

unsigned long long CegoDistManager::deleteDistDataTable(CegoTableObject& oe, CegoPredicate* pPred, CegoProcBlock* pBlock)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Deleting from global table ") + oe.getName());
#endif

    if ( _authEnabled == true && _pDBMng->verifyAccess(oe.getTabSetId(), oe.getName(), oe.getType(), CegoXMLSpace::WRITE, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + oe.getName();
	throw Exception(EXLOC, msg); 
    }
    
    Chain hostName = _pDBMng->getPrimary(oe.getTabSetId());

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	return deleteLocalDataTable(oe, pPred, pBlock);
    }
    else
    {
	Chain tableSet =  _pDBMng->getTabSetName(oe.getTabSetId());

	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	// no more supported with XML encoding is deprecated
	// pSH->reqDeleteOp(tableSet, oe.getName(), pPred);
	unsigned long long delCount = pSH->getAffected();
	_pDBMng->releaseSession(pSH);

	return delCount;
    }
}

unsigned long long CegoDistManager::deleteLocalDataTable(CegoTableObject& oe, CegoPredicate* pPred, CegoProcBlock* pBlock)
{
    unsigned long long delCount = 0;

    bool forceTransaction=false;
    if ( getTID(oe.getTabSetId()) == 0 )
    {
	forceTransaction=true;
	beginTransaction(oe.getTabSetId(), true);
    }
    
    _pDBMng->useObject(oe.getTabSetId(), oe.getName(), oe.getType(), CegoDatabaseManager::SHARED, this);
    
    try
    {		
	ListT<CegoTableObject> idxList;
	ListT<CegoBTreeObject> btreeList;
	ListT<CegoKeyObject> keyList;
	ListT<CegoCheckObject> checkList;
	ListT<CegoTriggerObject> triggerList;
	ListT<CegoAliasObject> aliasList;
        unsigned numInvalid;
	
	getObjectListByTable(oe.getTabSetId(), oe.getTabName(), idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
	
	if ( numInvalid > 0 )
	{
	    throw Exception(EXLOC, Chain("Invalid index detected, must be valid for delete opertions"));
	}

	CegoTriggerObject *pTO = triggerList.First();
	while ( pTO )
	{
	    if ( pTO->isOnDelete() && pTO->isBefore() )
	    {
		CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());
		pTrigger->execute();		
	    }
	    pTO = triggerList.Next();
	}
	
	delCount = deleteDataTable(oe, idxList, btreeList, keyList, pPred, pBlock, true);

	pTO = triggerList.First();
	while ( pTO )
	{
	    if ( pTO->isOnDelete() && pTO->isBefore() == false )
	    {
		CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());
		pTrigger->execute();		
	    }
	    pTO = triggerList.Next();
	}	
    }
    catch ( Exception e)
    {	
	_pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);

	if ( forceTransaction )
	{    	    
	    _pDBMng->useObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE, CegoDatabaseManager::SHARED, this);

	    try
	    {
		rollbackTransaction(oe.getTabSetId(), true);
	    }
	    catch ( Exception e )
	    {
		_pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
		throw Exception(EXLOC, Chain("Cannot rollback transaction"), e);
	    }
	    _pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
	}

	throw Exception(EXLOC, Chain("Cannot delete data"), e);	
    }

    _pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
    
    if ( forceTransaction )
    {
	_pDBMng->useObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE, CegoDatabaseManager::EXCLUSIVE_WRITE, this);
	
	try
	{
	    // for the commit, we have to use the table object in exclusive mode
	    commitTransaction(oe.getTabSetId(), true);
	}
	catch ( Exception e )
	{
	    _pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
	    throw Exception(EXLOC, Chain("Cannot commit transaction"), e);
	}

	_pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
    }

    return delCount;
}

unsigned long long CegoDistManager::updateDistDataTable(CegoTableObject& oe,  
							CegoPredicate* pPred, 
							ListT<CegoField>& updSchema, 
							ListT<CegoExpr*>& exprList,
							bool returnOnFirst,
							ListT<CegoField>& returnList, 
							CegoProcBlock* pBlock)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Updating global table ") + oe.getName());
#endif

    if ( _authEnabled == true && _pDBMng->verifyAccess(oe.getTabSetId(), oe.getName(), oe.getType(), CegoXMLSpace::WRITE, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + oe.getName();
	throw Exception(EXLOC, msg); 
    }
    
    Chain hostName = _pDBMng->getPrimary(oe.getTabSetId());

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	return updateLocalDataTable(oe, pPred, updSchema, exprList, returnOnFirst, returnList, pBlock);
    }
    else
    {
	Chain tableSet =  _pDBMng->getTabSetName(oe.getTabSetId());

	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	// no more supported with XML encoding is deprecated
	// pSH->reqUpdateOp(tableSet, oe.getName(), updSchema, exprList, pPred);
	long updCount = pSH->getAffected();
	_pDBMng->releaseSession(pSH);
	
	return updCount;
    }
}

unsigned long long CegoDistManager::updateLocalDataTable(CegoTableObject& oe,  
							 CegoPredicate* pPred, 
							 ListT<CegoField>& updList, 
							 ListT<CegoExpr*>& exprList,
							 bool returnOnFirst,
							 ListT<CegoField>& returnList, 
							 CegoProcBlock* pBlock)
{
    unsigned long long updCount = 0;
    bool forceTransaction=false;
    if ( getTID(oe.getTabSetId()) == 0 )
    {
	forceTransaction=true;
	beginTransaction(oe.getTabSetId(), true);
    }

    _pDBMng->useObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE, CegoDatabaseManager::SHARED, this);
    
    try
    {
	ListT<CegoTableObject> idxList;
	ListT<CegoBTreeObject> btreeList;
	ListT<CegoKeyObject> keyList;
	ListT<CegoCheckObject> checkList;
	ListT<CegoTriggerObject> triggerList;
	ListT<CegoAliasObject> aliasList;
	unsigned numInvalid;
	
	getObjectListByTable(oe.getTabSetId(), oe.getTabName(), idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
	if ( numInvalid > 0 )
	{	    
	    throw Exception(EXLOC, Chain("Invalid index detected, must be valid for update operations"));    
	}

	CegoTriggerObject *pTO = triggerList.First();
	while ( pTO )
	{
	    if ( pTO->isOnUpdate() && pTO->isBefore() )
	    {
		CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());

		ListT<CegoField> tupdList = updList;
		CegoField* pF = tupdList.First();
		CegoExpr** pExpr = exprList.First();
		while ( pF )
		{
		    CegoFieldValue fv = (*pExpr)->evalFieldValue(0, 0);
		    pF->setValue(fv);		    
		    pF = tupdList.Next();
		    pExpr = exprList.Next();
		}
		pTrigger->execute(&tupdList);
		
	    }
	    pTO = triggerList.Next();
	}
	
	updCount = updateDataTable(oe.getTabSetId(), oe.getTabName(), oe.getTabAlias(),
				   idxList, btreeList, keyList, checkList,
				   pPred, updList, exprList, returnOnFirst, returnList, pBlock);

	pTO = triggerList.First();
	while ( pTO )
	{
	    if ( pTO->isOnUpdate() && pTO->isBefore() == false )
	    {
		CegoTrigger *pTrigger = getTrigger(pTO->getTabSetId(), pTO->getName());

		ListT<CegoField> tupdList = updList;
		CegoField* pF = tupdList.First();
		CegoExpr** pExpr = exprList.First();
		while ( pF )
		{
		    CegoFieldValue fv = (*pExpr)->evalFieldValue(0, 0);
		    pF->setValue(fv);		    
		    pF = tupdList.Next();
		    pExpr = exprList.Next();
		}
		pTrigger->execute(&tupdList);
		
	    }
	    pTO = triggerList.Next();
	}
    }
    catch ( Exception e )
    {
	_pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);

	if ( forceTransaction )
	{    	    
	    _pDBMng->useObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE, CegoDatabaseManager::SHARED, this);
   
	    try
	    {
		rollbackTransaction(oe.getTabSetId(), true);
	    }
	    catch ( Exception e )
	    {
		_pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
		throw Exception(EXLOC, Chain("Cannot rollback transaction"), e);
	    }
	}

	_pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
	throw Exception(EXLOC, Chain("Cannot update data"), e);	
    }

    _pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
    
    if ( forceTransaction )
    {
	_pDBMng->useObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE, CegoDatabaseManager::EXCLUSIVE_WRITE, this);
	
	try
	{
	    // for the commit, we have to use the table object in exclusive mode
	    commitTransaction(oe.getTabSetId(), true);
	}
	catch ( Exception e )
	{	    
	    _pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
	    throw Exception(EXLOC, Chain("Cannot commit transaction"), e);
	}
	_pDBMng->unuseObject(oe.getTabSetId(), oe.getTabName(), CegoObject::TABLE);
    }

    return updCount;
}

void CegoDistManager::alterDistDataTable(CegoTableObject& oe, const ListT<CegoAlterDesc>& alterList)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Altering global table ") + oe.getName());
#endif

    if ( _authEnabled == true && _pDBMng->verifyAccess(oe.getTabSetId(), oe.getName(), oe.getType(), CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + oe.getName();
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(oe.getTabSetId());

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	alterDataTableSynced(oe, alterList);
    }
    else
    {
	Chain tableSet =  _pDBMng->getTabSetName(oe.getTabSetId());

	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	// no more supported
	// pSH->reqAlterTableOp(tableSet, oe.getName(), alterList);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::renameDistObject(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type,  const Chain& newObjName)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Renaming global table ") + objName);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, objName, type, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + objName;
	throw Exception(EXLOC, msg); 
    }
    
    Chain hostName = _pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	renameLocalObject(tabSetId, objName, type, newObjName);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	pSH->reqRenameOp(tableSet, objName, type, newObjName);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::renameLocalObject(unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type,  const Chain& newObjName)
{
    if ( objectExists(tabSetId, newObjName, type) )
    {
	Chain msg = Chain("Object ") +  newObjName + Chain(" already exists");
	throw Exception(EXLOC, msg);
    }
    
    _pDBMng->removeObject(tabSetId, objName, type);
    renameObject(tabSetId, objName, type, newObjName);
    _pDBMng->addObject(tabSetId, newObjName, type);
}

void CegoDistManager::createDistView(const Chain& tableSet, const Chain& viewName, const ListT<CegoField>& schema, const Chain& viewText)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global view ") + viewName + Chain(" in tableset ") + tableSet);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, viewName, CegoObject::VIEW, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + viewName;
	throw Exception(EXLOC, msg); 
    }

    Chain hostName =_pDBMng->getPrimary(tabSetId);    

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	createLocalView(tabSetId, viewName, schema, viewText);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	pSH->reqCreateViewOp(tableSet, viewName, schema, viewText);      
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::createLocalView(unsigned tabSetId, const Chain& viewName, const ListT<CegoField>& schema, const Chain& viewText)
{
    CegoViewObject vo( tabSetId, viewName, schema, viewText);

    CegoLogRecord lr;
    lr.setLSN(_pDBMng->nextLSN(tabSetId));
    
    createViewObject(vo);
    
    // create log entry
    lr.setObjectInfo(vo.getName(), vo.getType());
    lr.setAction(CegoLogRecord::LOGREC_CREATE);
 
    char *buf;
    buf = (char*)malloc(vo.getEntrySize());
    vo.encode(buf);
    lr.setData(buf);
    lr.setDataLen(vo.getEntrySize());
    // lr.setTID(0);
    _pDBMng->logIt(vo.getTabSetId(), lr, _pLockHandle);
    free(buf);

    _pDBMng->addObject(tabSetId, viewName, CegoObject::VIEW);
}

void CegoDistManager::createDistTrigger(const Chain& tableSet, const Chain& triggerName,
					bool isBefore, bool isOnInsert, bool isOnUpdate, bool isOnDelete,
					const Chain& tableName, const Chain& triggerText)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global trigger ") + triggerName + Chain(" in tableset ") + tableSet);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, triggerName, CegoObject::TRIGGER, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + triggerName;
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	createLocalTrigger(tabSetId, triggerName, isBefore, isOnInsert, isOnUpdate, isOnDelete, tableName, triggerText);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	// TODO 
	pSH->reqCreateTriggerOp(tableSet, triggerName, tableName, triggerText);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::createLocalTrigger(unsigned tabSetId, const Chain& triggerName,
					 bool isBefore, bool isOnInsert, bool isOnUpdate, bool isOnDelete,
					 const Chain& tableName, const Chain& triggerText)
{
    createTrigger(tabSetId, triggerName, isBefore, isOnInsert, isOnUpdate, isOnDelete, tableName, triggerText);
    
    _pDBMng->addObject(tabSetId, triggerName, CegoObject::TRIGGER);
}

void CegoDistManager::createDistAlias(const Chain& tableSet, const Chain& aliasName,				 
				      const Chain& tableName, const ListT<CegoAttrAlias>& aliasList)
{

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global alias ") + aliasName + Chain(" in tableset ") + tableSet);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, aliasName, CegoObject::ALIAS, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + aliasName;
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	createLocalAlias(tabSetId, aliasName, tableName, aliasList);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	// TODO 
	// pSH->reqCreateAliasOp(tableSet, aliasName, tableName, aliasList);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::createLocalAlias(unsigned tabSetId, const Chain& aliasName,
					 const Chain& tableName, const ListT<CegoAttrAlias>& aliasList)
{
    createAlias(tabSetId, aliasName, tableName, aliasList);    
    _pDBMng->addObject(tabSetId, aliasName, CegoObject::ALIAS);
}

void CegoDistManager::createDistProc(const Chain& tableSet, const Chain& procName, const Chain& procText)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating global procedure ") + procName + Chain(" in tableset ") + tableSet);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    if ( _authEnabled == true && _pDBMng->verifyAccess(tabSetId, procName, CegoObject::PROCEDURE, CegoXMLSpace::MODIFY, _authUser) == false )
    {
	Chain msg = Chain("Access not allowed for object ") + procName;
	throw Exception(EXLOC, msg); 
    }

    Chain hostName = _pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	createLocalProc(tabSetId, procName, procText);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	pSH->reqCreateProcOp(tableSet, procName, procText);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::createLocalProc(unsigned tabSetId, const Chain& procName, const Chain& procText)
{
    CegoProcObject po(tabSetId, procName, procText);

    CegoLogRecord lr;
    lr.setLSN(_pDBMng->nextLSN(tabSetId));
    
    createProcObject(po);

    // create log entry

    lr.setObjectInfo(po.getName(), po.getType());
    lr.setAction(CegoLogRecord::LOGREC_CREATE);
 
    char *buf;
    buf = (char*)malloc(po.getEntrySize());
    po.encode(buf);
    lr.setData(buf);
    lr.setDataLen(po.getEntrySize());
    // lr.setTID(0);
    _pDBMng->logIt(po.getTabSetId(), lr, _pLockHandle);
    free(buf);

    _pDBMng->addObject(tabSetId,procName, CegoObject::PROCEDURE);
}

void CegoDistManager::getDistObjectList(const Chain& tableSet, CegoObject::ObjectType type, ListT<Chain> &objList)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Getting global object list in tableset ") + tableSet);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);    
    Chain hostName = _pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	getLocalObjectList(tabSetId, type, objList);
    }
    else
    {	
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;
    
	res = pSH->reqGetObjectListOp(tabSetId, type);

	if ( res == CegoDbHandler::DB_INFO )
	{
	    pSH->getObjectList(objList);
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

void CegoDistManager::getLocalObjectList(unsigned tabSetId, CegoObject::ObjectType type, ListT<Chain> &objList)
{
    getObjectList(tabSetId, type, objList);
}

void CegoDistManager::getDistObjectByTableList(const Chain& tableSet, 
					       const Chain& tabName,
					       ListT<CegoTableObject>& idxList,
					       ListT<CegoBTreeObject>& btreeList,
					       ListT<CegoKeyObject>& keyList,
					       ListT<CegoCheckObject>& checkList,
					       ListT<CegoTriggerObject>& triggerList,
					       ListT<CegoAliasObject>& aliasList,
					       unsigned& numInvalid)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Getting global object list by table ") + tabName + Chain(" in tableset ") + tableSet);
#endif

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
    Chain hostName = _pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	getObjectListByTable(tabSetId, tabName, idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;

	res = pSH->reqGetObjectByTableListOp(tableSet, tabName);

	if ( res == CegoDbHandler::DB_INFO )
	{
	    // TODO btree retrievel still not implemented
	    pSH->getObjectByTableList(idxList, keyList, checkList);
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}
    }
}

unsigned CegoDistManager::getDistPageCount(const Chain& tableSet, const Chain& tabName, CegoObject::ObjectType type)
{
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);    
    Chain hostName = _pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {
	return getPageCount(tabSetId, tabName, type);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	CegoDbHandler::ResultType res;

	res = pSH->reqGetPageCount(tableSet, tabName, type);

	unsigned pageCount = 0;

	if ( res == CegoDbHandler::DB_INFO )
	{
	    pSH->getPageCount(pageCount);
	    _pDBMng->releaseSession(pSH);	    
	}
	else if ( res == CegoDbHandler::DB_ERROR )
	{
	    Chain msg = pSH->getMsg();
	    _pDBMng->releaseSession(pSH);
	    throw Exception(EXLOC, msg); 
	}

	return pageCount;
    }
}

CegoLockHandler* CegoDistManager::getLockHandle()
{
    return _pLockHandle;
}

void CegoDistManager::syncDistTableSet(const Chain& tableSet, const Chain& msg, const Chain& escCmd, unsigned timeout)
{
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);    
    Chain hostName =_pDBMng->getPrimary(tabSetId);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);

    if ( hostName == dbHost )
    {    
	syncTableSet(tableSet, msg, escCmd, timeout);
    }
    else
    {
	Chain user;
	Chain password;

	getActiveUser(tableSet, user, password);
	
	CegoDistDbHandler *pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);

	pSH->reqSyncOp(tableSet, escCmd, timeout);
	_pDBMng->releaseSession(pSH);
    }
}

void CegoDistManager::enableAuth()
{
    _authEnabled = true;
}

void CegoDistManager::disableAuth()
{
    _authEnabled = false;
}

void CegoDistManager::setActiveUser(const Chain& tableSet, const Chain& user, const Chain& passwd)
{
    _authUser = user;

    if ( _userList.Find(ActiveTSUser(tableSet)) == 0 )
	_userList.Insert( ActiveTSUser(tableSet, user, passwd));
}

const Chain& CegoDistManager::getUser() const
{
    return _authUser;
}

void CegoDistManager::getActiveUser(const Chain& tableSet, Chain& user, Chain& passwd)
{
    ActiveTSUser *pTSA = _userList.Find( ActiveTSUser(tableSet));
    if ( pTSA )
    {
	user = pTSA->getUserName();
	passwd = pTSA->getPasswd();
	return;
    }

    Chain msg = Chain("No active user for tableset <") + tableSet + Chain(">");
    throw Exception(EXLOC, msg);
}

Element* CegoDistManager::verifyTable(const Chain& tableSet, const Chain& tableName)
{   
    _pDBMng->log(_modId, Logger::NOTICE, Chain("Verifying table ") + tableName + Chain(" ..."));

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    Element *pVerification = new Element(XML_VERIFICATION_ELEMENT);
    
    unsigned errorCount = 0;
    
    ListT<CegoTableObject> idxList;
    ListT<CegoBTreeObject> btreeList;
    ListT<CegoKeyObject> keyList;
    ListT<CegoCheckObject> checkList;
    ListT<CegoTriggerObject> triggerList;
    ListT<CegoAliasObject> aliasList;
    unsigned numInvalid;

    getObjectListByTable(tabSetId, tableName, idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
	
    CegoTableObject oe;
    getObject(tabSetId, tableName, CegoObject::TABLE, oe);
    	
    CegoTableCursor tc(this, tabSetId, tableName);
    ListT<CegoField> fl = oe.getSchema();
    CegoDataPointer dp;
    long tabCount = 0;

    if ( tc.getFirst(fl, dp) )
    {
	tabCount++;
	while ( tc.getNext(fl, dp) )
	{
	    tabCount++;
	}
    }

    // verify keys
    bool keyCheck = checkKey(tabSetId, tableName, oe.getSchema(), keyList);
    
    if ( keyCheck == false )
    {
	errorCount++;
	Chain tableStatus = Chain("Key constraint violation");
	Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
	pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
	pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
	pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
	pVerification->addContent(pTableCheck);	    
    }
    
    // verify indexes
    CegoTableObject *pIdx = idxList.First();
    while ( pIdx ) 
    {
	if ( pIdx->isValid() )
	{
	    CegoAVLIndexManager idxMng(this);
	    char c = idxMng.checkIndex(tabSetId, pIdx->getName(), pIdx->getType());
	    
	    if ( c < 0 )
	    {
		errorCount++;
		Chain tableStatus = Chain("Index ") + pIdx->getName() + Chain(" due to height corrupted ");
		Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
		pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
		pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
		pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
		pVerification->addContent(pTableCheck);      
	    }
	    
	    CegoAVLIndexCursor ic(this, tabSetId, pIdx->getName(), pIdx->getType(), 0, false, false);
	    
	    ListT<CegoField> fl = oe.getSchema();
	    CegoDataPointer dp;
	    long idxCount = 0;
	    if ( ic.getFirst(fl, dp) )
	    {
		idxCount++;
		while ( ic.getNext(fl, dp) )
		    idxCount++;
	    }
	    if ( tabCount != idxCount )
	    {
		errorCount++;
		Chain tableStatus = Chain("Index ") + pIdx->getName() + Chain(" row mismatch (") + Chain(tabCount) + Chain("/") + Chain(idxCount) + Chain(")");
		Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
		pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
		pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
		pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
		pVerification->addContent(pTableCheck);
	    }
	}
	else
	{
	    errorCount++;
	    Chain tableStatus = Chain("Index ") + pIdx->getName() + Chain(" not valid");
	    Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
	    pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
	    pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
	    pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
	    pVerification->addContent(pTableCheck);	    
	}
	
	pIdx = idxList.Next();
    }

    // verify btree objects
    CegoBTreeObject *pBTree = btreeList.First();
    while ( pBTree ) 
    {
	if ( pBTree->isValid() )
	{

	    CegoBTreeManager btreeMng(this, pBTree);
	    if ( btreeMng.verifyBTree() == false )
	    {
		errorCount++;
		Chain tableStatus = Chain("Btree ") + pBTree->getName() + Chain(" node corruption ");
		Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
		pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
		pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
		pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
		pVerification->addContent(pTableCheck);		
	    }
	    	  
	    CegoBTreeCursor bc(this, tabSetId, pBTree->getName(), pBTree->getType(), 0, false, false);
	    
	    ListT<CegoField> fl = oe.getSchema();
	    CegoDataPointer dp;
	    long btreeCount = 0;
	    if ( bc.getFirst(fl, dp) )
	    {
		btreeCount++;
		while ( bc.getNext(fl, dp) )
		    btreeCount++;
	    }
	    if ( tabCount != btreeCount )
	    {
		errorCount++;
		Chain tableStatus = Chain("Btree ") + pBTree->getName() + Chain(" row mismatch (") + Chain(tabCount) + Chain("/") + Chain(btreeCount) + Chain(")");
		Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
		pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
		pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
		pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
		pVerification->addContent(pTableCheck);
	    }	    
	}
	else
	{
	    errorCount++;
	    Chain tableStatus = Chain("Btree ") + pIdx->getName() + Chain(" not valid");
	    Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
	    pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
	    pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
	    pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
	    pVerification->addContent(pTableCheck);	    
	}	
	pBTree = btreeList.Next();
    }

    if ( errorCount == 0 )
    {
	Chain tableStatus("ok");
	Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
	pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
	pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
	pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
	pVerification->addContent(pTableCheck);			      
    }
    return pVerification;
}

Element* CegoDistManager::verifyView(const Chain& tableSet, const Chain& viewName)
{
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);    
    Element *pVerification = new Element(XML_VERIFICATION_ELEMENT);
    
    CegoViewObject vo;
    getObject(tabSetId, viewName, CegoObject::VIEW, vo);
    
    Chain loadString = Chain("load ") + vo.getViewStmt();
    
    CegoAction *pPA = getParser();
    pPA->cleanUp();
    pPA->setTableSet(tableSet);
    pPA->setCommandChain(loadString);
    pPA->parse();
    
    Chain viewStatus("ok");	
    Element *pViewCheck = new Element(XML_CHECK_ELEMENT);
    pViewCheck->setAttribute(XML_TYPE_ATTR, Chain("View"));
    pViewCheck->setAttribute(XML_NAME_ATTR, viewName);
    pViewCheck->setAttribute(XML_VALUE_ATTR, viewStatus);
    pVerification->addContent(pViewCheck);			      
    
    return pVerification;
}

Element* CegoDistManager::verifyProcedure(const Chain& tableSet, const Chain& procName)
{
    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);    
    Element *pVerification = new Element(XML_VERIFICATION_ELEMENT);
    
    CegoProcObject po;
    getObject(tabSetId, procName, CegoObject::PROCEDURE, po);
	
    Chain loadString = Chain("load ") + po.getProcText();

    CegoAction *pPA = getParser();
    pPA->cleanUp();
    pPA->setTableSet(tableSet);
    pPA->setCommandChain(loadString);
    pPA->parse();
    
    Chain procStatus("ok");	
    Element *pProcCheck = new Element(XML_CHECK_ELEMENT);
    pProcCheck->setAttribute(XML_TYPE_ATTR, Chain("Procedure"));
    pProcCheck->setAttribute(XML_NAME_ATTR, procName);
    pProcCheck->setAttribute(XML_VALUE_ATTR, procStatus);
    pVerification->addContent(pProcCheck);			      
    
    return pVerification;
}

Element* CegoDistManager::correctTable(const Chain& tableSet, const Chain& tableName)
{   
    _pDBMng->log(_modId, Logger::NOTICE, Chain("Correcting table ") + tableName + Chain(" ..."));

    unsigned tabSetId = _pDBMng->getTabSetId(tableSet);

    Element *pCorrection = new Element(XML_CORRECTION_ELEMENT);
    
    unsigned errorCount = 0;
    
    ListT<CegoTableObject> idxList;
    ListT<CegoBTreeObject> btreeList;
    ListT<CegoKeyObject> keyList;
    ListT<CegoCheckObject> checkList;
    ListT<CegoTriggerObject> triggerList;
    ListT<CegoAliasObject> aliasList;
    unsigned numInvalid;

    getObjectListByTable(tabSetId, tableName, idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
    
    // correct indexes
    CegoTableObject *pIdx = idxList.First();
    while ( pIdx ) 
    {
	if ( pIdx->isValid() == false )
	{
	    errorCount++;

	    dropIndex(tabSetId, pIdx->getName());
	    bool isCached=false;
	    createIndexTable(tabSetId, pIdx->getName(), pIdx->getTabName(), pIdx->getSchema(), pIdx->getType(), isCached);

	    Chain tableStatus = Chain("Index ") + pIdx->getName() + Chain(" was corrected");
	    Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
	    pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
	    pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
	    pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
	    pCorrection->addContent(pTableCheck);	    
	}
	
	pIdx = idxList.Next();
    }
    
    // correct btrees
    CegoBTreeObject *pBTree = btreeList.First();
    while ( pBTree ) 
    {	
	if ( pBTree->isValid() == false )
	{
	    errorCount++;
	    
	    dropBTree(tabSetId, pBTree->getName());
	    bool doSync=false;
	    bool isCached=false;
	    createBTree(tabSetId, pBTree->getName(), pBTree->getTabName(), pBTree->getSchema(), pBTree->getType(), doSync, isCached);

	    Chain tableStatus = Chain("Btree ") + pBTree->getName() + Chain(" was corrected");
	    Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
	    pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
	    pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
	    pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
	    pCorrection->addContent(pTableCheck);	       
	}
	pBTree = btreeList.Next();
    }
    
    if ( errorCount == 0 )
    {
	Chain tableStatus("ok");
	Element *pTableCheck = new Element(XML_CHECK_ELEMENT);
	pTableCheck->setAttribute(XML_TYPE_ATTR, Chain("Table"));
	pTableCheck->setAttribute(XML_NAME_ATTR, tableName);
	pTableCheck->setAttribute(XML_VALUE_ATTR, tableStatus);
	pCorrection->addContent(pTableCheck);			      
    }
    
    return pCorrection;
}

bool CegoDistManager::checkKey(unsigned tabSetId, const Chain& tableName, const ListT<CegoField>& schema, const ListT<CegoKeyObject>& keyList)
{    
    CegoKeyObject* pKey = keyList.First();
    while ( pKey )
    {
        if ( (Chain)pKey->getTabName() == (Chain)tableName )
        {
            CegoTableCursor ktc(this, tabSetId, pKey->getTabName());

	    CegoTableObject oe;
	    getObject(tabSetId, pKey->getTabName(), CegoObject::TABLE, oe);
           
            ListT<CegoField> kfl = oe.getSchema();

            CegoDataPointer dp;
            bool moreTuple = ktc.getFirst(kfl, dp);
	    
	    while ( moreTuple )
	    {
		ListT<CegoField> rfl = pKey->getRefSchema();
		CegoField *pKF = kfl.First();
		CegoField* pRF = rfl.First();
		while ( pKF && pRF)
		{
		    pRF->setValue(pKF->getValue());		    
		    pKF = kfl.Next();		    
		    pRF = rfl.Next();
		}
		
		CegoAttrCond ac;
		pRF = rfl.First();
		while ( pRF )
		{
		    // cout << "Searching for " << pRF->getTableName() << "." << pRF->getAttrName() << " = " << pRF->getValue() << endl;
		    ac.add(CegoAttrComp(pRF->getTableName(), pRF->getAttrName(), EQUAL, pRF->getValue()));		    
		    pRF = rfl.Next();
		}
		
		CegoTableCursor rtc(this, tabSetId, pKey->getRefTable());
		// no need for indexMatch value : CegoAttrCond::IndexMatch indexMatch = rtc.setup(ac);
		rtc.setup(ac);
		
		if (  rtc.getFirst(rfl, dp) == false )
		    return false;
		
		moreTuple = ktc.getNext(kfl, dp);		
            }
	}
	pKey = keyList.Next();
    }
    return true;    
}
		
void CegoDistManager::getObjectDesc(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type, 
		    ListT<CegoField>& schema, ListT< ListT<CegoFieldValue> > &fa)
{
    switch ( type )
    {
    case CegoObject::SYSTEM:
    case CegoObject::TABLE:
    {
	CegoTableObject to;

	getDistObject(tableSet, objName, type, to);
	
	unsigned maxAttrLen=10;
	CegoField *pF = to.getSchema().First();
	while ( pF )
	{
	    if ( (unsigned)pF->getAttrName().length() > maxAttrLen )
		maxAttrLen = pF->getAttrName().length();
	    pF = to.getSchema().Next();
	}
	    
	schema.Insert(CegoField(Chain("TABLEDESC"), Chain("TABLEDESC"), Chain("ATTR"), VARCHAR_TYPE, maxAttrLen));
	schema.Insert(CegoField(Chain("TABLEDESC"), Chain("TABLEDESC"), Chain("TYPE"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("TABLEDESC"), Chain("TABLEDESC"), Chain("LENGTH"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("TABLEDESC"), Chain("TABLEDESC"), Chain("DIM"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("TABLEDESC"), Chain("TABLEDESC"), Chain("DEFAULT"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("TABLEDESC"), Chain("TABLEDESC"), Chain("NULLABLE"), VARCHAR_TYPE, 10));

	pF = to.getSchema().First();
	while ( pF )
	{
	    ListT<CegoFieldValue> fvl;	    
	   
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pF->getAttrName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, CegoTypeConverter::getTypeString(pF->getType())));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain(pF->getLength())));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain(pF->getDim())));
	    
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pF->getValue().valAsChain(false)));
	    
	    if ( pF->isNullable() )
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain("yes")));
	    else
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain("no")));
	    
	    fa.Insert(fvl);
	    
	    pF = to.getSchema().Next();
	}
	break;
    }
    case CegoObject::AVLTREE:
    case CegoObject::UAVLTREE:
    case CegoObject::PAVLTREE:
    {
	CegoTableObject io;
	getDistObject(tableSet, objName, type, io);
	
	unsigned maxAttrLen=10;
	CegoField *pF = io.getSchema().First();
	while ( pF )
	{
	    if ( (unsigned)pF->getAttrName().length() > maxAttrLen )
		maxAttrLen = pF->getAttrName().length();
	    pF = io.getSchema().Next();
	}

	unsigned maxTabLen=10;	
	if ( (unsigned)io.getTabName().length() > maxTabLen )
	    maxTabLen = io.getTabName().length();
	    
	schema.Insert(CegoField(Chain("INDEXDESC"), Chain("TABLEDESC"), Chain("ATTR"), VARCHAR_TYPE, maxAttrLen));
	schema.Insert(CegoField(Chain("INDEXDESC"), Chain("TABLEDESC"), Chain("TABLE"), VARCHAR_TYPE, maxTabLen));
	schema.Insert(CegoField(Chain("INDEXDESC"), Chain("TABLEDESC"), Chain("TYPE"), VARCHAR_TYPE, 10));
	
	pF = io.getSchema().First();
	while ( pF )
	{
	    ListT<CegoFieldValue> fvl;	    
	   
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pF->getAttrName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, io.getTabName()));
	    
	    if ( io.getType() == CegoObject::AVLTREE) 
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, XML_INDEX_VALUE));
	    else if ( io.getType() == CegoObject::PAVLTREE)
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, XML_PINDEX_VALUE));
	    else if ( io.getType() == CegoObject::UAVLTREE)
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, XML_UINDEX_VALUE));
	    
	    fa.Insert(fvl);
	    
	    pF = io.getSchema().Next();
	}
	break;
    }
    case CegoObject::BTREE:
    case CegoObject::UBTREE:
    case CegoObject::PBTREE:
    {
	CegoBTreeObject bto;
	getDistObject(tableSet, objName, type, bto);
	
	unsigned maxAttrLen=10;
	CegoField *pF = bto.getSchema().First();
	while ( pF )
	{
	    if ( (unsigned)pF->getAttrName().length() > maxAttrLen )
		maxAttrLen = pF->getAttrName().length();
	    pF = bto.getSchema().Next();
	}

	unsigned maxTabLen=10;	
	if ( (unsigned)bto.getTabName().length() > maxTabLen )
	    maxTabLen = bto.getTabName().length();
	    
	schema.Insert(CegoField(Chain("BTREEDESC"), Chain("TABLEDESC"), Chain("ATTR"), VARCHAR_TYPE, maxAttrLen));
	schema.Insert(CegoField(Chain("BTREEDESC"), Chain("TABLEDESC"), Chain("TABLE"), VARCHAR_TYPE, maxTabLen));
	schema.Insert(CegoField(Chain("BTREEDESC"), Chain("TABLEDESC"), Chain("TYPE"), VARCHAR_TYPE, 10));
	
	pF = bto.getSchema().First();
	while ( pF )
	{
	    ListT<CegoFieldValue> fvl;
	   
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pF->getAttrName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, bto.getTabName()));
	    
	    if ( bto.getType() == CegoObject::BTREE) 
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, XML_BTREE_VALUE));
	    else if ( bto.getType() == CegoObject::PBTREE)
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, XML_PBTREE_VALUE));
	    else if ( bto.getType() == CegoObject::UBTREE)
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, XML_UBTREE_VALUE));
	    
	    fa.Insert(fvl);
	    
	    pF = bto.getSchema().Next();
	}
	break;
    }
    case CegoObject::VIEW:
    {
	CegoViewObject vo;
	getDistObject(tableSet, objName, type, vo);

	unsigned maxAttrLen=10;
	CegoField *pF = vo.getSchema().First();
	while ( pF )
	{
	    if ( (unsigned)pF->getAttrName().length() > maxAttrLen )
		maxAttrLen = pF->getAttrName().length();
	    pF = vo.getSchema().Next();
	}

	schema.Insert(CegoField(Chain("VIEWDESC"), Chain("VIEWDESC"), Chain("ATTR"), VARCHAR_TYPE, maxAttrLen));
	schema.Insert(CegoField(Chain("VIEWDESC"), Chain("VIEWDESC"), Chain("TYPE"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("VIEWDESC"), Chain("VIEWDESC"), Chain("LENGTH"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("VIEWDESC"), Chain("VIEWDESC"), Chain("DIM"), VARCHAR_TYPE, 10));

	pF = vo.getSchema().First();
	while ( pF )
	{
	    ListT<CegoFieldValue> fvl;
	    
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pF->getAttrName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, CegoTypeConverter::getTypeString(pF->getType())));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain(pF->getLength())));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain(pF->getDim())));
	    
	    fa.Insert(fvl);

	    pF = vo.getSchema().Next();
	}
	break;
    }
    case CegoObject::PROCEDURE:
    {
	unsigned tabSetId = getDBMng()->getTabSetId(tableSet);

	if ( checkCompProcedure(tabSetId, objName) == false )
	    reloadProcedure( tabSetId, objName );

	CegoProcedure* pProc = getCompProcedure(tabSetId, objName);

	ListT<CegoProcVar> argList;
	pProc->getArgList(argList);

	unsigned maxAttrLen=10;
	CegoProcVar *pArg = argList.First();
	while ( pArg )
	{
	    if ( (unsigned)pArg->getName().length() > maxAttrLen )
		maxAttrLen = pArg->getName().length();
	    pArg = argList.Next();
	}

	schema.Insert(CegoField(Chain("PROCDESC"), Chain("PROCDESC"), Chain("ATTR"), VARCHAR_TYPE, maxAttrLen));
	schema.Insert(CegoField(Chain("PROCDESC"), Chain("PROCDESC"), Chain("TYPE"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("PROCDESC"), Chain("PROCDESC"), Chain("LENGTH"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("PROCDESC"), Chain("PROCDESC"), Chain("DIM"), VARCHAR_TYPE, 10));
	schema.Insert(CegoField(Chain("PROCDESC"), Chain("PROCDESC"), Chain("INOUT"), VARCHAR_TYPE, 10));
	
	pArg = argList.First();
	while ( pArg )
	{
	    ListT<CegoFieldValue> fvl;	    

	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pArg->getName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, CegoTypeConverter::getTypeString(pArg->getType())));	  
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain(pArg->getLength())));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain(pArg->getDim())));
	    		       
	    if ( pArg->getVarType() == CegoProcVar::INVAR )
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain("in")));
	    else if ( pArg->getVarType() == CegoProcVar::OUTVAR )
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain("out")));
	    else
		fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain("n/a")));

	    fa.Insert(fvl);

	    pArg = argList.Next();
	}
	break;
    }
    case CegoObject::FKEY:
    {	    
	CegoKeyObject ko;
	getDistObject(tableSet, objName, type, ko);
	
	unsigned maxTabLen=10;
	if ( (unsigned)objName.length() > maxTabLen )
	    maxTabLen = objName.length();
	
	unsigned maxAttrLen=10;
	CegoField *pK = ko.getKeySchema().First();
	while ( pK )
	{
	    if ( (unsigned)pK->getAttrName().length() > maxAttrLen )
		maxAttrLen = pK->getAttrName().length();
	    
	    pK = ko.getKeySchema().Next();
	}
	CegoField *pR = ko.getRefSchema().First();
	while ( pR )
	{
	    if ( (unsigned)pR->getAttrName().length() > maxAttrLen )
		maxAttrLen = pR->getAttrName().length();
	    pR = ko.getRefSchema().Next();
	}	
	
	schema.Insert(CegoField(Chain("KEYDESC"), Chain("KEYDESC"), Chain("TABLE"), VARCHAR_TYPE, maxTabLen));
	schema.Insert(CegoField(Chain("KEYDESC"), Chain("KEYDESC"), Chain("ATTR"), VARCHAR_TYPE, maxAttrLen));
	schema.Insert(CegoField(Chain("KEYDESC"), Chain("KEYDESC"), Chain("TYPE"), VARCHAR_TYPE, 10));

	pK = ko.getKeySchema().First();
	while ( pK )
	{
	    ListT<CegoFieldValue> fvl;	    
	    
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, ko.getTabName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pK->getAttrName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain("key")));
	    
	    fa.Insert(fvl);

	    pK = ko.getKeySchema().Next();
	}
	pR = ko.getRefSchema().First();
	while ( pR )
	{
	    ListT<CegoFieldValue> fvl;	    
	    
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, ko.getRefTable()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pR->getAttrName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, Chain("reference")));
	    
	    fa.Insert(fvl);

	    pR = ko.getRefSchema().Next();
	}	
	break;
    }
    case CegoObject::CHECK:
    {
	CegoCheckObject co;
	getDistObject(tableSet, objName, type, co);

	unsigned tabSetId = _pDBMng->getTabSetId(tableSet);
	    
	unsigned maxTabLen=10;
	if ( (unsigned)objName.length() > maxTabLen )
	    maxTabLen = objName.length();
	
	unsigned maxCheckLen=30;
	schema.Insert(CegoField(Chain("CHECKDESC"), Chain("CHECKDESC"), Chain("TABLE"), VARCHAR_TYPE, maxTabLen));
	schema.Insert(CegoField(Chain("CHECKDESC"), Chain("CHECKDESC"), Chain("CONDITION"), VARCHAR_TYPE, maxCheckLen));

	ListT<CegoFieldValue> fvl;
	
	fvl.Insert(CegoFieldValue(VARCHAR_TYPE, co.getTabName()));
	fvl.Insert(CegoFieldValue(VARCHAR_TYPE, co.getPredDesc()->toChain(tabSetId)));

	fa.Insert(fvl);

	break;
    }
    case CegoObject::TRIGGER:
    {
	CegoTriggerObject to;
	getDistObject(tableSet, objName, type, to);
	
	unsigned maxTabLen=10;
	if ( (unsigned)objName.length() > maxTabLen )
	    maxTabLen = objName.length();
	
	unsigned maxCheckLen=30;
	schema.Insert(CegoField(Chain("TRIGGERDESC"), Chain("TRIGGERDESC"), Chain("TABLE"), VARCHAR_TYPE, maxTabLen));
	schema.Insert(CegoField(Chain("TRIGGERDESC"), Chain("TRIGGERDESC"), Chain("TRIGGERTEXT"), VARCHAR_TYPE, maxCheckLen));

	ListT<CegoFieldValue> fvl;
	
	fvl.Insert(CegoFieldValue(VARCHAR_TYPE, to.getTabName()));
	fvl.Insert(CegoFieldValue(VARCHAR_TYPE, to.getTriggerText()));

	fa.Insert(fvl);

	break;
    }
    case CegoObject::ALIAS:
    {
	CegoAliasObject ao;
	getDistObject(tableSet, objName, type, ao);
	
	unsigned maxTabLen=10;
	if ( (unsigned)objName.length() > maxTabLen )
	    maxTabLen = objName.length();
	
	unsigned maxAliasLen=30;
	schema.Insert(CegoField(Chain("ALIASDESC"), Chain("ALIASDESC"), Chain("TABLE"), VARCHAR_TYPE, maxTabLen));
	schema.Insert(CegoField(Chain("ALIASDESC"), Chain("ALIASDESC"), Chain("ATTRIBUTE"), VARCHAR_TYPE, maxAliasLen));
	schema.Insert(CegoField(Chain("ALIASDESC"), Chain("ALIASDESC"), Chain("ALIAS"), VARCHAR_TYPE, maxAliasLen));

	CegoAttrAlias *pAlias = ao.getAliasList().First();
	while ( pAlias )
	{
	    ListT<CegoFieldValue> fvl;
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, ao.getTabName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pAlias->getAttrName()));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, pAlias->getAliasName()));
	    fa.Insert(fvl);
	    pAlias = ao.getAliasList().Next();
	}
		       
	break;
    }
    case CegoObject::RBSEG:
    case CegoObject::JOIN:
    case CegoObject::UNDEFINED:	
    {
	throw Exception(EXLOC, Chain("Cannot get description for object type"));
    }
    }
}

CegoView* CegoDistManager::getView(unsigned tabSetId, const Chain& viewName)
{
    if ( checkCompView(tabSetId, viewName) == false )
	reloadView( tabSetId, viewName );
    return getCompView(tabSetId, viewName );
}


CegoProcedure* CegoDistManager::getProcedure(unsigned tabSetId, const Chain& procName)
{
    if ( checkCompProcedure(tabSetId, procName) == false )
	reloadProcedure( tabSetId, procName );
    return getCompProcedure(tabSetId, procName );
}

CegoAction* CegoDistManager::getParser()
{
    if ( _pPA == 0 )
    {
	_pPA = new CegoAction(this);
	_pPA->setGraceMode(_isGrace);
    }
	
    return _pPA;
}

void CegoDistManager::reloadView(unsigned tabSetId, const Chain& viewName)
{
    CegoViewObject vo;
    getObject(tabSetId, viewName, CegoObject::VIEW, vo);

    Chain loadString = Chain("load ") + vo.getViewStmt();

    // since this message is produced for each database thread, we avoid this
    // _pDBMng->log(_modId, Logger::NOTICE, Chain("View ") + viewName + Chain(" is reloaded"));

    // since views can be nested, we have to use dedicated parser here
    // in this phase by using view objects ( instead of creating it),
    // no grace mode is allowed and treated
    CegoAction *pPA = new CegoAction(this);
	
    try
    {	
	Chain tableSet = _pDBMng->getTabSetName(tabSetId);
	
	pPA->setTableSet(tableSet);
	pPA->setCommandChain(loadString);
	pPA->parse();	
	CegoSelect* pSelect = pPA->getSelect(true);
       
	CegoView *pView = new CegoView(viewName, pSelect);

	pSelect->setTabSetId(tabSetId);

	pSelect->prepare();
	
	ListT<CegoField> schema;
	pSelect->getSchema(schema);

	addCompView(tabSetId, pView);

	// if schema is zero, recompile is required
	if ( vo.getSchema().Size() == 0 )
	{
	    _pDBMng->log(_modId, Logger::NOTICE, Chain("View ") + viewName + Chain(" detected as invalid"));

	    dropView(tabSetId, viewName);
	    Chain viewStmt = Chain("view ") + viewName + Chain(" as\n") + pSelect->toChain(tabSetId) + Chain(";");
	    createLocalView( tabSetId, viewName, schema, viewStmt);
	    
	    _pDBMng->log(_modId, Logger::NOTICE, Chain("View ") + viewName + Chain(" validated sucessful"));
	}	
    }
    catch ( Exception e )
    {
	if ( pPA )
	    delete pPA;
	
	Chain msg = Chain("Cannot load view ") + viewName; 
	throw Exception(EXLOC, msg, e);
    }

    delete pPA;
}

void CegoDistManager::reloadProcedure(unsigned tabSetId, const Chain& procName)
{    
    CegoProcObject po;
    getObject(tabSetId, procName, CegoObject::PROCEDURE, po);
    
    Chain loadString = Chain("load ") + po.getProcText();

    // cout << "Load String = " << loadString << endl;
    // since this message is produced for each database thread, we avoid this
    // _pDBMng->log(_modId, Logger::NOTICE, Chain("Procedure ") + procName + Chain(" is reloaded"));        

    // since procedures can be nested, we have to use dedicated parser here
    CegoAction *pPA = new CegoAction(this);
    
    try 
    {
	Chain tableSet = _pDBMng->getTabSetName(tabSetId);
	pPA->setTableSet(tableSet);
	pPA->setCommandChain(loadString);
	pPA->parse();
	
	CegoProcedure* pProc = pPA->getProcedure(true);

	pProc->enableProcCache(_pDBMng->getProcCacheEnabled(tabSetId));
	
	addCompProcedure(tabSetId, pProc);
    }
    catch ( Exception e )
    {
	if ( pPA )
	    delete pPA;
	
	Chain msg = Chain("Cannot load procedure ") + procName; 
	throw Exception(EXLOC, msg, e);
    }

    delete pPA;
}


CegoTrigger* CegoDistManager::getTrigger(unsigned tabSetId, const Chain& triggerName)
{
    if ( checkCompTrigger(tabSetId, triggerName) == false )
	reloadTrigger( tabSetId, triggerName );
    
    return getCompTrigger(tabSetId, triggerName );
}

void CegoDistManager::reloadTrigger(unsigned tabSetId, const Chain& triggerName)
{    
    CegoTriggerObject to;
    getObject(tabSetId, triggerName, CegoObject::TRIGGER, to);
    
    Chain loadString = Chain("load ") + to.toChain() + Chain(";");

    // since this message is produced for each database thread, we avoid this
    // _pDBMng->log(_modId, Logger::NOTICE, Chain("Trigger ") + triggerName + Chain(" is reloaded"));        

    CegoAction *pPA = new CegoAction(this);
    
    try 
    {
	Chain tableSet = _pDBMng->getTabSetName(tabSetId);
	pPA->setTableSet(tableSet);
	pPA->setCommandChain(loadString);
	pPA->parse();
	
	CegoTrigger* pTrigger = pPA->getTrigger(true);
	
	addCompTrigger(tabSetId, pTrigger);
    }
    catch ( Exception e )
    {
	if ( pPA )
	    delete pPA;
	
	Chain msg = Chain("Cannot load trigger ") + triggerName; 
	throw Exception(EXLOC, msg, e);
    }

    delete pPA;
}

void CegoDistManager::setGraceMode(bool isGrace)
{
    _isGrace = isGrace;
}

bool CegoDistManager::isGraceMode() const
{
    return _isGrace;
}

CegoReplacer* CegoDistManager::getReplacer(const Chain& pattern, const Chain& replace)
{
    CegoReplacer *pRep = _pRepHash->Find(CegoReplacer(pattern, replace));

    if ( pRep == 0 )
    {
	CegoReplacer rep(pattern, replace);
	bool wasInserted = _pRepHash->Insert(rep);
	if ( wasInserted )
	{
	    return _pRepHash->Find(CegoReplacer(pattern, replace));
	}
	else
	{
	    throw Exception(EXLOC, "Replace Hash Error");
	}
    }
    
    return pRep;
}

