///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoViewObject.cc 
// -----------------
// Cego view object entry implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoViewObject
// 
// Description: Cego view object structure
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// base includes
#include <lfcbase/Tokenizer.h>

// cego includes
#include "CegoViewObject.h"
#include "CegoXMLdef.h"
#include "CegoTypeConverter.h"

#include <string.h>
#include <stdlib.h>

CegoViewObject::CegoViewObject()
{
    _subCOList.Insert(this);
}

CegoViewObject::CegoViewObject(const CegoViewObject& vo) : CegoContentObject(vo)
{
    _viewStmt = vo._viewStmt;
    _subCOList.Insert(this);
}

CegoViewObject::CegoViewObject(const Chain& viewName, unsigned tabSetId) : CegoContentObject(tabSetId, CegoObject::VIEW, viewName)
{
    _subCOList.Insert(this);
}

CegoViewObject::CegoViewObject(unsigned tabSetId, const Chain& viewName, const ListT<CegoField>& schema, const Chain& viewStmt) : CegoContentObject(tabSetId, CegoObject::VIEW, viewName, viewName, schema)
{
    _viewStmt = viewStmt;    
    _subCOList.Insert(this);
}

CegoViewObject::~CegoViewObject()
{   
}

void CegoViewObject::setSchema(const ListT<CegoField>& schema)
{
    _schema = schema;
}

unsigned CegoViewObject::getEntrySize() const
{
    unsigned entrySize = CegoContentObject::getBaseContentSize();
    entrySize += _viewStmt.length() + 1;
    return entrySize;
}

void CegoViewObject::encode(char *buf) const
{
    char* bufPtr = buf;
    unsigned entrySize = getEntrySize();
    CegoContentObject::encodeBaseContent(bufPtr, entrySize);
    bufPtr += CegoContentObject::getBaseContentSize();
    memcpy (bufPtr, (char*)_viewStmt , _viewStmt.length() + 1);
}

void CegoViewObject::decode(char *buf)
{
    char* bufPtr = buf;
    unsigned size;
    CegoContentObject::decodeBaseContent(bufPtr, size);
    bufPtr += CegoContentObject::getBaseContentSize();
    _viewStmt = Chain(bufPtr);
}

void CegoViewObject::setViewStmt(const Chain& viewStmt)
{
    _viewStmt = viewStmt;
}

const Chain& CegoViewObject::getViewStmt() const
{
    return _viewStmt;
}

CegoViewObject& CegoViewObject::operator = ( const CegoViewObject& vo)
{
    CegoContentObject::operator=(vo);
    _viewStmt = vo._viewStmt;
    return (*this);
}

bool CegoViewObject::operator == ( const CegoViewObject& vo)
{
    return CegoContentObject::operator==(vo);
}

Chain CegoViewObject::getId(CegoProcBlock *pBlock) const
{
    // since it can not be ad hoc decided, if the tuple result of the view objects has been changed,
    // we have to throw MOD_QUERY_ID
    throw Exception(EXLOC, Chain(MOD_QUERY_ID));
}

Chain CegoViewObject::toChain(unsigned defTabSetId) const
{
    if ( ( defTabSetId == getTabSetId() && defTabSetId != 0 ) || defTabSetId == 0 )
	return getTabName() + Chain(" ") + getName();
    else
	return getTabName() + Chain("@") + getTableSet() + Chain(" ") + getName();
}

Chain CegoViewObject::getInfo() const
{
    Chain s;
    s = Chain("Stmt=")+ _viewStmt + Chain("\n");
    return s;
}

CegoContentObject* CegoViewObject::clone(bool isAttrRef)
{
    CegoViewObject *pClone = new CegoViewObject(*this);
    pClone->setViewStmt(_viewStmt);
    return (pClone);
}
