#ifndef _CEGODATABASEMANAGER_H_INCLUDED_
#define _CEGODATABASEMANAGER_H_INCLUDED_
///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoDatabaseManager.h
// ---------------------
// Cego database manager
//     
// Design and Implementation by Bjoern Lemke
//
// (C)opyright 2000-2025 Bjoern Lemke
//
// INTERFACE MODULE
//
// Class: CegoDatabaseManager 
// 
// Description: General database manager
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/ListT.h>
#include <lfcbase/Chain.h>

// CEGO INCLUDES
#include "CegoBufferPool.h"
#include "CegoDistDbHandler.h"
#include "CegoQueryCache.h"
#include "CegoTableCache.h"
#include "CegoDefs.h"

class CegoDatabaseManager : public CegoBufferPool {
    
 public:

    enum RecoveryMode { REQOFF, OFF, ON};

    enum ObjectUseMode { SHARED, EXCLUSIVE_WRITE };

    CegoDatabaseManager(const Chain& xmlDef, const Chain& lckFileName, const Chain& lockExpire, const Chain& logFile, const Chain& progName, CegoDbHandler::ProtocolType protType = CegoDbHandler::SERIAL);
    ~CegoDatabaseManager();

    void allocateQueryCache(const Chain& tableSet);
    void allocateTableCache(const Chain& tableSet);
    void releaseQueryCache(const Chain& tableSet);
    void releaseTableCache(const Chain& tableSet);

    CegoQueryCache* getQueryCache(const Chain& tableSet) const;
    CegoQueryCache* getQueryCache(unsigned tabSetId) const;
    CegoTableCache* getTableCache(const Chain& tableSet) const;
    CegoTableCache* getTableCache(unsigned tabSetId) const;

    void cleanCache(unsigned tabSetId, CegoObject::ObjectType objType, const Chain& objName);
    
    void beat();

    void checkTableSetRunState(unsigned tabSetId);
    
    void startRecovery(const Chain& tableSet);

    void startCopy(const Chain& tableSet, 
		   const Chain& targetHost, 
		   const Chain& mediatorHost, 
		   const Chain& user, 
		   const Chain& passwd,
		   const Chain& msg);

    RecoveryMode getRecoveryMode(unsigned tabSetId);
    void setRecoveryMode(unsigned tabSetId, RecoveryMode m);

    void setAllRecoveryOff();

    bool nextRecovery(Chain& tableSet);
    bool nextCopy(unsigned &id, Chain& tableSet, Chain& targetHost, Chain& mediatorHost, Chain& user, Chain& passwd);
    void setCopyStatus(unsigned id, const Chain& msg);

    void addObject(unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type);

    void useObject(unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type, ObjectUseMode mode, CegoTableManager *pTabMng);

    void unuseObject(unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type, unsigned long long tid = 0 );
    void removeObject(unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type);
    void removeAllObjects(unsigned tabSetId);
    
    bool objectExists(unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type);
    
    void printObjectList();

    void setThreadInfo(unsigned numDbThread, unsigned numAdmThread, unsigned numLogThread);
    void getThreadInfo(unsigned& numDbThread, unsigned& numAdmThread, unsigned& numLogThread,
		       unsigned& activeDbThread, unsigned& activeAdmThread, unsigned& activeLogThread);
    
    void increaseActiveAdmThread();
    void decreaseActiveAdmThread();

    void increaseActiveDbThread();
    void decreaseActiveDbThread();

    void increaseActiveLogThread();
    void decreaseActiveLogThread();

    void getDBMLockStat(Chain& lockName, unsigned& lockCount, unsigned long long &numRdLock, unsigned long long &numWrLock, unsigned long long &sumRdDelay, unsigned long long &sumWrDelay);

    CegoDistDbHandler* allocateSession(const Chain& hostName, const Chain& tableSet, 
				       const Chain& userName, const Chain& password);
    void releaseSession(CegoDistDbHandler* pSH);
    void cleanSession(unsigned lifetime);
    
    Element* getSessionInfo(unsigned lifetime);
    Element* getCopyInfo();

    void configureLogger();

    bool isLoggerConfigured();

    bool verifyJDBC(const Chain& user);
    bool verifyAccess(const unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type, CegoXMLSpace::AccessMode mode, const Chain& user);

    void initLogFiles(const Chain& tableSet, bool overwrite);
    void releaseLogFiles(const Chain& tableSet, bool waitForArchive);

    CegoDbHandler::ProtocolType getProtType() const;

    bool getQCLockStat(unsigned tabSetId, unsigned& lockCount, unsigned long long &numRdLock, unsigned long long &numWrLock, unsigned long long &sumRdDelay, unsigned long long &sumWrDelay);

    void getAggQCLockStat(Chain& lockGroup, unsigned& numLock, unsigned& lockCount, unsigned long long &numRdLock, unsigned long long &numWrLock, unsigned long long &sumRdDelay, unsigned long long &sumWrDelay);

    bool getTCLockStat(unsigned tabSetId, unsigned& lockCount, unsigned long long &numRdLock, unsigned long long &numWrLock, unsigned long long &sumRdDelay, unsigned long long &sumWrDelay);

    void getAggTCLockStat(Chain& lockGroup, unsigned& numLock, unsigned& lockCount, unsigned long long &numRdLock, unsigned long long &numWrLock, unsigned long long &sumRdDelay, unsigned long long &sumWrDelay);
    
private:

    CegoDistDbHandler* createSession(const Chain& hostName, const Chain& tableSet,
				     const Chain& userName, const Chain& password);

    void closeSession(CegoDistDbHandler* pSH);

    void PR();
    void PW();
    void V();

    void configureLogger(Logger::LogLevel level);
	
    class ObjectRecord {
	
    public:
	
	ObjectRecord();
	ObjectRecord(const unsigned tabSetId, const Chain& objName, CegoObject::ObjectType type);

	~ObjectRecord();

	const unsigned getTabSetId() const;

	const Chain& getName() const;
	const CegoObject::ObjectType getType() const;

	void incUsed();
	void decUsed();
	unsigned numUsed() const;

	void setMode(ObjectUseMode mode); 
	ObjectUseMode getMode() const;

	void setTid(unsigned long long tid);	
	unsigned long long getTid() const;

	ObjectRecord& operator = ( const ObjectRecord& t);
        bool operator == ( const ObjectRecord& t);
	
	friend ostream& operator << (ostream& s, const ObjectRecord& t)
	{
	    s << "(" << t._objName << "," << t._tabSetId << ")";
	    return s;
	}

    private:

	Chain _objName;
	CegoObject::ObjectType _type;
	unsigned _tabSetId;
	unsigned _numUsed;
	ObjectUseMode _mode;
	unsigned long long _tid;
    };

    class CopyRecord {
	
    public:
	
	CopyRecord();
	CopyRecord(const Chain& tableSet, const Chain& targetHost, const Chain& mediatorHost, const Chain& user, const Chain& passwd, const Chain& msg);
	~CopyRecord();

	unsigned getId() const;
	void setId(unsigned id);

	const Chain& getTableSet() const;
	const Chain& getTargetHost() const;
	const Chain& getMediatorHost() const;
	
	const Chain& getUser() const;
	const Chain& getPasswd() const;
	
	void setMsg(const Chain& msg);
	const Chain& getMsg() const;

	CopyRecord& operator = ( const CopyRecord& cr);
        bool operator == ( const CopyRecord& cr);
	
	friend ostream& operator << (ostream& s, const CopyRecord& cr)
	{
	    s << "(" << cr._tableSet << "," << cr._targetHost << "," << cr._mediatorHost << ","  << cr._user << "," << cr._passwd << ")";
	    return s;
	}

    private:

	unsigned _id;
	Chain _tableSet;
	Chain _targetHost;
	Chain _mediatorHost;

	Chain _user;
	Chain _passwd;	
	Chain _msg;
    };

    class DbSessionRecord {
	
    public:
	
	DbSessionRecord();
	DbSessionRecord(CegoDistDbHandler* pHandler);
	DbSessionRecord(const Chain& hostName, const Chain& tableSet, const Chain& userName, CegoDistDbHandler* pHandler);
	~DbSessionRecord();

	const Chain& getHostName() const;
	const Chain& getTableSet() const;
	const Chain& getUserName() const;

	bool isUsed() const;
	void setUsed(bool isUsed);
	long long getTSLastUsed();
	void setTSLastUsed(long long ts);

	CegoDistDbHandler* getDbHandler() const;

	DbSessionRecord& operator = ( const DbSessionRecord& sr);
        bool operator == ( const DbSessionRecord& sr);
	
	friend ostream& operator << (ostream& s, const DbSessionRecord& sr)
	{
	    s << "(" << sr._tableSet << "," << sr._hostName << ")";
	    return s;
	}

    private:

	Chain _hostName;
	Chain _tableSet;
	Chain _userName;
	long long _lastUsed;
	bool _isUsed;
	CegoDistDbHandler* _pHandler;
    };

    ListT<ObjectRecord> _objList;
    ListT<CopyRecord> _copyList;
    ListT<DbSessionRecord> _dbSessionList;

    RecoveryMode _recoveryMode[TABMNG_MAXTABSET];
    ListT<Chain> _recoveryList;
    
    unsigned _numDbThread;
    unsigned _numAdmThread;
    unsigned _numLogThread;

    unsigned _activeDbThread;
    unsigned _activeAdmThread;
    unsigned _activeLogThread;

    unsigned _nextCopyId;

    CegoDbHandler::ProtocolType _protType;

    Chain _lckFileName;
    bool _logConfigured;

    CegoQueryCache* _pQueryCache[TABMNG_MAXTABSET];
    CegoTableCache* _pTableCache[TABMNG_MAXTABSET];
    
    unsigned long _modId;
};
#endif
