///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoViewObject.cc 
// -----------------
// Cego view object entry implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoViewObject
// 
// Description: Cego view object structure
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// base includes
#include <lfcbase/Tokenizer.h>

// cego includes
#include "CegoViewObject.h"
#include "CegoXMLdef.h"
#include "CegoTypeConverter.h"

#include <string.h>
#include <stdlib.h>

CegoViewObject::CegoViewObject()
{
    _subCOList.Insert(this);
}

CegoViewObject::CegoViewObject(const CegoViewObject& vo) : CegoContentObject(vo)
{
    _viewStmt = vo._viewStmt;
    _subCOList.Insert(this);
}

CegoViewObject::CegoViewObject(const Chain& viewName, int tabSetId) : CegoContentObject(tabSetId, CegoObject::VIEW, viewName)
{
    _subCOList.Insert(this);
}

CegoViewObject::CegoViewObject(int tabSetId, const Chain& viewName, const ListT<CegoField>& schema, const Chain& viewStmt) : CegoContentObject(tabSetId, CegoObject::VIEW, viewName, viewName, schema)
{
    _viewStmt = viewStmt;    
    _subCOList.Insert(this);
}

CegoViewObject::~CegoViewObject()
{   
}

void CegoViewObject::setSchema(const ListT<CegoField>& schema)
{
    _schema = schema;
}

int CegoViewObject::getEntrySize() const
{
    int entrySize = CegoContentObject::getBaseContentSize();
    entrySize += _viewStmt.length() + 1;
    return entrySize;
}

void CegoViewObject::encode(char *buf) const
{
    char* bufPtr = buf;
    int entrySize = getEntrySize();
    CegoContentObject::encodeBaseContent(bufPtr, entrySize);
    bufPtr += CegoContentObject::getBaseContentSize();
    memcpy (bufPtr, (char*)_viewStmt , _viewStmt.length() + 1);
}

void CegoViewObject::decode(char *buf)
{
    char* bufPtr = buf;
    int size;
    CegoContentObject::decodeBaseContent(bufPtr, size);
    bufPtr += CegoContentObject::getBaseContentSize();
    _viewStmt = Chain(bufPtr);
}

void CegoViewObject::setViewStmt(const Chain& viewStmt)
{
    _viewStmt = viewStmt;
}

const Chain& CegoViewObject::getViewStmt() const
{
    return _viewStmt;
}

CegoViewObject& CegoViewObject::operator = ( const CegoViewObject& vo)
{
    CegoContentObject::operator=(vo);
    _viewStmt = vo._viewStmt;
    return (*this);
}

bool CegoViewObject::operator == ( const CegoViewObject& vo)
{
    return CegoContentObject::operator==(vo);
}

Chain CegoViewObject::getId(CegoProcBlock *pBlock) const
{
    // since it can not be ad hoc decided, if the tuple result of the view objects has been changed,
    // we have to throw MOD_QUERY_ID
    throw Exception(EXLOC, Chain(MOD_QUERY_ID));
}

Chain CegoViewObject::toChain(int defTabSetId) const
{
    if ( defTabSetId == getTabSetId() && defTabSetId != 0)
	return getTabName() + Chain(" ") + getName();
    else
	return getTabName() + Chain("@") + getTableSet() + Chain(" ") + getName();
}

Chain CegoViewObject::getFormatted() const
{
    Chain s;
	
    int maxAttrLen = 12;
    CegoField* pF = _schema.First();
    while (pF)
    {
	if (maxAttrLen < pF->getAttrName().length())
	    maxAttrLen = pF->getAttrName().length();
	pF = _schema.Next();
    }

    int maxFillLen = maxAttrLen + 28;

    Tokenizer pvt(_viewStmt, "\n");
    
    Chain line;
    
    while ( pvt.nextToken(line) )
    {
	if ( line.length() > maxFillLen )
	    maxFillLen = line.length();
    }
    
    s += Chain("+") + fill("-", maxAttrLen) + fill(Chain("-"), 28) + Chain("+\n");
    s += "| ObjectName : ";
    s += getName();
    s += fill(" ", maxAttrLen + 15 - getName().length()) + Chain("|\n");
	
    s += Chain("| ObjectType : ");
    s += Chain("view           ");	
    s += fill(" ", maxAttrLen-1 ) + Chain("|\n");
	
    s += Chain("+-----------") + fill("-", maxAttrLen-10) + Chain("+---------------+----------+\n");
    s += Chain("| Attribute ") +  fill(" ", maxAttrLen-10) + Chain("| Type          | Nullable |\n");
    s += Chain("+-----------") + fill("-", maxAttrLen-10) + Chain("+---------------+----------+\n");
	
    pF = _schema.First();
    while (pF)
    {	
	int num = maxAttrLen - pF->getAttrName().length() ;
	s += Chain("| ") + pF->getAttrName() + fill(" ", num) + Chain(" |");

	switch (pF->getType())
	{
	case INT_TYPE:
	{
	    s += Chain("  int          |");
	    break;
	}
	case LONG_TYPE:
	{
	    s += Chain("  long         |");
	    break;
	}
	case BOOL_TYPE:
	{
	    s += Chain("  bool         |");
	    break;
	}
	case DATETIME_TYPE:
	{
	    s += Chain("  datetime     |");
	    break;
	}
	case VARCHAR_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  string[") + l + Chain("]") + fill(" ", 6 - l.length()) + Chain("|");
	    break;
	}
	case BIGINT_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  bigint[") + l + Chain("]") + fill(" ", 6 - l.length()) + Chain("|");
	    break;
	}
	case SMALLINT_TYPE:
	{
	    s += Chain("  smallint     |");
	    break;
	}
	case TINYINT_TYPE:
	{
	    s += Chain("  tinyint      |");
	    break;
	}
	case DECIMAL_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  decimal[") + l + Chain("]") + fill(" ", 5 - l.length()) + Chain("|");
	    break;
	}
	case FLOAT_TYPE:
	{
	    s += Chain(" float         |");
	    break;
	}
	case DOUBLE_TYPE:
	{
	    s += Chain(" double        |");
	    break;
	}
	case BLOB_TYPE:
	{
	    s += Chain(" blob          |");
	    break;
	}
	case CLOB_TYPE:
	{
	    s += Chain(" clob          |");
	    break;
	}
	case NULL_TYPE:
	case PAGEID_TYPE:
	    // not formatted
	    break;
	}
	if ( pF->isNullable() )
	    s +=  Chain(" y        |");
	else
	    s += " n        |";
	
	pF = _schema.Next();
	    
	s += Chain("\n");
    }
    s += Chain("+") + fill("-", maxAttrLen + 1) + Chain("+---------------+----------+\n");

    Tokenizer vt(_viewStmt, "\n");
        
    while ( vt.nextToken(line) )
    {
	s += Chain("| ") + line + fill(" ", maxFillLen-line.length()) + Chain("|\n");
    }
    s += Chain("+") + fill("-", maxFillLen) + Chain("+\n");

    return s;
}

Chain CegoViewObject::fill(const Chain& s, int num) const
{
    Chain fs = Chain();
    while (num > 0)
    {
	fs = fs + s;
	num--;
    }

    return fs;
}

CegoContentObject* CegoViewObject::clone(bool isAttrRef)
{
    CegoViewObject *pClone = new CegoViewObject(*this);
    pClone->setViewStmt(_viewStmt);
    return (pClone);
}
