///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoLogReader.cc
// ----------------                           
// Cego log reading and printing
// 
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: main
// 
// Description: Utiliy program to read and list cego redo log files
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/GetLongOpt.h>
#include <lfcbase/Chain.h>
#include <lfcbase/Datetime.h>

// CEGO INCLUDES
#include "CegoDefs.h"
#include "CegoLogRecord.h"
#include "CegoQueryHelper.h"
#include "CegoLogManager.h"
#include "CegoTableObject.h"
#include "CegoKeyObject.h"
#include "CegoViewObject.h"
#include "CegoProcObject.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

#define USAGE "Usage: cglog --log=<logfile>\n\
          [ --version  ] [ --help ]"

extern char __lfcVersionString[];
extern char __lfcxmlVersionString[];

int main(int argc, char **argv)
{
    try 
    {
	GetLongOpt longOpt(argc, argv);
	
	longOpt.addOpt("version");
	longOpt.addOpt("help");
	longOpt.addOpt("log");
	
	try
	{
	    longOpt.parseOpt(); 
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    cerr << msg << endl;
	    cerr << USAGE << endl;
	    exit(1);	
	}
		
	if ( longOpt.isSet( Chain("help") ) )
	{	
	    cerr << USAGE << endl;
	    exit(0);
	}
	
	if ( longOpt.isSet( Chain("version") ) )
	{
	    cout << CEGO_PRODUCT << " Log Reader (" << sizeof(void*) * 8 << " bit), Version " << CEGO_VERSION 
		 << " [ lfc: " << __lfcVersionString  
		 << ", lfcxml: " <<  __lfcxmlVersionString << " ]" << endl;
	    cout << CEGO_COPYRIGHT << endl;
	    exit(0);
	}
	
	if ( longOpt.isSet( Chain("log") ) == false )
	{
	    throw Exception(EXLOC, "Redo logfile file must be specified");
	}
	
	Chain logFileName = longOpt.getOptValue("log");
	
	File logFile(logFileName);
	
	logFile.open(File::READ);
		
	int offset;
	int c = logFile.readByte((char*)&offset, sizeof(int));	
	
	int pos = sizeof(int);
	       
	while (pos < offset)
	{
	    int len;
	    c = logFile.readByte((char*)&len, sizeof(int));	
	    
	    cout << "Read len : " << len << endl;
	    
	    pos += len + sizeof(int);
	    
	    char* buf = new char[len];
	    
	    logFile.readByte(buf, len);
	    CegoLogRecord lr;
	    
	    lr.decode(buf);

	    delete[] buf;

	    Datetime dt( lr.getTS() );

	    cout << "LSN=" << lr.getLSN() << ":";
	    cout << "TS=" << dt.asChain() << ":";
	    cout << "TID=" << lr.getTID() << ":";
	    cout << "TAStep=" << lr.getTAStep() << ":";

	    if ( lr.getObjName().length() > 0 )
	    { 
		cout << "ObjName=" << lr.getObjName() << ":";
		cout << "ObjType=" << lr.getObjType() << ":";
	    }

	    cout << "LogAction=";

	    switch ( lr.getAction() )
	    {
	    case CegoLogRecord::LOGREC_CREATE:
	    {
		cout << "CREATE:" << endl;
		
		cout << "--- Log Data Start ---" << endl;
		
		switch ( lr.getObjType() )
		{
		case CegoObject::TABLE:
		case CegoObject::SYSTEM:
		case CegoObject::AVLTREE:
		case CegoObject::PAVLTREE:
		case CegoObject::UAVLTREE:
		{
		    CegoTableObject to;
		    to.decode(lr.getData());
		    cout << to.toChain(0);
		    break;
		}
		case CegoObject::PBTREE:
		case CegoObject::UBTREE:
		case CegoObject::BTREE:
		{
		    // TODO
		    break;
		}
		case CegoObject::VIEW:
		{
		    CegoViewObject vo;
		    vo.decode(lr.getData());
		    cout << vo.toChain(0);
		    break;		    
		}
		case CegoObject::PROCEDURE:
		{
		    CegoProcObject po;
		    po.decode(lr.getData());
		    cout << po.toChain();
		    break;		    
		}
		case CegoObject::FKEY:
		{
		    CegoKeyObject ko;
		    ko.decode(lr.getData());
		    cout << ko.toChain();
		    break;		    
		}
		case CegoObject::CHECK:
		{
		    // TODO
		    break;
		}
		case CegoObject::TRIGGER:
		{
		    // TODO
		    break;
		}
		case CegoObject::ALIAS:
		{
		    // TODO
		    break;
		}
		case CegoObject::RBSEG:
		case CegoObject::JOIN:
		case CegoObject::UNDEFINED:
		{
		    // ignore
		    break;
		}
		}
		
		cout << "--- Log Data End   ---" << endl;
       
		break;
	    }
	    case CegoLogRecord::LOGREC_ALTER:
	    {
		cout << "ALTER:" << endl;

		cout << "--- Log Data Start ---" << endl;

		switch ( lr.getObjType() )
		{
		case CegoObject::TABLE:
		case CegoObject::SYSTEM:
		case CegoObject::AVLTREE:
		case CegoObject::PAVLTREE:
		case CegoObject::UAVLTREE:
		{
		    CegoTableObject to;
		    to.decode(lr.getData());
		    cout << to.toChain(0);
		    break;
		}
		case CegoObject::PBTREE:
		case CegoObject::UBTREE:
		case CegoObject::BTREE:
		{
		    // TODO
		    break;
		}
		
		case CegoObject::VIEW:
		{
		    CegoViewObject vo;
		    vo.decode(lr.getData());
		    cout << vo.toChain(0);
		    break;		    
		}
		case CegoObject::PROCEDURE:
		{
		    CegoProcObject po;
		    po.decode(lr.getData());
		    cout << po.toChain();
		    break;		    
		}
		case CegoObject::FKEY:
		{
		    CegoKeyObject ko;
		    ko.decode(lr.getData());
		    cout << ko.toChain();
		    break;		    
		}
		case CegoObject::CHECK:
		{
		    // TODO
		    break;
		}
		case CegoObject::TRIGGER:
		{
		    // TODO
		    break;
		}
		case CegoObject::ALIAS:
		{
		    // TODO
		    break;
		}
		case CegoObject::RBSEG:
		case CegoObject::JOIN:
		case CegoObject::UNDEFINED:
		{
		    // ignore
		    break;
		}
		}		
		cout << "--- Log Data End   ---" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_RENAME:
	    {
		cout << "RENAME:" << endl;
		cout << "--- Log Data Start ---" << endl;
		Chain newObjName(lr.getData(), lr.getDataLen());
		cout << "NewName=" << newObjName << endl; 
		cout << "--- Log Data End   ---" << endl;
	    }
	    case CegoLogRecord::LOGREC_DROP:
	    {
		cout << "DROP:" << endl;
		cout << "--- No log data ---" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_TRUNCATE:
	    {
		cout << "TRUNCATE:" << endl;	       
		cout << "--- No log data ---" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_INSERT:
	    {
		cout << "INSERT:" << endl;	
		cout << "--- Log Data Start ---" << endl;
		
		unsigned long long tid;
		unsigned long long tastep;
		CegoTupleState ts;

		ListT<CegoFieldValue> fvl;
		ListT<CegoBlob> blobList;
		ListT<CegoClob> clobList;

		int toff = CegoQueryHelper::decodeTupleHeader(tid, tastep, ts, lr.getData());

		char* tp =  lr.getData() + toff;
		int tlen = lr.getDataLen() - toff;

		CegoQueryHelper::decodeNativeFVL(fvl, blobList, clobList, tp, tlen);

		CegoFieldValue *pFV = fvl.First();
		while ( pFV )
		{
		    if ( pFV->getLength() > 0 )
		    {
			cout << *pFV << "|";
		    }
		    
		    // cout << "Field " << pF->getId() << " Len=" << pF->getFieldValue().getLength() << endl;
		    pFV = fvl.Next();
		}
		cout << endl;
		if ( blobList.Size() > 0 )
		{
		    CegoBlob *pBlob = blobList.First();
		    while ( pBlob )
		    {
			cout << "Blob [ ... ] " << pBlob->getSize() << " bytes" << endl;
			pBlob = blobList.Next();
		    }
		}
		if ( clobList.Size() > 0 )
		{
		    CegoClob *pClob = clobList.First();
		    while ( pClob )
		    {
			cout << "Clob [" << pClob->getHead() << "] " << pClob->getSize() << " bytes" << endl;
			pClob = clobList.Next();
		    }
		}

		cout << "--- Log Data End   ---" << endl;
		
		// cout << lr << endl;;

		break;
	    }
	    case CegoLogRecord::LOGREC_DELETE:
	    {
		cout << "DELETE:" << endl;
		
		cout << "--- Log Data Start ---" << endl;	

		CegoPredicate *pPred = 0;
		Chain tableAlias;

		CegoQueryHelper::decodeDelRec(tableAlias, pPred, lr.getData(), lr.getDataLen(), 0, 0);

		cout << " TableAlias=" << tableAlias << endl;
		
		if ( pPred )
		{
		    cout << "where " << pPred->toChain(0) << endl;
		}
		
		cout << "--- Log Data End   ---" << endl;
		
		break;
	    }
	    case CegoLogRecord::LOGREC_UPDATE:
	    {
		cout << "UPDATE:" << endl;		
		cout << "--- Log Data Start ---" << endl;	
		Chain tableAlias;
		CegoPredicate *pPred = 0;
		ListT<CegoField> updList;
		ListT<CegoExpr*> exprList;
		bool returnOnFirst;
		
		CegoQueryHelper::decodeUpdRec(tableAlias,
					      pPred, 
					      updList,
					      exprList,
					      returnOnFirst,
					      lr.getData(), lr.getDataLen(), 0, 0);
				
		cout << "TableAlias=" << tableAlias << endl;
		CegoField *pF = updList.First();
		CegoExpr **pExpr = exprList.First();
		while ( pF && pExpr )
		{
		    cout << "Set " << pF->getAttrName() << " = " << (*pExpr)->toChain(0) << endl;
		    pF = updList.Next();
		    pExpr = exprList.Next();
		}
		if ( pPred )
		    cout << "where " << pPred->toChain(0) << endl;
		cout << "--- Log Data End   ---" << endl;
		
		break;
	    }
	    case CegoLogRecord::LOGREC_BEGIN:
	    {
		cout << "BEGIN:" << endl;
		cout << "--- No log data ---" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_COMMIT:
	    {
		cout << "COMMIT:" << endl;
		cout << "TID=" << lr.getTID() << ":";
		break;
	    }
	    case CegoLogRecord::LOGREC_ABORT:
	    {
		cout << "ABORT:" << endl;
		cout << "TID=" << lr.getTID() << endl;
		break;
	    }	    
	    case CegoLogRecord::LOGREC_SYNC:
	    {
		cout << "SYNC:" << endl;
		cout << "--- No log data ---" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_BUPAGE:
	    {
		cout << "BUPAGE:" << endl;
		cout << "[... page data ...]" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_BUFBM:
	    {
		cout << "BUFBM:" << endl;
		cout << "[ ... file bitmap ...]" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_BUFIN:
	    {
		cout << "BUFIN:" << endl;
		cout << "--- No log data ---" << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_ADDCOUNTER:
	    {
		cout << "ADDCOUNTER:" << endl;
		Chain counterName(lr.getData(), lr.getDataLen());		
		cout << "Counter="<< counterName << endl;
		break;
	    }
	    case CegoLogRecord::LOGREC_DELCOUNTER:
	    {
		cout << "DELCOUNTER:" << endl;
		Chain counterName(lr.getData(), lr.getDataLen());		
		cout << "Counter="<< counterName << endl;
		break;
	    }
	    default:
		cout << " -- Unknown logaction " << lr.getAction() << endl;
		break;   		             	
	    }
	}
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	cerr << msg << endl;
	exit(1);
    }
    exit(0);
}
