///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoDistCursor.cc
// -----------------
// Cego global table cursor class implementation
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoDistCursor
// 
// Description: Table cursor for distributed table access
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Net.h>

// CEGO INCLUDES
#include "CegoDistCursor.h"
#include "CegoDistManager.h"
#include "CegoSelect.h"
#include "CegoTableObject.h"
#include "CegoJoinObject.h"
#include "CegoXMLdef.h"

CegoDistCursor::CegoDistCursor()
{
}

CegoDistCursor::CegoDistCursor(CegoDistManager* pGTM, CegoContentObject *pCO)
{
    _isLocal = true;
    _moreTuple = false;
    
    _pSelect = 0;
    _pSH = 0;
    _pGTM = pGTM;

    _cursorObjectUsed = false;
    
    _tableName = pCO->getTabName();
    _tableAlias = pCO->getName();
    _tabSetId = pCO->getTabSetId();
    
    _pDBMng = _pGTM->getDBMng(); 

    _useCache = false;
    _pCache = _pDBMng->getTableCache(_tabSetId);
    _isCached = false;
    _pCacheArray = 0;
    _pCacheList = 0;
    _pCO = pCO;

    _pTCTab = 0;
    _pTO = 0;
    _pTCLeft = 0;
    _pTCRight = 0;
    _pTC = 0;
    _pC = 0;

    _pFLA = 0;
    _pFLAmap = 0;
    
    _idxMatch = CegoAttrCond::INAPP;

    checkType();
    
    _modId = _pDBMng->getModId("CegoDistCursor");
}

CegoDistCursor::~CegoDistCursor()
{
    try
    {
	finishCaching();
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Cannot finish caching ( ignored ) : " + msg ));
    }
    
    if ( _pTC )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing global table cursor for ") + _tableName);
#endif

	delete _pTC;
    }
    if ( _pSelect )
    {
	delete _pSelect;			
    }
    if ( _pCO->getType() == CegoObject::JOIN )
    {
#ifdef CGDEBUG
	CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	CegoContentObject *pCOLeft = pJCO->getLeftObject();
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing global table cursor for ") + pCOLeft->getTabName());
#endif	
	delete _pTCLeft;
	
#ifdef CGDEBUG
	CegoContentObject *pCORight = pJCO->getRightObject();
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing global table cursor for ") + pCORight->getTabName());
#endif	
	delete _pTCRight;		
    }

    if ( _pCO->getType() == CegoObject::ALIAS )
    {
	if ( _pTCTab )
	    delete _pTCTab;
	if ( _pTO )
	    delete _pTO;
	if ( _pFLAmap )
	    delete _pFLAmap;
    }
    if ( _pC )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing object cursor for ") + _tableName);
#endif
	delete _pC;
    }

    try
    {
	unuseCursorObject();
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Cannot unuse cursor object  ( ignored ) : " + msg ));
    }
    
    try
    {
	if ( _pSH )
	{
	    _pDBMng->releaseSession(_pSH);
	}	
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Cannot release session  ( ignored ) : " + msg ));
    }

    if ( _pCacheList )
    {
	delete _pCacheList;
    }
	
    if ( _pCacheArray && _pCache )
    {
	_pCache->releaseEntry(_tableName);
	_pCacheArray=0;
	_isCached = true;
    }
	
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Destructor for DistCursor finished"));
#endif
}

void CegoDistCursor::useCursorObject()
{
    if ( _cursorObjectUsed == false )
    {
	if ( _pCO->getType() == CegoObject::VIEW )
	{
	    _pDBMng->useObject(_tabSetId, _tableName, CegoObject::VIEW, CegoDatabaseManager::SHARED, _pGTM);
	}
	else if ( _pCO->getType() == CegoObject::TABLE
		  || _pCO->getType() == CegoObject::ALIAS )
	{
	    _pDBMng->useObject(_tabSetId, _tableName, CegoObject::TABLE, CegoDatabaseManager::SHARED, _pGTM);
	}
	else if ( _pCO->getType() == CegoObject::JOIN )
	{
	    CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	    CegoContentObject *pCOLeft = pJCO->getLeftObject();
	    CegoContentObject *pCORight = pJCO->getRightObject();

	    if ( pCOLeft->getType() == CegoObject::VIEW || pCOLeft->getType() == CegoObject::TABLE )
		_pDBMng->useObject(pCOLeft->getTabSetId(), pCOLeft->getTabName(), pCOLeft->getType(), CegoDatabaseManager::SHARED, _pGTM);
	    if ( pCORight->getType() == CegoObject::VIEW || pCORight->getType() == CegoObject::TABLE )
		_pDBMng->useObject(pCORight->getTabSetId(), pCORight->getTabName(), pCORight->getType(), CegoDatabaseManager::SHARED, _pGTM);
	}

	_cursorObjectUsed = true;
    }
}

void CegoDistCursor::unuseCursorObject()
{
    if ( _cursorObjectUsed == true )
    {
	if ( _pCO->getType() == CegoObject::VIEW )
	{
	    _pDBMng->unuseObject(_tabSetId, _tableName, CegoObject::VIEW);	
	}
	else if ( _pCO->getType() == CegoObject::TABLE
		  || _pCO->getType() == CegoObject::ALIAS )
	{
	    _pDBMng->unuseObject(_tabSetId, _tableName, CegoObject::TABLE);	
	}
	else if ( _pCO->getType() == CegoObject::JOIN )
	{	
	    CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	    CegoContentObject *pCOLeft = pJCO->getLeftObject();
	    CegoContentObject *pCORight = pJCO->getRightObject();

	    if ( pCOLeft->getType() == CegoObject::VIEW || pCOLeft->getType() == CegoObject::TABLE )
		_pDBMng->unuseObject(pCOLeft->getTabSetId(), pCOLeft->getTabName(), pCOLeft->getType());	
	    if ( pCORight->getType() == CegoObject::VIEW || pCORight->getType() == CegoObject::TABLE )
		_pDBMng->unuseObject(pCORight->getTabSetId(), pCORight->getTabName(), pCORight->getType());	
	}
	_cursorObjectUsed = false;
    }
}

void CegoDistCursor::checkType()
{
    useCursorObject();
    
    if ( _pCO->getType() == CegoObject::VIEW )
    {
	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("View ") + _tableName + Chain(" detected as the cursor source"));
#endif

	CegoView *pView;

	try
	{
	    pView = _pGTM->getView(_tabSetId, _tableName);
	}
	catch ( Exception e )
	{
	    _pDBMng->unuseObject(_tabSetId, _tableName, CegoObject::VIEW);
	    throw Exception(EXLOC, Chain("Cannot get view object ") + _tableName, e);
	}

	// since a view might be used several times inside a single query, we have to use a clone	
	_pSelect = pView->getSelect()->clone(false);
	_pSelect->cleanUp();
    }
    else if ( _pCO->getType() == CegoObject::ALIAS )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Alias ") + _tableAlias + Chain(" detected as the cursor source"));
#endif	
	Chain tableSet = _pDBMng->getTabSetName(_tabSetId);
	_pTO = new CegoTableObject();
	_pGTM->getDistObject(tableSet, _tableName, CegoObject::TABLE, *_pTO);

	_pTCTab = new CegoDistCursor(_pGTM, _pTO);	
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Table ") + _tableName + Chain(" detected as the cursor source"));
#endif
	Chain tableSet = _pDBMng->getTabSetName(_tabSetId);

	CegoTableObject oe;
	_pGTM->getDistObject(tableSet, _tableName, CegoObject::TABLE, oe);
	
	if ( oe.isLocal() )
	{
#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Native Table ") + _tableName + Chain(" detected as the cursor source"));
#endif
	    _pTC = new CegoTableCursor(_pGTM, _tabSetId, _tableName, false);	

	    _cacheSchema = oe.getSchema();
	    _isLocal = true;
	}
	else
	{
#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Remote Table ") + _tableName + Chain(" detected as the cursor source"));
#endif
	    _isLocal = false;

	    int tabSetId = oe.getTabSetId();
	    Chain hostName = _pDBMng->getPrimary(tabSetId);
	    int portNo;
	    _pDBMng->getDataPort(portNo);

	    _schema = oe.getSchema();

	    Chain user;
	    Chain password;
	    
	    _pGTM->getActiveUser(tableSet, user, password);
	    
	    _pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);
	    	    
	    _pSH->reqTableDataOp(tabSetId, _tableName,  CegoObject::TABLE);
	}      
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {	
	CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	CegoContentObject *pCOLeft = pJCO->getLeftObject();
	CegoContentObject *pCORight = pJCO->getRightObject();
	
	_pTCLeft = new CegoDistCursor(_pGTM, pCOLeft);
	_pTCRight = new CegoDistCursor(_pGTM, pCORight);
    }

    // we have to release object here to avoid dead locks
    unuseCursorObject();    
}

void CegoDistCursor::distSetup( ListT<CegoField>** pFLA)
{
    _isFirst=true;
    _doEval = false;
    _idxMatch = CegoAttrCond::INAPP;
    _isAttrCondValid = false;
    _pFLA = pFLA;
    
    if ( _pCO->getType() == CegoObject::VIEW )
    {
	_pSelect->setTabSetId(_tabSetId);
	CegoAttrCond noCond;

	_pSelect->setViewCond(noCond, pFLA);
	_pSelect->prepare();
	_pSelect->checkValidRef();
    }
    else if ( _pCO->getType() == CegoObject::ALIAS )
    {
	_pTCTab->distSetup(_pFLA);
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {	
	CegoAttrCond attrCond; // empty cond
	_pTC->setup(attrCond);
	
	if ( _pCache )
	{
	    _useCache = true;
	    
	    finishCaching();

	    if ( _pCache->matchEntry(_tableName) )
	    {
		_pCacheArray = _pCache->claimEntry( _tableName, _cacheRows, _cacheCols);
		
		if ( _pCacheArray )
		{
		    _isCached = true;
		}
		else
		{
		    _isCached = false;
		    _cacheEntrySize = 0;
		    _pCacheList = new ListT< ListT< CegoFieldValue > >;
		}
	    }
	}	
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {
	sysSetup();
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {
	CegoAttrCond attrCond;
	joinSetup(attrCond);
    }
}

void CegoDistCursor::distSetup( CegoAttrCond& attrCond, ListT<CegoField>** pFLA)
{
    
    if ( attrCond.numComp() == 0 )
    {
	return distSetup(pFLA);
    }
    _isAttrCondValid = false;
    _idxMatch = CegoAttrCond::INAPP;
    _isFirst=true;
    _doEval = false;
    _pFLA = pFLA;
    
    if ( _pCO->getType() == CegoObject::VIEW )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Setting condition for view ")
		     + _tableName
		     + Chain(" [") + _tableAlias + Chain("] to ") + attrCond.toChain());
#endif
	
	// first setup tabSetId with anabling / disabling quey cache
	_pSelect->setTabSetId(_tabSetId);
	
	// we setup view condition with the given attrcond
	// if cannot be setup completely ( aggregation in expression ),
	// we force a dedicated evaluation

	if ( _pSelect->setViewCond(attrCond, pFLA) )
	    _doEval=false;
	else
	    _doEval=true;

	_pSelect->prepare();

	_pSelect->checkValidRef();

	_cursorCond = attrCond;
    }
    else if ( _pCO->getType() == CegoObject::ALIAS )
    {
	CegoAliasObject *pAO = (CegoAliasObject*)_pCO;
	CegoAttrCond mapAttrCond = CegoQueryHelper::mapAttrCond(attrCond, pAO);
	
	_pTCTab->distSetup(mapAttrCond, pFLA);
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
	if ( _isLocal == true )
	{
	    if ( _pCache )
	    {
		// check if a previous cursor trace has to be finished to complete cache 
		finishCaching();
	    }
	    
	    _useCache = false;

	    _idxMatch = _pTC->setup(attrCond);
	    
	    if ( _idxMatch != CegoAttrCond::FULL )
	    {
		_doEval=true;

		// if index match was inappropriate, all tuples are retrieved from table cursor
		// In this case, we can use cache
		if ( _pCache && _idxMatch == CegoAttrCond::INAPP )
		{
		    if ( _pCache->matchEntry(_tableName) )	   
		    {
			_useCache = true;
			_pCacheArray = _pCache->claimEntry( _tableName, _cacheRows, _cacheCols);
			
			if ( _pCacheArray )
			{
			    _isCached = true;
			}
			else
			{
			    _isCached = false;
			    _cacheEntrySize = 0;
			    _pCacheList = new ListT< ListT< CegoFieldValue > >;
			}
		    }
		}
	    }
	    _cursorCond = attrCond;  
	}
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {
	sysSetup();
	_cursorCond = attrCond;
	_doEval = true;	
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Setting join condition ") + attrCond.toChain());	
#endif
	_cursorCond = attrCond;

	// for inner joins, predicate is evaluated inside subjoin, so no need for evaluation	
	joinSetup(attrCond);
    }
}

bool CegoDistCursor::nextTuple(ListT<CegoField>** pFLA, int pos, int size)
{
    if ( _pGTM->isAborted() )
    {
	throw Exception(EXLOC, "Query aborted");
    }
    
    useCursorObject();
    
    while ( getTuple(pFLA, pos, size) )
    {
	/*
	cout << "### TUPLE = ";
	for ( int i=0; i<size; i++)
	{
	    
	    CegoField *pF = pFLA[pos+i]->First();
	    while ( pF )
	    {
		cout << pF->getTableAlias() << "." << pF->getAttrName() << "=" << pF->getValue() << " ";
		pF = pFLA[pos+i]->Next();
	    }		
	}
	cout << endl;
	*/
	
	if ( _doEval == true )
	{
	    if ( evalCondition(pFLA, pos, size) )
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Returning tuple with eval condition"));
#endif
		return true;
	    }
	}
	else
	{
#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Returning tuple without eval condition"));
#endif
	    return true;
	}
    }
    return false;
}

bool CegoDistCursor::evalCondition(ListT<CegoField>** pFLA, int pos, int size)
{
    CegoAttrComp *pAC = _cursorCond.getAttrCompSet().First();
    
    while ( pAC )
    {
	// cout << "Eval condition " << *pAC << endl;	
	// cout << "Condition " << pAC->getTableName() << " . " <<  pAC->getAttrName() << endl;
	
	CegoField *pF = 0;
	    
	int i=0;
	while ( pF == 0 && i < size)
	{
	    if ( _pCO->getType() == CegoObject::VIEW || _pCO->getType() == CegoObject::TABLE )
		pF = pFLA[pos+i]->Find( CegoField(_tableAlias, pAC->getAttrName()));
	    else
		pF = pFLA[pos+i]->Find( CegoField(pAC->getTableName(), pAC->getAttrName()) );
	    i++;
	}
	
	if ( pF )
	{
	    if ( pAC->getCompMode() == CegoAttrComp::BTWN )
	    {
		if ( ( (CegoFieldValue)pF->getValue() >= (CegoFieldValue)(pAC->getFieldValue()) 
		       && (CegoFieldValue)pF->getValue() <= (CegoFieldValue)(pAC->getFieldValue2()) ) == false )
		    
		    return false;	    
	    }
	    else if ( pAC->getCompMode() == CegoAttrComp::ISLIKE )
	    {
		if ( pAC->getMatcher()->match(pF->getValue().valAsChain()) == false)		
		    return false;
	       
	    }
	    else if ( pAC->getCompMode() == CegoAttrComp::ISNOTLIKE )
	    {
		if ( pAC->getMatcher()->match(pF->getValue().valAsChain()))
		    return false;		
	    }
	    else
	    {		
		switch ( pAC->getComparison() )
		{	    
		case EQUAL:
		{
		    // cout << "Eval " << pF->getValue() << " = " << pAC->getFieldValue() << endl;
		    if ( ( (CegoFieldValue)pF->getValue() == (CegoFieldValue)(pAC->getFieldValue()) ) == false )		    
			return false;		    
		    break;
		}
		case NOT_EQUAL:
		{
		    if ( ( (CegoFieldValue)pF->getValue() != (CegoFieldValue)(pAC->getFieldValue()) ) == false )		    
			return false;	    
		    break;
		}
		case LESS_THAN:
		{
		    if ( ( (CegoFieldValue)pF->getValue() < (CegoFieldValue)(pAC->getFieldValue()) ) == false )		     
			return false;
		    break;
		}
		case MORE_THAN: 
		{
		    if ( ( (CegoFieldValue)pF->getValue() > (CegoFieldValue)(pAC->getFieldValue()) ) == false )
			return false;
		    break;
		}
		case LESS_EQUAL_THAN:
		{
		    if ( ( (CegoFieldValue)pF->getValue() <= (CegoFieldValue)(pAC->getFieldValue()) ) == false )
			return false;
		    break;
		}
		case MORE_EQUAL_THAN:
		{
		    if ( ( (CegoFieldValue)pF->getValue() >= (CegoFieldValue)(pAC->getFieldValue()) ) == false )
			return false;	    
		    break;
		}
		}
	    }
	}
	else
	{
	    if ( pAC->isSetup() == false )
	    {
		Chain msg = Chain("Cannot get value for attribute ") + pAC->getAttrName();
		throw Exception(EXLOC, msg);
	    }
	}
	pAC = _cursorCond.getAttrCompSet().Next();
    }
    return true;
}

bool CegoDistCursor::getTuple(ListT<CegoField>** pFLA, int offset, int size)
{
    if ( _pCO->getType() == CegoObject::VIEW )
    {
	try 
	{
	    ListT<CegoField> fvl;
	    _moreTuple = _pSelect->nextTuple(fvl);

	    /*
	    cout << "FVL Size = " << fvl.Size() << endl;

	    CegoField *pX = fvl.First();
	    while ( pX ) 
	    {
		cout << "Alias = " << pX->getTableAlias() << " Attr =" << pX->getAttrName() << " " << pX->getValue() << endl;
		
		pX = fvl.Next();
	    }
	    */
		    
	    if ( _moreTuple )
	    {
		CegoField *pF = pFLA[offset]->First();
		while ( pF ) 
		{
		    // cout << "Checking FLA .. Alias =" << pF->getTableAlias() << " Attr = " << pF->getAttrName() << endl;
		    
		    CegoField *pSF = fvl.First();
		    bool notFound=true;
		    while ( pSF && notFound)
		    {
			if ( pSF->getAttrName() == pF->getAttrName() )
			{
			    pF->setValue(pSF->getValue());
			    notFound=false;
			}
			pSF = fvl.Next();
		    }
		    pF = pFLA[offset]->Next();
		}
	    }
	    return _moreTuple;
	}
	catch ( Exception e )
	{
	    _pSelect->cleanUp();
	    throw Exception(EXLOC, Chain("Cannot get tuple from view"), e);
	}
    }
    else if ( _pCO->getType() == CegoObject::ALIAS )
    {
	CegoAliasObject *pAO = (CegoAliasObject*)_pCO;
	
	if ( _isFirst ) 
	{
	    _pFLAmap = new ListT<CegoField>();
	    CegoQueryHelper::mapFLA(_pFLAmap, pFLA, offset, size, pAO);
	    _isFirst = false;
	}
	
	if ( _pTCTab->nextTuple(&_pFLAmap, 0, 1) )
	{
	    CegoQueryHelper::propFLA(_pFLAmap, pFLA, offset, size, pAO);
	    return true;
	}
	return false;	
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
	if ( _isLocal )
	{
	    if ( _useCache )
	    {
		if ( _isCached )
		{
		    CegoFieldValue** pCacheRow = 0;
		    if ( _isFirst ) 
		    {
			_isFirst = false;
			_cachePos = 0;
			if ( _cacheRows > 0 )
			{
			    pCacheRow = _pCacheArray[_cachePos];
			    _cachePos++;			
			}			  
		    }
		    else
		    {
			if ( _cachePos < _cacheRows )
			{
			    pCacheRow = _pCacheArray[_cachePos];
			    _cachePos++;
			}
		    }
		    	
		    if ( pCacheRow  )
		    {
			CegoField *pF = pFLA[offset]->First();
			
			while ( pF )
			{
			    CegoField *pSF = _cacheSchema.First();
			    
			    int pos = 0;
			    while ( pF && pSF && ( *pF != *pSF ) )
			    {				
				pSF = _cacheSchema.Next();
				pos++;
			    }
			    if ( *pF == *pSF )
			    {
				pF->setValue(*pCacheRow[pos]);
			    }
			    pF = pFLA[offset]->Next();
			}

			_moreTuple = true;
		    }
		    else
		    {			
			if ( _pCacheArray && _pCache )
			{
			    _pCache->releaseEntry( _tableName);
			    _pCacheArray = 0;
			    _isCached = true;
			}
			_moreTuple = false;
		    }

		    return _moreTuple;
		}
		else // still not cached, so we have to retrieve and cache it
		{		    
		    CegoDataPointer dp;
		    if ( _isFirst )
		    {
			_moreTuple = _pTC->getFirst(_cacheSchema, dp);			
			_isFirst = false;
		    }
		    else
		    {
			_moreTuple = _pTC->getNext(_cacheSchema, dp);	    
		    }
		    
		    if ( _moreTuple )
		    {
			// put tuple into cache

			if ( _pCacheList )
			{
			    ListT<CegoFieldValue> staticFieldList;
			    CegoField* pF = _cacheSchema.First();
			    while ( pF )
			    {			
				staticFieldList.Insert(pF->getValue().getLocalCopy());			       
				_cacheEntrySize += pF->getValue().size();
				pF = _cacheSchema.Next();
			    }

			    if ( _pCache->getMaxSize() > _cacheEntrySize )
			    {
				_pCacheList->Insert(staticFieldList);
			    }
			    else
			    {
				delete _pCacheList;
				_pCacheList = 0;
			    }
			}

			// propagate  tuple to flArray
			CegoField *pF = pFLA[offset]->First();			
			while ( pF )
			{
			    CegoField *pSF = _cacheSchema.First();
			    while ( pF && pSF && ( *pF != *pSF ) )
			    {
				pSF = _cacheSchema.Next();
			    }
			    // cout << "Comparing " << pF->getTableAlias() << "." << pF->getAttrName()  << " and " << pSF->getTableAlias() << "." << pSF->getAttrName() << endl;
			    if ( *pF == *pSF )
			    {
				// cout << "MATCH" << endl;
				pF->setValue(pSF->getValue());
			    }
			    pF = pFLA[offset]->Next();
			}
		    }
		    else
		    {
			if ( _pCacheList )
			{
			    // put table cache entry into cache
			    _pCache->addEntry( _tableName, _pCacheList );
			    _isCached = true;
			    delete _pCacheList;
			    _pCacheList = 0;
			}
		    }
		    return _moreTuple;
		}
	    }
	    else // retrieve tuple via native table cursor
	    {
		CegoDataPointer dp;
		if ( _isFirst )
		{		
		    _moreTuple = _pTC->getFirst(*pFLA[offset], dp);
		    _isFirst = false;		  
	        }
		else
		{
		    _moreTuple = _pTC->getNext(*pFLA[offset], dp);		    		   
		}
		return _moreTuple;
	    }
	}
	else
	{
	    *pFLA[offset] = _schema;
	    _moreTuple = false;

	    if ( _pSH->receiveTableData(*pFLA[offset]) == CegoDbHandler::DB_DATA )
	    {
		_moreTuple = true;
	    }

	    return _moreTuple;	    
	}
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {
	if ( _pC )
	{
	    CegoDataPointer dp;

	    int len;
	    char *pc;
	    if ( _isFirst )
	    {
		pc = (char*)_pC->getFirst(len, dp);		
		_isFirst=false;
	    }
	    else
	    {
		pc = (char*)_pC->getNext(len, dp);
	    }

	    if ( pc && len > 0 )
	    {
		unsigned long long tid;
		unsigned long long tastep;
		CegoTupleState ts;

		int toff = CegoQueryHelper::decodeTupleHeader(tid, tastep, ts, pc);

		char* tp = pc + toff;
		int tlen = len - toff;

		CegoQueryHelper::decodeFVL(*pFLA[offset], tp, tlen);
		
		return true;   
	    }
	    else
	    {
		return false;
	    }
	}
	else
	{
	    Chain *pName = 0;
	    if ( _isFirst )
	    {
		_isFirst = false;
		pName = _sysObjList.First();
	    }	
	    else
	    {
		pName = _sysObjList.Next();	    
	    }
	    
	    if ( pName )
	    {
		CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_NAME_ATTR)));
		if ( pF )		    
		{
		    pF->setValue( CegoFieldValue(VARCHAR_TYPE, *pName));
		}
		
		if ( _tableName == Chain(SYSTAB_TABLE_ID) 
		     || _tableName == Chain(SYSTAB_INDEX_ID)
		     || _tableName == Chain(SYSTAB_BTREE_ID) )
		{	    
		    int pageCount;
		    
		    if ( _tableName == Chain(SYSTAB_TABLE_ID)  )
			pageCount = _pGTM->getPageCount(_tabSetId, *pName, CegoObject::TABLE);
		    else if ( _tableName == Chain(SYSTAB_INDEX_ID)  )
			pageCount = _pGTM->getPageCount(_tabSetId, *pName, CegoObject::AVLTREE);
		    else if ( _tableName == Chain(SYSTAB_BTREE_ID)  )
			pageCount = _pGTM->getPageCount(_tabSetId, *pName, CegoObject::BTREE);

		    CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_SIZE_ATTR)));
		    if ( pF )		    
		    {
			pF->setValue( CegoFieldValue(INT_TYPE, pageCount));
		    }
		    pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_STATUS_ATTR)));
		    if ( pF )
		    {
			if ( pageCount > 0 )
			    pF->setValue( CegoFieldValue(VARCHAR_TYPE, Chain("valid")));
			else
			    pF->setValue( CegoFieldValue(VARCHAR_TYPE, Chain("invalid")));
		    }
		    
		}
		else if ( _tableName == Chain(SYSTAB_VIEW_ID) )
		{
		    Chain status("not compiled");
		    if ( _pGTM->checkCompView(_tabSetId, *pName) )
			status = Chain("compiled");
		    
		    CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_STATUS_ATTR)));
		    if ( pF )		    
		    {
			pF->setValue( CegoFieldValue(VARCHAR_TYPE, status));
		    }		
		}
		else if ( _tableName == Chain(SYSTAB_PROC_ID) )
		{
		    Chain status("not compiled");
		    if ( _pGTM->checkCompProcedure(_tabSetId, *pName) )
			status = Chain("compiled");
		    
		    CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_STATUS_ATTR)));
		    if ( pF )		    
		    {
			pF->setValue( CegoFieldValue(VARCHAR_TYPE, status));
		    }				
		}
		_moreTuple = true;	    
	    }
	    else
	    {
		_moreTuple = false;
	    }
	}
	return _moreTuple;
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {	
	if ( _isFirst )
	{
	    _moreLeft = true;
	    _moreRight = true;
	}

	CegoJoinObject *pJO = (CegoJoinObject*)_pCO;
	
	if ( pJO->getJoinType() == CegoJoinObject::INNER )
	{   
	    bool isEvaluated = false;
	    
	    while ( isEvaluated == false && _moreLeft )
	    {
		if ( _isFirst )
		{
		    _pTCLeft->distSetup(_outerCond, _pFLA);
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    if ( _moreLeft )
		    {			
			_innerCond.setup(pFLA, offset);

			_pTCRight->reset();
			
			if ( _isAttrCondValid )
			{
			    _pTCRight->distSetup(_innerCond, _pFLA);
			}
			else
			{
			    _pTCRight->distSetup(_pFLA);
			}
			_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    }
		    _isFirst = false;
		}
		else if ( _moreRight )
		{
		    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    if ( _moreRight == false )
		    {
			_moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
			if ( _moreLeft )
			{
			    _innerCond.setup(pFLA, offset);
			    _pTCRight->reset();

			    if ( _isAttrCondValid )
			    {
				_pTCRight->distSetup(_innerCond, _pFLA);
			    }
			    else
			    {
				_pTCRight->distSetup(_pFLA);
			    }
			    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
			}
		    }
		}
		else if ( _moreLeft )
		{
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    if ( _moreLeft )
		    {
			_innerCond.setup(pFLA, offset);
			_pTCRight->reset();

			if ( _isAttrCondValid )
			{
			    _pTCRight->distSetup(_innerCond, _pFLA);
			}
			else
			{
			    _pTCRight->distSetup(_pFLA);
			}
			_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    }
		}

		if ( _moreLeft && _moreRight )
		{
		    if ( _evalPredicate )
		    {		       
			// attr cache is left for performace
			// pJO->getPredDesc()->clearAttrCache();
			
			isEvaluated = pJO->getPredDesc()->eval(0,
							       0,
							       pFLA,
							       offset,							       
							       0); 
		    }
		    else
		    {			
			isEvaluated = true;
		    }
		}
	    }
	    return _moreLeft && _moreRight;
	}
	else if ( pJO->getJoinType() == CegoJoinObject::LEFTOUTER )
	{
	    if ( _isFirst )
	    {
		_pTCLeft->distSetup(_outerCond, _pFLA);	    
		_moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		
		if ( _moreLeft )
		{
		    _innerCond.setup(pFLA, offset);
		    _pTCRight->reset();
		    
		    if ( _isAttrCondValid )
		    {
			_pTCRight->distSetup(_innerCond, _pFLA);
		    }
		    else
		    {
			_pTCRight->distSetup(_pFLA);
		    }
		    
		    nextRight(pFLA, offset, size);
		}
		_isFirst = false;
	    }
	    else if ( _moreRight )
	    {
		nextRight(pFLA, offset, size);
		
		if ( _moreRight == false )
		{
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    if ( _moreLeft )
		    {
			_innerCond.setup(pFLA, offset);
			
			_pTCRight->reset();
			
			if ( _isAttrCondValid )
			{
			    _pTCRight->distSetup(_innerCond, _pFLA);
			}
			else
			{
			    _pTCRight->distSetup(_pFLA);
			}

			nextRight(pFLA, offset, size);

		    }
		}		
	    }
	    else if ( _moreLeft )
	    {
		_moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		if ( _moreLeft )
		{
		    _innerCond.setup(pFLA, offset);
		    _pTCRight->reset();
		    
		    if ( _isAttrCondValid )
		    {
			_pTCRight->distSetup(_innerCond, _pFLA);
		    }
		    else
		    {
			_pTCRight->distSetup(_pFLA);
		    }

		    nextRight(pFLA, offset, size);
		}
	    }
	    
	    if ( _moreLeft )
	    {		
		if ( _moreRight == false  )
		{
		    // make right fields null
		    CegoField *pF = pFLA[offset+size-1]->First();
		    while ( pF )
		    {
			CegoFieldValue nullValue;
			pF->setValue(nullValue);
			pF = pFLA[offset+size-1]->Next();
		    }
		}
		return true;
	    }
	    return false;
	}	
	else if ( pJO->getJoinType() == CegoJoinObject::RIGHTOUTER )
	{    
	    if ( _isFirst )
	    {		
		_pTCRight->distSetup(_outerCond, _pFLA);
		
		_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		
		if ( _moreRight )
		{
		    _innerCond.setup(pFLA, offset+size-1);	    
		    _pTCLeft->reset();
		    
		    if ( _isAttrCondValid )
			_pTCLeft->distSetup(_innerCond, _pFLA);
		    else
			_pTCLeft->distSetup(_pFLA);

		    nextLeft(pFLA, offset, size);
		}
		_isFirst = false;
	    }
	    else if ( _moreLeft )
	    {
		nextLeft(pFLA, offset, size);
		
		if ( _moreLeft == false )
		{
		    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    if ( _moreRight )
		    {
			_innerCond.setup(pFLA, offset+size-1);
			
			_pTCLeft->reset();
			
			if ( _isAttrCondValid )
			    _pTCLeft->distSetup(_innerCond, _pFLA);
			else
			    _pTCLeft->distSetup(_pFLA);

			nextLeft(pFLA, offset, size);
		    }
		}		
	    }
	    else if ( _moreRight )
	    {
		_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		if ( _moreRight )
		{
		    _innerCond.setup(pFLA, offset+size-1);
		    
		    _pTCLeft->reset();
		    
		    if ( _isAttrCondValid )
			_pTCLeft->distSetup(_innerCond, _pFLA);
		    else
			_pTCLeft->distSetup(_pFLA);

		    nextLeft(pFLA, offset, size);
		}	   	
	    }

	    if ( _moreRight )
	    {
		if ( _moreLeft == false )
		{
		    for ( int i=0; i<size-1; i++)
		    {
			// make right fields null
			CegoField *pF = pFLA[offset+i]->First();
			while ( pF )
			{
			    CegoFieldValue nullValue;
			    pF->setValue(nullValue);
			    pF = pFLA[offset+i]->Next();
			}
		    }
		}
		return true;
	    }
	    return false;
	}	    
    }

    Chain msg = Chain("Invalid cursor object ") + _pCO->getTabAlias() + Chain(" (") + _pCO->getTabName() + Chain(")");
    throw Exception(EXLOC, msg);
}

void CegoDistCursor::nextRight(ListT<CegoField>** pFLA, int offset, int size)
{
    CegoJoinObject *pJO = (CegoJoinObject*)_pCO;
	
    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
    
    bool notFound = true;
    while ( _evalPredicate && _moreRight && notFound )
    {
	// attr cache is left for performace
	// pJO->getPredDesc()->clearAttrCache();
	
	if ( pJO->getPredDesc()->eval(0,
				      0,
				      pFLA,
				      offset,
				      0) )
	{
	    notFound = false;
	}
	else
	{
	    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
	}			      
    }
}

void CegoDistCursor::nextLeft(ListT<CegoField>** pFLA, int offset, int size)
{
    CegoJoinObject *pJO = (CegoJoinObject*)_pCO;
    
    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
    
    bool notFound = true;
    while ( _evalPredicate && _moreLeft && notFound )
    {
	
	if ( pJO->getPredDesc()->eval(0,
				      0,
				      pFLA,
				      offset,
				      0) )
	{
	    notFound = false;
	}
	else
	{
	    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
	}			      
    }
}

void CegoDistCursor::finishCaching()
{    
    if ( _useCache && _pCache && _pCacheList && _isCached == false && _pTC && _isFirst == false )
    {	
	CegoDataPointer dp;
	
	while ( _pTC->getNext(_cacheSchema, dp) && _pCacheList )
	{

	    ListT<CegoFieldValue> staticFieldList;
	    CegoField* pF = _cacheSchema.First();
	    while ( pF )
	    {			
		staticFieldList.Insert(pF->getValue().getLocalCopy());			       
		_cacheEntrySize += pF->getValue().size();
		pF = _cacheSchema.Next();
	    }
	    
	    if ( _pCache->getMaxSize() > _cacheEntrySize )
	    {
		_pCacheList->Insert(staticFieldList);
	    }
	    else
	    {
		delete _pCacheList;
		_pCacheList = 0;
	    }
	}
       	    
	if ( _pCacheList )
	{
	    // put table cache entry into cache
	    _pCache->addEntry( _tableName, _pCacheList );
	    _isCached = true;
	    delete _pCacheList;
	    _pCacheList = 0;
	}
    }
}

void CegoDistCursor::reset()
{
    _isFirst=true;
    _moreTuple = false;

    if ( _pCO->getType() == CegoObject::VIEW )
    {
	_pSelect->reset(true);
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
	if ( _isLocal )
	{
	    if ( _pTC )
		_pTC->abort();

	    // we have to check, if cache was created
	    if ( _isCached == false && _pCacheList )
	    {
		delete _pCacheList;
		_pCacheList = 0;
	    }
	    
	    if ( _pCacheArray && _pCache )
	    {
		_pCache->releaseEntry( _tableName);
		_pCacheArray = 0;
		_isCached = 0;
	    }
	}
	else
	{
	    if ( _pSH )
		_pDBMng->releaseSession(_pSH);
	}
    }
    else if ( _pCO->getType() == CegoObject::ALIAS )
    {
	if ( _pTCTab )
	    _pTCTab->reset();
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {	
	if ( _pTCLeft )
	    _pTCLeft->reset();
	if ( _pTCRight )
	    _pTCRight->reset();
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {
	if ( _pC )
	    _pC->reset();
    }

    unuseCursorObject();
}

void CegoDistCursor::sysSetup()
{    
    Chain tableSet = _pDBMng->getTabSetName(_tabSetId);	
    if ( _tableName == Chain(SYSTAB_TABLE_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::TABLE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_PROC_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::PROCEDURE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_VIEW_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::VIEW, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_INDEX_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::AVLTREE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_BTREE_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::BTREE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_KEY_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::FKEY, _sysObjList);
    }
    else
    {
	_pC = _pGTM->getObjectCursor(_tabSetId, _tableName, _tableName, CegoObject::SYSTEM);
    }
}

void CegoDistCursor::joinSetup(const CegoAttrCond& attrCond)
{
    CegoJoinObject *pJO = (CegoJoinObject*)_pCO;
        
    ListT<CegoField> outerSchema;
    ListT<CegoField> innerSchema;
       
    if ( pJO->getJoinType() == CegoJoinObject::INNER || pJO->getJoinType() == CegoJoinObject::LEFTOUTER ) 
    {
	outerSchema = pJO->getLeftObject()->getSchema();
	innerSchema = pJO->getRightObject()->getSchema();

	if ( pJO->getJoinType() == CegoJoinObject::INNER )
	{
	    _outerCond = attrCond.getFilterCond(outerSchema, false);
	    _innerCond = attrCond.getFilterCond(innerSchema, false);

	    _doEval = false;
	}
	else
	{
	    _outerCond = attrCond.getFilterCond(outerSchema, true);
	    _innerCond = attrCond.getFilterCond(innerSchema, true);

	    _doEval = true;
	}    
    }
    else if (  pJO->getJoinType() == CegoJoinObject::RIGHTOUTER )
    {
	innerSchema = pJO->getLeftObject()->getSchema();
        outerSchema = pJO->getRightObject()->getSchema();

	_outerCond = attrCond.getFilterCond(outerSchema, true);
	_innerCond = attrCond.getFilterCond(innerSchema, true);

	// for right outer join, we have to evaluate the where condition outside the join 
	_doEval = true;	
    }

    // cout << "Setting up join with " << pJO->getPredDesc()->toChain() << endl;

    ListT<CegoPredicate*> conjunctionList;

    CegoQueryHelper::createConjunctionList(pJO->getPredDesc(), &conjunctionList);

    _evalPredicate = false;

    CegoPredicate **pPred = conjunctionList.First();
    while ( pPred )
    {
	CegoAttrCond ac;
	// CegoQueryHelper::AttrCondMatch m  = CegoQueryHelper::checkAttrCond(ac, pJO->getPredDesc(), innerSchema, &outerSchema, 1, 0);

	CegoQueryHelper::AttrCondMatch m  = CegoQueryHelper::checkAttrCond(ac, *pPred, innerSchema, &outerSchema, 1, 0);

	if ( m == CegoQueryHelper::COMPLETE )
	{
	    _innerCond = _innerCond + ac;
	}
	else if ( m == CegoQueryHelper::PARTIAL )
	{
	    // for Partial conditions, we have to evaluate in any case
	    _evalPredicate = true;
	    _innerCond = _innerCond + ac;
	}
	else
	{
	    // for inapp conditions, we have to evaluate the whole condition
	    _evalPredicate = true;
	}
	pPred = conjunctionList.Next();
    }

    _isAttrCondValid = true;    
}

Element* CegoDistCursor::getPlan()
{
    Element *pCursorPlan = new Element(XML_JOIN_ELEMENT);

    pCursorPlan->setAttribute(XML_TABLENAME_ATTR, _tableName);       
    pCursorPlan->setAttribute(XML_NAME_ATTR, _tableAlias);

    if ( _pCO->getType() == CegoObject::VIEW )
    {
	pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_VIEW_VALUE);

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Getting view plan for ") + _tableAlias);
#endif

	
	pCursorPlan->addContent( _pSelect->getPlan() );


#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("View plan for ") + _tableAlias + Chain(" retrieved"));
#endif

    }
    else if ( _pCO->getType() == CegoObject::ALIAS )
    {
	if ( _pTCTab )
	{
	    pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_ALIAS_VALUE);
	    pCursorPlan->setAttribute(XML_NAME_ATTR, _pCO->getName());
	    // Chain tableName = _pTCTab->getPlan()->getAttributeValue(XML_TABLENAME_ATTR);
	    // Chain aliasName = _pTCTab->getPlan()->getAttributeValue(XML_NAME_ATTR);
	    // Chain tableType = _pTCTab->getPlan()->getAttributeValue(XML_TABLETYPE_ATTR);
	    // cout << "TableName=" << tableName << endl;
	    // cout << "AliasName=" << aliasName << endl;
	    // cout << "TableType=" << tableType << endl;
	    
	    pCursorPlan->addContent( _pTCTab->getPlan() );
	}
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
	pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_TABLE_VALUE);
	
	if ( _idxMatch == CegoAttrCond::FULL)
	{
	    pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("full index trace on ") + _cursorCond.toChain()); 
	}
	else if( _idxMatch == CegoAttrCond::PART )
	{
	    pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("index support on ") + _cursorCond.toChain() + Chain(" using index ") + _pTC->getIndexName()); 
	}
	else
	{
	    if ( _cursorCond.numComp() > 0 )
		pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("full table scan using condition ") + _cursorCond.toChain());
	    else
		pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("full table scan with no condition "));
	}	
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {
	CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	
	if ( pJCO->getJoinType() == CegoJoinObject::INNER )	    
	    pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_INNERJOIN_VALUE);
	else if ( pJCO->getJoinType() == CegoJoinObject::LEFTOUTER)	    
	    pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_LEFTOUTERJOIN_VALUE);
	else if ( pJCO->getJoinType() == CegoJoinObject::RIGHTOUTER)	    
	    pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_RIGHTOUTERJOIN_VALUE);

	if ( _evalPredicate )	    
	    pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("with condition evaluation"));
	else
	    pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("no condition evaluation"));

	if ( _pTCLeft )
	{
	    _pTCLeft->distSetup(_outerCond, _pFLA);
	}
	if ( _pTCRight )
	{	    
	    if ( _isAttrCondValid )
	    {
		_pTCRight->distSetup(_innerCond, _pFLA);
	    }
	    else
	    {
		_pTCRight->distSetup(_pFLA);
	    }	    
	}
	if ( _pTCLeft )
	    pCursorPlan->addContent( _pTCLeft->getPlan() );
	if ( _pTCRight )
	    pCursorPlan->addContent( _pTCRight->getPlan() );
    }
    else
    {
	delete pCursorPlan;
	throw Exception(EXLOC, Chain("Invalid content type"));		
    }

    return pCursorPlan;
}
