///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoCondition.cc
// ----------------
// Cego condition description
//
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoCondDesc
// 
// Description: Condition container class used for queries and stores procedures
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoCondition.h"
#include "CegoXMLdef.h"
#include "CegoQueryHelper.h"
#include "CegoDatabaseFormater.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoCondition::CegoCondition(char *buf, CegoDistManager *pGTM, CegoProcBlock *pBlock, int tabSetId)
{
    _pLeft=0;
    _pRight=0;
    decode(buf, pGTM, pBlock, tabSetId);
}

CegoCondition::CegoCondition(CegoCondition::CondType ct)
{
    _condType = ct;
    _pLeft=0;
    _pRight=0;
}

CegoCondition::~CegoCondition()
{
    if (_pLeft)
    {
	delete _pLeft;
    }
    if (_pRight)
    {
	delete _pRight;
    }
}

void CegoCondition::setTabSetId(int tabSetId)
{
    if (_pLeft)
    {
	_pLeft->setTabSetId(tabSetId);
    }
    if (_pRight)
    {
	_pRight->setTabSetId(tabSetId);
    }
}

void CegoCondition::getPlanList(ListT<Element*>& planList)
{
    if (_pLeft)
    {
	_pLeft->getPlanList(planList);
    }
    if (_pRight)
    {
	_pRight->getPlanList(planList);
    }
}

void CegoCondition::cleanUp()
{
    if (_pLeft)
    {
	_pLeft->cleanUp();
    }
    if (_pRight)
    {
	_pRight->cleanUp();
    }
}

CegoPredicate* CegoCondition::Left()
{
    return _pLeft;
}

CegoPredicate* CegoCondition::Right()
{
    return _pRight;
}

void CegoCondition::setLeft(CegoPredicate* pP)
{
    _pLeft = pP;
}

void CegoCondition::setRight(CegoPredicate* pP)
{
    _pRight = pP;
}

void CegoCondition::setCondType(CegoCondition::CondType ct)
{
    _condType = ct;
}

CegoCondition::CondType CegoCondition::getCondType()
{
    return _condType;
}

void CegoCondition::encode(char *buf, CegoProcBlock *pBlock)
{
    char* pP = (char*)buf;

    memcpy( pP, &_condType, sizeof(CegoCondition::CondType));
    pP = pP + sizeof(CegoCondition::CondType);

    switch (_condType )
    {
    case CegoCondition::AND:
    case CegoCondition::OR:
    {
	_pLeft->encode(pP, pBlock);
	pP = pP + _pLeft->getEncodingLength(pBlock);
	_pRight->encode(pP, pBlock);
	pP = pP + _pRight->getEncodingLength(pBlock);
	break;
    }
    case CegoCondition::PRED:
    {
	_pLeft->encode(pP, pBlock);
	pP = pP + _pLeft->getEncodingLength(pBlock);	
	break;
    }
    }
}

void CegoCondition::decode(char *buf, CegoDistManager *pGTM, CegoProcBlock *pBlock, int tabSetId)
{
    if ( _pLeft ) 
	delete _pLeft;
    if ( _pRight ) 
	delete _pRight;
    
    char* pP = (char*)buf;

    memcpy( &_condType, pP, sizeof(CegoCondition::CondType));
    pP = pP + sizeof(CegoCondition::CondType);
    
    switch (_condType )
    {
    case CegoCondition::AND:
    case CegoCondition::OR:
    {
	_pLeft = new CegoPredicate(pP, pGTM, pBlock, tabSetId);
	pP = pP + _pLeft->getEncodingLength(pBlock);
	_pRight = new CegoPredicate(pP, pGTM, pBlock, tabSetId);
	pP = pP + _pRight->getEncodingLength(pBlock);
	break;
    }
    case CegoCondition::PRED:
    {
	_pLeft = new CegoPredicate(pP, pGTM, pBlock, tabSetId);
	pP = pP + _pLeft->getEncodingLength(pBlock);
	break;
    }
    }
}

void CegoCondition::clearAttrCache()
{    
    switch (_condType )
    {
    case CegoCondition::AND:
    case CegoCondition::OR:
    {
	_pLeft->clearAttrCache();
	_pRight->clearAttrCache();
	break;
    }
    case CegoCondition::PRED:
    {
	_pLeft->clearAttrCache();
	break;
    }
    }
}

int CegoCondition::getEncodingLength(CegoProcBlock *pBlock) const
{
    int len = 0;

    len += sizeof(CegoCondition::CondType);
    
    switch (_condType )
    {
    case CegoCondition::AND:
    case CegoCondition::OR:
    {
	len += _pLeft->getEncodingLength(pBlock);
	len += _pRight->getEncodingLength(pBlock);
	break;
    }
    case CegoCondition::PRED:
    {
	len += _pLeft->getEncodingLength(pBlock);
	break;
    }
    }
    return len;
}

ListT<CegoAttrDesc*> CegoCondition::getAttrRefList() const
{
    ListT<CegoAttrDesc*> al;

    switch( _condType )
    {
    case CegoCondition::AND:
    case CegoCondition::OR:
    {
	al = _pLeft->getAttrRefList();
	al += _pRight->getAttrRefList();
	break;
    }
    case CegoCondition::PRED:
    {
	al += _pLeft->getAttrRefList();
	break;
    }
    }

    return al;
}

int CegoCondition::evalReferences(CegoContentObject *pCO, const ListT<CegoField>& fl)
{
    int refCount = 0;

    switch( _condType )
    {
    case CegoCondition::AND:
    case CegoCondition::OR:
    {
	refCount += _pLeft->evalReferences(pCO, fl);
	refCount += _pRight->evalReferences(pCO, fl);
	break;
    }
    case CegoCondition::PRED:
    {
	refCount += _pLeft->evalReferences(pCO, fl);
	break;
    }
    }

    return refCount;
}

ListT<CegoAggregation*> CegoCondition::getAggregationList()
{
    ListT<CegoAggregation*> aggList;	

    if (_pLeft)
    {
	aggList = aggList + _pLeft->getAggregationList();
    }
    if (_pRight)
    {
	aggList = aggList + _pRight->getAggregationList();
    }

    return aggList;
}

void CegoCondition::getFieldList(ListT<CegoField>& fl, ListT<CegoField>** pFLA) const
{    
    switch( _condType )
    {
    case CegoCondition::AND:
    case CegoCondition::OR:
    {
	_pLeft->getFieldList(fl, pFLA);
	_pRight->getFieldList(fl, pFLA);
	break;
    }
    case CegoCondition::PRED:
    {
	_pLeft->getFieldList(fl, pFLA);
	break;
    }
    }
}

CegoField CegoCondition::evalField(const ListT<CegoField>& fl, bool graceful) const
{
    CegoField f = CegoField( Chain("COND"), Chain("COND"), Chain("COND"), BOOL_TYPE, sizeof(char));
    return f;
}

CegoFieldValue CegoCondition::evalFieldValue(ListT<CegoField>** pFLA, CegoProcBlock *pBlock) const
{
    bool isTrue=false;
    switch( _condType )
    {
    case CegoCondition::AND:
    {
	isTrue = ( _pLeft->eval(0, 0, pFLA, 0, pBlock)
		 && _pRight->eval(0, 0, pFLA, 0, pBlock));
	break;
    }
    case CegoCondition::OR:
    {
	isTrue = ( _pLeft->eval(0, 0, pFLA, 0, pBlock)
		 || _pRight->eval(0, 0, pFLA, 0, pBlock));
	break;
    }
    case CegoCondition::PRED:
    {
	isTrue =  _pLeft->eval(0, 0, pFLA, 0, pBlock);
	break;
    }
    }

    if ( isTrue )
	return CegoFieldValue(BOOL_TYPE, Chain("true"));
    return CegoFieldValue(BOOL_TYPE, Chain("false"));
}

void CegoCondition::setAggregationValue(ListT<CegoField>& jfl)
{
    if (_pLeft)
    {
	_pLeft->setAggregationValue(jfl);
    }
    if (_pRight)
    {
	_pRight->setAggregationValue(jfl);
    }
}

CegoCondition* CegoCondition::clone(bool isAttrRef)
{
    CegoCondition* pCD = new CegoCondition(_condType);
    
    if ( _pLeft )
	pCD->setLeft(_pLeft->clone(isAttrRef));
    else
	pCD->setLeft(0);

    if ( _pRight )
	pCD->setRight(_pRight->clone(isAttrRef));
    else
	pCD->setRight(0);
    
    return pCD;
}

Chain CegoCondition::getId(CegoProcBlock *pBlock) const
{
    Chain s;
    s = Chain("c:");

    switch( _condType )
    {
    case CegoCondition::AND:
    {
	s += _pLeft->getId(pBlock) + Chain("&") + _pRight->getId(pBlock);
	break;
    }
    case CegoCondition::OR:
    {
	s += _pLeft->getId(pBlock) + Chain("|") + _pRight->getId(pBlock);
	break;
    }
    case CegoCondition::PRED:
    {
	s += _pLeft->getId(pBlock);
	break;
    }
    }
    return s;
}


Chain CegoCondition::dbFormat(CegoDatabaseFormater *pForm)
{
    return pForm->formatCondition(_condType, _pLeft, _pRight);
}


Chain CegoCondition::toChain(int defTabSetId) const
{
    Chain s;

    switch( _condType )
    {
    case CegoCondition::AND:
    {
	s = _pLeft->toChain(defTabSetId) + Chain(" and ") + _pRight->toChain(defTabSetId);
	break;
    }
    case CegoCondition::OR:
    {
	s = _pLeft->toChain(defTabSetId) + Chain(" or ") + _pRight->toChain(defTabSetId);
	break;
    }
    case CegoCondition::PRED:
    {
	s = _pLeft->toChain(defTabSetId);
	break;
    }
    }
    return s;
}
