#!/usr/bin/env node

/**
 * IPC Documentation Generator
 *
 * Automatically scans the codebase for all IPC channel registrations
 * (ipcMain.handle and ipcMain.on) and generates documentation.
 *
 * Usage: node scripts/generateIpcDocs.js
 */

const fs = require('node:fs');
const path = require('node:path');
const { execSync } = require('node:child_process');

const APP_DIR = path.join(__dirname, '..', 'app');
const DOCS_OUTPUT = path.join(__dirname, '..', 'docs-site', 'docs', 'development', 'ipc-api-generated.md');
const GITHUB_REPO = 'https://github.com/IsmaelMartinez/teams-for-linux';

// IPC channel categories based on file location
const CATEGORIES = {
  'app/index.js': 'Core Application',
  'app/notifications/': 'Notifications',
  'app/notificationSystem/': 'Notifications (Custom)',
  'app/screenSharing/': 'Screen Sharing',
  'app/idle/': 'Idle Monitoring',
  'app/partitions/': 'Partitions & Zoom',
  'app/menus/': 'Menus & Tray',
  'app/customBackground/': 'Custom Background',
  'app/mainAppWindow/': 'Main Window',
  'app/login/': 'Authentication',
  'app/connectionManager/': 'Connection Management',
  'app/incomingCallToast/': 'Incoming Calls',
  'app/graphApi/': 'Microsoft Graph API',
};

function findCategory(filePath) {
  for (const [pattern, category] of Object.entries(CATEGORIES)) {
    if (filePath.includes(pattern)) {
      return category;
    }
  }
  return 'Other';
}

function extractIpcChannels() {
  console.log('Scanning for IPC channels...');

  // Check if ripgrep is available
  try {
    execSync('rg --version', { stdio: 'ignore' });
  } catch (error) {
    // ripgrep not found - provide helpful error message
    console.error('\n❌ Error: ripgrep (rg) is not installed or not in your PATH.');
    console.error(`   Reason: ${error.message}`);
    console.error('This script requires ripgrep to scan for IPC channels.\n');
    console.error('Please install it from: https://github.com/BurntSushi/ripgrep#installation\n');
    process.exit(1);
  }

  const grepCommand = String.raw`rg -n "ipcMain\.(handle|on)\(" --type js ${APP_DIR}`;

  let output;
  try {
    output = execSync(grepCommand, { encoding: 'utf8' });
  } catch (error) {
    // ripgrep failed (could be no matches found or actual error)
    if (error.status === 1) {
      // Exit code 1 means no matches found
      console.log('No IPC channels found.');
      return [];
    }
    // Other errors (exit code 2+) are actual failures
    console.error('Error running ripgrep:', error.message);
    process.exit(1);
  }

  return output.trim().split('\n').map(line => {
    const match = line.match(/^([^:]+):(\d+):(.+)$/);
    if (!match) return null;

    const [, filePath, lineNumber, content] = match;
    const channelMatch = content.match(/ipcMain\.(handle|on)\(\s*["']([^"']+)["']/);
    if (!channelMatch) return null;

    const [, type, channelName] = channelMatch;
    const relativePath = path.relative(path.join(__dirname, '..'), path.resolve(filePath));
    const description = extractDescription(filePath, Number.parseInt(lineNumber, 10) - 1);

    return {
      name: channelName,
      type: type === 'handle' ? 'Request/Response' : 'Event',
      category: findCategory(relativePath),
      file: relativePath,
      lineNumber,
      description,
    };
  }).filter(Boolean);
}

function extractDescription(filePath, lineIndex) {
  const fileLines = fs.readFileSync(filePath, 'utf8').split('\n');
  const commentLines = [];

  // Look up to 5 lines above for comments
  for (let i = lineIndex - 1; i >= Math.max(0, lineIndex - 5); i--) {
    const line = fileLines[i].trim();
    if (line.startsWith('//') || line.startsWith('*')) {
      commentLines.unshift(line.replace(/^(\/\/|\*)\s*/, ''));
    } else if (line && !line.startsWith('/*') && line !== '*/') {
      break;
    }
  }

  return commentLines.join(' ').trim() || 'No description available';
}

function generateMarkdown(channels) {
  // Group channels by category
  const channelsByCategory = channels.reduce((acc, channel) => {
    if (!acc[channel.category]) {
      acc[channel.category] = [];
    }
    acc[channel.category].push(channel);
    return acc;
  }, {});

  const sortedCategories = Object.keys(channelsByCategory).sort();

  let markdown = `# IPC API Reference (Auto-Generated)

> **Note**: This file is auto-generated by \`scripts/generateIpcDocs.js\`. Do not edit manually.
>
> **Last Generated**: ${new Date().toISOString()}

## Overview

This document lists all IPC (Inter-Process Communication) channels registered in the application.

- **Request/Response** channels use \`ipcMain.handle()\` and expect a return value
- **Event** channels use \`ipcMain.on()\` for fire-and-forget notifications

**Total Channels**: ${channels.length}

---

`;

  for (const category of sortedCategories) {
    const categoryChannels = channelsByCategory[category];

    markdown += `## ${category}\n\n`;
    markdown += `| Channel | Type | Description | Location |\n`;
    markdown += `|---------|------|-------------|----------|\n`;

    for (const channel of categoryChannels.sort((a, b) => a.name.localeCompare(b.name))) {
      const location = `[\`${channel.file}:${channel.lineNumber}\`](${GITHUB_REPO}/blob/develop/${channel.file}#L${channel.lineNumber})`;
      markdown += `| \`${channel.name}\` | ${channel.type} | ${channel.description} | ${location} |\n`;
    }

    markdown += '\n';
  }

  markdown += `---

## Channel Security

All IPC channels are validated through the security layer in \`app/security/ipcValidator.js\`.
See the [IPC Channel Validation documentation](./security-architecture.md#ipc-channel-validation) for more information.

## Adding New Channels

When adding a new IPC channel:

1. Register the channel with \`ipcMain.handle()\` or \`ipcMain.on()\`
2. Add the channel to the allowlist in \`app/security/ipcValidator.js\`
3. Add a comment above the registration describing its purpose
4. Run \`npm run generate-ipc-docs\` to update this documentation

---

*Generated by \`scripts/generateIpcDocs.js\`*
`;

  return markdown;
}

function main() {
  console.log('IPC Documentation Generator');
  console.log('============================\n');

  const channels = extractIpcChannels();
  console.log(`Found ${channels.length} IPC channels\n`);

  const markdown = generateMarkdown(channels);

  // Ensure output directory exists
  const outputDir = path.dirname(DOCS_OUTPUT);
  if (!fs.existsSync(outputDir)) {
    fs.mkdirSync(outputDir, { recursive: true });
  }

  fs.writeFileSync(DOCS_OUTPUT, markdown, 'utf8');
  console.log(`✅ Documentation generated: ${DOCS_OUTPUT}`);

  // Output summary by category
  console.log('\nChannel Summary:');
  const summary = channels.reduce((acc, ch) => {
    acc[ch.category] = (acc[ch.category] || 0) + 1;
    return acc;
  }, {});

  Object.entries(summary).sort().forEach(([category, count]) => {
    console.log(`  ${category}: ${count} channels`);
  });
}

main();
