#include <Functions/IFunction.h>
#include <Functions/FunctionFactory.h>
#include <DataTypes/DataTypesNumber.h>


namespace DB
{
namespace
{

/** ignore(...) is a function that takes any arguments, and always returns 0.
  */
class FunctionIgnore : public IFunction
{
public:
    static constexpr auto name = "ignore";
    static FunctionPtr create(ContextPtr)
    {
        return std::make_shared<FunctionIgnore>();
    }

    bool isVariadic() const override
    {
        return true;
    }
    size_t getNumberOfArguments() const override
    {
        return 0;
    }

    bool useDefaultImplementationForNulls() const override { return false; }
    bool isSuitableForConstantFolding() const override { return false; }
    bool isSuitableForShortCircuitArgumentsExecution(const DataTypesWithConstInfo & /*arguments*/) const override { return false; }

    /// We should never return LowCardinality result, cause we declare that result is always constant zero.
    /// (in getResultIfAlwaysReturnsConstantAndHasArguments)
    bool useDefaultImplementationForLowCardinalityColumns() const override { return false; }

    bool useDefaultImplementationForSparseColumns() const override { return false; }

    String getName() const override
    {
        return name;
    }

    DataTypePtr getReturnTypeImpl(const DataTypes & /*arguments*/) const override
    {
        return std::make_shared<DataTypeUInt8>();
    }

    ColumnPtr executeImpl(const ColumnsWithTypeAndName &, const DataTypePtr &, size_t input_rows_count) const override
    {
        return DataTypeUInt8().createColumnConst(input_rows_count, 0u);
    }
};

}

void registerFunctionIgnore(FunctionFactory & factory)
{
    factory.registerFunction<FunctionIgnore>();
}

}
