#pragma once

#include <IO/ReadBuffer.h>
#include <IO/WriteBuffer.h>
#include "IMySQLReadPacket.h"
#include "IMySQLWritePacket.h"
#include "IO/MySQLPacketPayloadReadBuffer.h"
#include <base/shared_ptr_helper.h>

namespace DB
{

namespace MySQLProtocol
{

/* Writes and reads packets, keeping sequence-id.
 * Throws ProtocolError, if packet with incorrect sequence-id was received.
 */
class PacketEndpoint : public shared_ptr_helper<PacketEndpoint>
{
public:
    uint8_t & sequence_id;
    ReadBuffer * in;
    WriteBuffer * out;

    MySQLPacketPayloadReadBuffer getPayload();

    void receivePacket(IMySQLReadPacket & packet);

    bool tryReceivePacket(IMySQLReadPacket & packet, UInt64 millisecond = 0);

    /// Sets sequence-id to 0. Must be called before each command phase.
    void resetSequenceId();

    template<class T>
    void sendPacket(const T & packet, bool flush = false)
    {
        static_assert(std::is_base_of<IMySQLWritePacket, T>());
        packet.writePayload(*out, sequence_id);
        if (flush)
            out->next();
    }

    /// Converts packet to text. Is used for debug output.
    static String packetToText(const String & payload);

protected:
    /// For writing.
    PacketEndpoint(WriteBuffer & out_, uint8_t & sequence_id_);

    /// For reading and writing.
    PacketEndpoint(ReadBuffer & in_, WriteBuffer & out_, uint8_t & sequence_id_);

    friend struct shared_ptr_helper<PacketEndpoint>;
};

using PacketEndpointPtr = std::shared_ptr<PacketEndpoint>;

}

}
