---
toc_title: "Функции для работы с индексами H3"
---

# Функции для работы с индексами H3 {#h3index}

[H3](https://eng.uber.com/h3/) — это система геокодирования, которая делит поверхность Земли на равные шестиугольные ячейки. Система поддерживает иерархию (вложенность) ячеек, т.е. каждый "родительский" шестиугольник может быть поделен на семь одинаковых вложенных "дочерних" шестиугольников, и так далее.

Уровень вложенности называется "разрешением" и может принимать значение от `0` до `15`, где `0` соответствует "базовым" ячейкам самого верхнего уровня (наиболее крупным).

Для каждой точки, имеющей широту и долготу, можно получить 64-битный индекс H3, соответствующий номеру шестриугольной ячейки, где эта точка находится.

Индексы H3 используются, в основном, для геопозиционирования и расчета расстояний.

## h3IsValid {#h3isvalid}

Проверяет корректность [H3](#h3index)-индекса.

**Синтаксис**

``` sql
h3IsValid(h3index)
```

**Параметр**

-   `h3index` — идентификатор шестриугольника. Тип данных: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемые значения**

-   1 — число является H3-индексом.
-   0 — число не является H3-индексом.

Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3IsValid(630814730351855103) AS h3IsValid;
```

Результат:

``` text
┌─h3IsValid─┐
│         1 │
└───────────┘
```

## h3GetResolution {#h3getresolution}

Извлекает разрешение [H3](#h3index)-индекса.

**Синтаксис**

``` sql
h3GetResolution(h3index)
```

**Параметр**

-   `h3index` — идентификатор шестиугольника. Тип данных: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемые значения**

-   Разрешение сетки. Диапазон значений: `[0, 15]`.
-   Для несуществующего идентификатора может быть возвращено произвольное значение. Используйте [h3IsValid](#h3isvalid) для проверки идентификаторов.

Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3GetResolution(639821929606596015) AS resolution;
```

Результат:

``` text
┌─resolution─┐
│         14 │
└────────────┘
```

## h3EdgeAngle {#h3edgeangle}

Рассчитывает средний размер стороны шестиугольника [H3](#h3index) в градусах.

**Синтаксис**

``` sql
h3EdgeAngle(resolution)
```

**Параметр**

-   `resolution` — требуемое разрешение индекса. Тип данных: [UInt8](../../../sql-reference/data-types/int-uint.md). Диапазон возможных значений: `[0, 15]`.

**Возвращаемое значение**

-   Средняя длина стороны шестиугольника [H3](#h3index) в градусах. Тип данных: [Float64](../../../sql-reference/data-types/float.md).

**Пример**

Запрос:

``` sql
SELECT h3EdgeAngle(10) AS edgeAngle;
```

Результат:

``` text
┌───────h3EdgeAngle(10)─┐
│ 0.0005927224846720883 │
└───────────────────────┘
```

## h3EdgeLengthM {#h3edgelengthm}

Рассчитывает средний размер стороны шестиугольника [H3](#h3index) в метрах.

**Синтаксис**

``` sql
h3EdgeLengthM(resolution)
```

**Параметр**

-   `resolution` — требуемое разрешение индекса. Тип данных — [UInt8](../../../sql-reference/data-types/int-uint.md). Диапазон возможных значений — `[0, 15]`.

**Возвращаемое значение**

-   Средняя длина стороны шестиугольника H3 в метрах, тип — [Float64](../../../sql-reference/data-types/float.md).

**Пример**

Запрос:

``` sql
SELECT h3EdgeLengthM(15) AS edgeLengthM;
```

Результат:

``` text
┌─edgeLengthM─┐
│ 0.509713273 │
└─────────────┘
```

## geoToH3 {#geotoh3}

Возвращает [H3](#h3index) индекс точки `(lon, lat)` с заданным разрешением.

**Синтаксис**

``` sql
geoToH3(lon, lat, resolution)
```

**Аргументы**

-   `lon` — географическая долгота. Тип данных — [Float64](../../../sql-reference/data-types/float.md).
-   `lat` — географическая широта. Тип данных — [Float64](../../../sql-reference/data-types/float.md).
-   `resolution` — требуемое разрешение индекса. Тип данных — [UInt8](../../../sql-reference/data-types/int-uint.md). Диапазон возможных значений — `[0, 15]`.

**Возвращаемые значения**

-   Порядковый номер шестиугольника.
-   0 в случае ошибки.

Тип данных: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT geoToH3(37.79506683, 55.71290588, 15) AS h3Index;
```

Результат:

``` text
┌────────────h3Index─┐
│ 644325524701193974 │
└────────────────────┘
```

## h3ToGeo {#h3togeo}

Возвращает географические координаты долготы и широты центра шестиугольника, соответствующие указанному [H3](#h3index)-индексу.

**Синтаксис**

``` sql
h3ToGeo(h3Index)
```

**Аргументы**

-   `h3Index` — [H3](#h3index)-индекс. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемые значения**

-   кортеж из двух значений: `tuple(lon,lat)`, где `lon` — долгота [Float64](../../../sql-reference/data-types/float.md), `lat` — широта [Float64](../../../sql-reference/data-types/float.md).

**Пример**

Запрос:

``` sql
SELECT h3ToGeo(644325524701193974) coordinates;
```

Результат:

``` text
┌─coordinates───────────────────────────┐
│ (37.79506616830252,55.71290243145668) │
└───────────────────────────────────────┘
```

## h3ToGeoBoundary {#h3togeoboundary}

Возвращает массив пар `(lon, lat)`, который соответствует границе указанного H3 индекса.

**Синтаксис**

``` sql
h3ToGeoBoundary(h3Index)
```

**Аргументы**

-   `h3Index` — H3 индекс. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемые значения**

-    Массив пар '(lon, lat)'.
Тип: [Array](../../../sql-reference/data-types/array.md)([Float64](../../../sql-reference/data-types/float.md), [Float64](../../../sql-reference/data-types/float.md)).


**Пример**

Запрос:

``` sql
SELECT h3ToGeoBoundary(644325524701193974) AS coordinates;
```

Результат:

``` text
┌─h3ToGeoBoundary(599686042433355775)────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [(37.2713558667319,-121.91508032705622),(37.353926450852256,-121.8622232890249),(37.42834118609435,-121.92354999630156),(37.42012867767779,-122.03773496427027),(37.33755608435299,-122.090428929044),(37.26319797461824,-122.02910130919001)] │
└────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3kRing {#h3kring}

Возвращает [H3](#h3index)-индексы шестиугольника в радиусе `k` от данного в произвольном порядке.

**Синтаксис**

``` sql
h3kRing(h3index, k)
```

**Аргументы**

-   `h3index` — идентификатор шестиугольника. Тип данных: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `k` — радиус. Тип данных: [целое число](../../../sql-reference/data-types/int-uint.md)

**Возвращаемые значения**

-   Массив из H3-индексов.

Тип данных: [Array](../../../sql-reference/data-types/array.md)([UInt64](../../../sql-reference/data-types/int-uint.md)).

**Пример**

Запрос:

``` sql
SELECT arrayJoin(h3kRing(644325529233966508, 1)) AS h3index;
```

Результат:

``` text
┌────────────h3index─┐
│ 644325529233966508 │
│ 644325529233966497 │
│ 644325529233966510 │
│ 644325529233966504 │
│ 644325529233966509 │
│ 644325529233966355 │
│ 644325529233966354 │
└────────────────────┘
```

## h3GetBaseCell {#h3getbasecell}

Определяет номер базовой (верхнеуровневой) шестиугольной [H3](#h3index)-ячейки для указанной ячейки.

**Синтаксис**

``` sql
h3GetBaseCell(index)
```

**Параметр**

-   `index` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   Индекс базовой шестиугольной ячейки.

Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3GetBaseCell(612916788725809151) AS basecell;
```

Результат:

``` text
┌─basecell─┐
│       12 │
└──────────┘
```

## h3HexAreaM2 {#h3hexaream2}

Определяет среднюю площадь шестиугольной [H3](#h3index)-ячейки заданного разрешения в квадратных метрах.

**Синтаксис**

``` sql
h3HexAreaM2(resolution)
```

**Параметр**

-   `resolution` — разрешение. Диапазон: `[0, 15]`. Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   Площадь в квадратных метрах. Тип: [Float64](../../../sql-reference/data-types/float.md).

**Пример**

Запрос:

``` sql
SELECT h3HexAreaM2(13) AS area;
```

Результат:

``` text
┌─area─┐
│ 43.9 │
└──────┘
```

## h3IndexesAreNeighbors {#h3indexesareneighbors}

Определяет, являются ли [H3](#h3index)-ячейки соседями.

**Синтаксис**

``` sql
h3IndexesAreNeighbors(index1, index2)
```

**Аргументы**

-   `index1` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `index2` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   `1` — ячейки являются соседями.
-   `0` — ячейки не являются соседями.

Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3IndexesAreNeighbors(617420388351344639, 617420388352655359) AS n;
```

Результат:

``` text
┌─n─┐
│ 1 │
└───┘
```

## h3ToChildren {#h3tochildren}

Формирует массив дочерних (вложенных) [H3](#h3index)-ячеек для указанной ячейки.

**Синтаксис**

``` sql
h3ToChildren(index, resolution)
```

**Аргументы**

-   `index` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `resolution` — разрешение. Диапазон: `[0, 15]`. Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   Массив дочерних H3-ячеек.

Тип: [Array](../../../sql-reference/data-types/array.md)([UInt64](../../../sql-reference/data-types/int-uint.md)).

**Пример**

Запрос:

``` sql
SELECT h3ToChildren(599405990164561919, 6) AS children;
```

Результат:

``` text
┌─children───────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [603909588852408319,603909588986626047,603909589120843775,603909589255061503,603909589389279231,603909589523496959,603909589657714687] │
└────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3ToParent {#h3toparent}

Определяет родительскую (более крупную) [H3](#h3index)-ячейку, содержащую указанную ячейку.

**Синтаксис**

``` sql
h3ToParent(index, resolution)
```

**Аргументы**

-   `index` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `resolution` — разрешение. Диапазон: `[0, 15]`. Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   Индекс родительской H3-ячейки.

Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3ToParent(599405990164561919, 3) AS parent;
```

Результат:

``` text
┌─────────────parent─┐
│ 590398848891879423 │
└────────────────────┘
```

## h3ToString {#h3tostring}

Преобразует [H3](#h3index)-индекс из числового представления `H3Index` в строковое.

``` sql
h3ToString(index)
```

**Параметр**

-   `index` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   Строковое представление H3-индекса.

Тип: [String](../../../sql-reference/data-types/string.md).

**Пример**

Запрос:

``` sql
SELECT h3ToString(617420388352917503) AS h3_string;
```

Результат:

``` text
┌─h3_string───────┐
│ 89184926cdbffff │
└─────────────────┘
```

## stringToH3 {#stringtoh3}

Преобразует [H3](#h3index)-индекс из строкового представления в числовое представление `H3Index`.

**Синтаксис**

``` sql
stringToH3(index_str)
```

**Параметр**

-   `index_str` — строковое представление H3-индекса. Тип: [String](../../../sql-reference/data-types/string.md).

**Возвращаемое значение**

-   Числовое представление индекса шестиугольной ячейки.
-   `0`, если при преобразовании возникла ошибка.

Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT stringToH3('89184926cc3ffff') AS index;
```

Результат:

``` text
┌──────────────index─┐
│ 617420388351344639 │
└────────────────────┘
```

## h3GetResolution {#h3getresolution}

Определяет разрешение [H3](#h3index)-ячейки.

**Синтаксис**

``` sql
h3GetResolution(index)
```

**Параметр**

-   `index` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   Разрешение ячейки. Диапазон: `[0, 15]`.

Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3GetResolution(617420388352917503) AS res;
```

Результат:

``` text
┌─res─┐
│   9 │
└─────┘
```

## h3IsResClassIII {#h3isresclassIII}

Проверяет, имеет ли индекс [H3](#h3index) разрешение с ориентацией Class III.

**Синтаксис**

``` sql
h3IsResClassIII(index)
```

**Параметр**

-   `index` — порядковый номер шестиугольника. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемые значения**

-   `1` — индекс имеет разрешение с ориентацией Class III.
-   `0` — индекс не имеет разрешения с ориентацией Class III.

Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3IsResClassIII(617420388352917503) AS res;
```

Результат:

``` text
┌─res─┐
│   1 │
└─────┘
```

## h3IsPentagon {#h3ispentagon}

Проверяет, является ли указанный индекс [H3](#h3index) пятиугольной ячейкой.

**Синтаксис**

``` sql
h3IsPentagon(index)
```

**Параметр**

-   `index` — порядковый номер шестиугольника. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемые значения**

-   `1` — индекс представляет собой пятиугольную ячейку.
-   `0` — индекс не является пятиугольной ячейкой.

Тип: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Пример**

Запрос:

``` sql
SELECT h3IsPentagon(644721767722457330) AS pentagon;
```

Результат:

``` text
┌─pentagon─┐
│        0 │
└──────────┘
```

## h3GetFaces {#h3getfaces}

Возвращает все грани многоугольника (икосаэдра), пересекаемые заданным [H3](#h3index) индексом.

**Синтаксис**

``` sql
h3GetFaces(index)
```

**Параметр**

-   `index` — индекс шестиугольной ячейки. Тип: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Возвращаемое значение**

-   Массив, содержащий грани многоугольника (икосаэдра), пересекаемые заданным H3 индексом.

Тип: [Array](../../../sql-reference/data-types/array.md)([UInt64](../../../sql-reference/data-types/int-uint.md)).

**Пример**

Запрос:

``` sql
SELECT h3GetFaces(599686042433355775) AS faces;
```

Результат:

``` text
┌─faces─┐
│ [7]   │
└───────┘
```

[Оригинальная статья](https://clickhouse.com/docs/ru/sql-reference/functions/geo/h3) <!--hide-->
