---
toc_priority: 35
toc_title: "Массивы"
---

# Массивы {#functions-for-working-with-arrays}

## empty {#function-empty}

Проверяет, является ли входной массив пустым.

**Синтаксис**

``` sql
empty([x])
```

Массив считается пустым, если он не содержит ни одного элемента.

!!! note "Примечание"
    Функцию можно оптимизировать, если включить настройку [optimize_functions_to_subcolumns](../../operations/settings/settings.md#optimize-functions-to-subcolumns). При `optimize_functions_to_subcolumns = 1` функция читает только подстолбец [size0](../../sql-reference/data-types/array.md#array-size) вместо чтения и обработки всего столбца массива. Запрос `SELECT empty(arr) FROM TABLE` преобразуется к запросу `SELECT arr.size0 = 0 FROM TABLE`.

Функция также поддерживает работу с типами [String](string-functions.md#empty) и [UUID](uuid-functions.md#empty).

**Параметры**

-   `[x]` — массив на входе функции. [Array](../data-types/array.md).

**Возвращаемое значение**

-   Возвращает `1` для пустого массива или `0` — для непустого массива.

Тип: [UInt8](../data-types/int-uint.md).

**Пример**

Запрос:

```sql
SELECT empty([]);
```

Ответ:

```text
┌─empty(array())─┐
│              1 │
└────────────────┘
```

## notEmpty {#function-notempty}

Проверяет, является ли входной массив непустым.

**Синтаксис**

``` sql
notEmpty([x])
```

Массив считается непустым, если он содержит хотя бы один элемент.

!!! note "Примечание"
    Функцию можно оптимизировать, если включить настройку [optimize_functions_to_subcolumns](../../operations/settings/settings.md#optimize-functions-to-subcolumns). При `optimize_functions_to_subcolumns = 1` функция читает только подстолбец [size0](../../sql-reference/data-types/array.md#array-size) вместо чтения и обработки всего столбца массива. Запрос `SELECT notEmpty(arr) FROM table` преобразуется к запросу `SELECT arr.size0 != 0 FROM TABLE`.

Функция также поддерживает работу с типами [String](string-functions.md#notempty) и [UUID](uuid-functions.md#notempty).

**Параметры**

-   `[x]` — массив на входе функции. [Array](../data-types/array.md).

**Возвращаемое значение**

-   Возвращает `1` для непустого массива или `0` — для пустого массива.

Тип: [UInt8](../data-types/int-uint.md).

**Пример**

Запрос:

```sql
SELECT notEmpty([1,2]);
```

Результат:

```text
┌─notEmpty([1, 2])─┐
│                1 │
└──────────────────┘
```

## length {#array_functions-length}

Возвращает количество элементов в массиве.
Тип результата - UInt64.
Функция также работает для строк.

Функцию можно оптимизировать, если включить настройку [optimize_functions_to_subcolumns](../../operations/settings/settings.md#optimize-functions-to-subcolumns). При `optimize_functions_to_subcolumns = 1` функция читает только подстолбец [size0](../../sql-reference/data-types/array.md#array-size) вместо чтения и обработки всего столбца массива. Запрос `SELECT length(arr) FROM table` преобразуется к запросу `SELECT arr.size0 FROM TABLE`.

## emptyArrayUInt8, emptyArrayUInt16, emptyArrayUInt32, emptyArrayUInt64 {#emptyarrayuint8-emptyarrayuint16-emptyarrayuint32-emptyarrayuint64}

## emptyArrayInt8, emptyArrayInt16, emptyArrayInt32, emptyArrayInt64 {#emptyarrayint8-emptyarrayint16-emptyarrayint32-emptyarrayint64}

## emptyArrayFloat32, emptyArrayFloat64 {#emptyarrayfloat32-emptyarrayfloat64}

## emptyArrayDate, emptyArrayDateTime {#emptyarraydate-emptyarraydatetime}

## emptyArrayString {#emptyarraystring}

Принимает ноль аргументов и возвращает пустой массив соответствующего типа.

## emptyArrayToSingle {#emptyarraytosingle}

Принимает пустой массив и возвращает массив из одного элемента, равного значению по умолчанию.


## range(end), range(\[start, \] end \[, step\]) {#range}

Возвращает массив чисел от `start` до `end - 1` с шагом `step`.

**Синтаксис**

``` sql
range([start, ] end [, step])
```

**Аргументы**

-   `start` — начало диапазона. Обязательно, когда указан `step`. По умолчанию равно `0`. Тип: [UInt](../data-types/int-uint.md)
-   `end` — конец диапазона. Обязательный аргумент. Должен быть больше, чем `start`. Тип: [UInt](../data-types/int-uint.md)
-   `step` — шаг обхода. Необязательный аргумент. По умолчанию равен `1`. Тип: [UInt](../data-types/int-uint.md)

**Возвращаемые значения**

-   массив `UInt` чисел от `start` до `end - 1` с шагом `step`

**Особенности реализации**

-   Не поддерживаются отрицательные значения аргументов: `start`, `end`, `step` имеют тип `UInt`.

-   Если в результате запроса создаются массивы суммарной длиной больше, чем количество элементов, указанное настройкой [function_range_max_elements_in_block](../../operations/settings/settings.md#settings-function_range_max_elements_in_block), то генерируется исключение.

**Примеры**

Запрос:
``` sql
SELECT range(5), range(1, 5), range(1, 5, 2);
```
Ответ:
```txt
┌─range(5)────┬─range(1, 5)─┬─range(1, 5, 2)─┐
│ [0,1,2,3,4] │ [1,2,3,4]   │ [1,3]          │
└─────────────┴─────────────┴────────────────┘
```


## array(x1, …), оператор \[x1, …\] {#arrayx1-operator-x1}

Создаёт массив из аргументов функции.
Аргументы должны быть константами и иметь типы, для которых есть наименьший общий тип. Должен быть передан хотя бы один аргумент, так как иначе непонятно, какого типа создавать массив. То есть, с помощью этой функции невозможно создать пустой массив (для этого используйте функции emptyArray\*, описанные выше).
Возвращает результат типа Array(T), где T - наименьший общий тип от переданных аргументов.

## arrayConcat {#arrayconcat}

Объединяет массивы, переданные в качестве аргументов.

``` sql
arrayConcat(arrays)
```

**Аргументы**

-   `arrays` – произвольное количество элементов типа [Array](../../sql-reference/functions/array-functions.md)
    **Пример**

<!-- -->

``` sql
SELECT arrayConcat([1, 2], [3, 4], [5, 6]) AS res
```

``` text
┌─res───────────┐
│ [1,2,3,4,5,6] │
└───────────────┘
```

## arrayElement(arr, n), operator arr\[n\] {#arrayelementarr-n-operator-arrn}

Достаёт элемент с индексом n из массива arr. n должен быть любым целочисленным типом.
Индексы в массиве начинаются с единицы.
Поддерживаются отрицательные индексы. В этом случае, будет выбран соответствующий по номеру элемент с конца. Например, arr\[-1\] - последний элемент массива.

Если индекс выходит за границы массива, то возвращается некоторое значение по умолчанию (0 для чисел, пустая строка для строк и т. п.), кроме случая с неконстантным массивом и константным индексом 0 (в этом случае будет ошибка `Array indices are 1-based`).

## has(arr, elem) {#hasarr-elem}

Проверяет наличие элемента elem в массиве arr.
Возвращает 0, если элемента в массиве нет, или 1, если есть.

`NULL` обрабатывается как значение.

``` sql
SELECT has([1, 2, NULL], NULL)
```

``` text
┌─has([1, 2, NULL], NULL)─┐
│                       1 │
└─────────────────────────┘
```

## hasAll {#hasall}

Проверяет, является ли один массив подмножеством другого.

``` sql
hasAll(set, subset)
```

**Аргументы**

-   `set` – массив любого типа с набором элементов.
-   `subset` – массив любого типа со значениями, которые проверяются на вхождение в `set`.

**Возвращаемые значения**

-   `1`, если `set` содержит все элементы из `subset`.
-   `0`, в противном случае.

**Особенности**

-   Пустой массив является подмножеством любого массива.
-   `NULL` обрабатывается как значение.
-   Порядок значений в обоих массивах не имеет значения.

**Примеры**

`SELECT hasAll([], [])` возвращает 1.

`SELECT hasAll([1, Null], [Null])` возвращает 1.

`SELECT hasAll([1.0, 2, 3, 4], [1, 3])` возвращает 1.

`SELECT hasAll(['a', 'b'], ['a'])` возвращает 1.

`SELECT hasAll([1], ['a'])` возвращает 0.

`SELECT hasAll([[1, 2], [3, 4]], [[1, 2], [3, 5]])` возвращает 0.

## hasAny {#hasany}

Проверяет, имеют ли два массива хотя бы один общий элемент.

``` sql
hasAny(array1, array2)
```

**Аргументы**

-   `array1` – массив любого типа с набором элементов.
-   `array2` – массив любого типа с набором элементов.

**Возвращаемые значения**

-   `1`, если `array1` и `array2` имеют хотя бы один одинаковый элемент.
-   `0`, в противном случае.

**Особенности**

-   `NULL` обрабатывается как значение.
-   Порядок значений в обоих массивах не имеет значения.

**Примеры**

`SELECT hasAny([1], [])` возвращает `0`.

`SELECT hasAny([Null], [Null, 1])` возвращает `1`.

`SELECT hasAny([-128, 1., 512], [1])` возвращает `1`.

`SELECT hasAny([[1, 2], [3, 4]], ['a', 'c'])` возвращает `0`.

`SELECT hasAll([[1, 2], [3, 4]], [[1, 2], [1, 2]])` возвращает `1`.

## indexOf(arr, x) {#indexofarr-x}

Возвращает индекс первого элемента x (начиная с 1), если он есть в массиве, или 0, если его нет.

Пример:

``` sql
SELECT indexOf([1, 3, NULL, NULL], NULL)
```

``` text
┌─indexOf([1, 3, NULL, NULL], NULL)─┐
│                                 3 │
└───────────────────────────────────┘
```

Элементы, равные `NULL`, обрабатываются как обычные значения.

## arrayCount(\[func,\] arr1, …) {#array-count}

Возвращает количество элементов массива `arr`, для которых функция `func` возвращает не 0. Если `func` не указана - возвращает количество ненулевых элементов массива.

Функция `arrayCount` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей можно передать лямбда-функцию.


## countEqual(arr, x) {#countequalarr-x}

Возвращает количество элементов массива, равных x. Эквивалентно arrayCount(elem -\> elem = x, arr).

`NULL` обрабатывается как значение.

Пример:

``` sql
SELECT countEqual([1, 2, NULL, NULL], NULL)
```

``` text
┌─countEqual([1, 2, NULL, NULL], NULL)─┐
│                                    2 │
└──────────────────────────────────────┘
```

## arrayEnumerate(arr) {#array_functions-arrayenumerate}

Возвращает массив \[1, 2, 3, …, length(arr)\]

Эта функция обычно используется совместно с ARRAY JOIN. Она позволяет, после применения ARRAY JOIN, посчитать что-либо только один раз для каждого массива. Пример:

``` sql
SELECT
    count() AS Reaches,
    countIf(num = 1) AS Hits
FROM test.hits
ARRAY JOIN
    GoalsReached,
    arrayEnumerate(GoalsReached) AS num
WHERE CounterID = 160656
LIMIT 10
```

``` text
┌─Reaches─┬──Hits─┐
│   95606 │ 31406 │
└─────────┴───────┘
```

В этом примере, Reaches - число достижений целей (строк, получившихся после применения ARRAY JOIN), а Hits - число хитов (строк, которые были до ARRAY JOIN). В данном случае, тот же результат можно получить проще:

``` sql
SELECT
    sum(length(GoalsReached)) AS Reaches,
    count() AS Hits
FROM test.hits
WHERE (CounterID = 160656) AND notEmpty(GoalsReached)
```

``` text
┌─Reaches─┬──Hits─┐
│   95606 │ 31406 │
└─────────┴───────┘
```

Также эта функция может быть использована в функциях высшего порядка. Например, с её помощью можно достать индексы массива для элементов, удовлетворяющих некоторому условию.

## arrayEnumerateUniq(arr, …) {#arrayenumerateuniqarr}

Возвращает массив, такого же размера, как исходный, где для каждого элемента указано, какой он по счету среди элементов с таким же значением.
Например: arrayEnumerateUniq(\[10, 20, 10, 30\]) = \[1, 1, 2, 1\].

Эта функция полезна при использовании ARRAY JOIN и агрегации по элементам массива.
Пример:

``` sql
SELECT
    Goals.ID AS GoalID,
    sum(Sign) AS Reaches,
    sumIf(Sign, num = 1) AS Visits
FROM test.visits
ARRAY JOIN
    Goals,
    arrayEnumerateUniq(Goals.ID) AS num
WHERE CounterID = 160656
GROUP BY GoalID
ORDER BY Reaches DESC
LIMIT 10
```

``` text
┌──GoalID─┬─Reaches─┬─Visits─┐
│   53225 │    3214 │   1097 │
│ 2825062 │    3188 │   1097 │
│   56600 │    2803 │    488 │
│ 1989037 │    2401 │    365 │
│ 2830064 │    2396 │    910 │
│ 1113562 │    2372 │    373 │
│ 3270895 │    2262 │    812 │
│ 1084657 │    2262 │    345 │
│   56599 │    2260 │    799 │
│ 3271094 │    2256 │    812 │
└─────────┴─────────┴────────┘
```

В этом примере, для каждого идентификатора цели, посчитано количество достижений целей (каждый элемент вложенной структуры данных Goals является достижением целей) и количество визитов. Если бы не было ARRAY JOIN, мы бы считали количество визитов как sum(Sign). Но в данном случае, строчки были размножены по вложенной структуре Goals, и чтобы после этого учесть каждый визит один раз, мы поставили условие на значение функции arrayEnumerateUniq(Goals.ID).

Функция arrayEnumerateUniq может принимать несколько аргументов - массивов одинаковых размеров. В этом случае, уникальность считается для кортежей элементов на одинаковых позициях всех массивов.

``` sql
SELECT arrayEnumerateUniq([1, 1, 1, 2, 2, 2], [1, 1, 2, 1, 1, 2]) AS res
```

``` text
┌─res───────────┐
│ [1,2,1,1,2,1] │
└───────────────┘
```

Это нужно при использовании ARRAY JOIN с вложенной структурой данных и затем агрегации по нескольким элементам этой структуры.

## arrayPopBack {#arraypopback}

Удаляет последний элемент из массива.

``` sql
arrayPopBack(array)
```

**Аргументы**

-   `array` – массив.

**Пример**

``` sql
SELECT arrayPopBack([1, 2, 3]) AS res;
```

``` text
┌─res───┐
│ [1,2] │
└───────┘
```

## arrayPopFront {#arraypopfront}

Удаляет первый элемент из массива.

``` sql
arrayPopFront(array)
```

**Аргументы**

-   `array` – массив.

**Пример**

``` sql
SELECT arrayPopFront([1, 2, 3]) AS res;
```

``` text
┌─res───┐
│ [2,3] │
└───────┘
```

## arrayPushBack {#arraypushback}

Добавляет один элемент в конец массива.

``` sql
arrayPushBack(array, single_value)
```

**Аргументы**

-   `array` – массив.
-   `single_value` – значение добавляемого элемента. В массив с числам можно добавить только числа, в массив со строками только строки. При добавлении чисел ClickHouse автоматически приводит тип `single_value` к типу данных массива. Подробнее о типах данных в ClickHouse читайте в разделе «[Типы данных](../../sql-reference/functions/array-functions.md#data_types)». Может быть равно `NULL`, в этом случае функция добавит элемент `NULL` в массив, а тип элементов массива преобразует в `Nullable`.

**Пример**

``` sql
SELECT arrayPushBack(['a'], 'b') AS res;
```

``` text
┌─res───────┐
│ ['a','b'] │
└───────────┘
```

## arrayPushFront {#arraypushfront}

Добавляет один элемент в начало массива.

``` sql
arrayPushFront(array, single_value)
```

**Аргументы**

-   `array` – массив.
-   `single_value` – значение добавляемого элемента. В массив с числам можно добавить только числа, в массив со строками только строки. При добавлении чисел ClickHouse автоматически приводит тип `single_value` к типу данных массива. Подробнее о типах данных в ClickHouse читайте в разделе «[Типы данных](../../sql-reference/functions/array-functions.md#data_types)». Может быть равно `NULL`, в этом случае функция добавит элемент `NULL` в массив, а тип элементов массива преобразует в `Nullable`.

**Пример**

``` sql
SELECT arrayPushFront(['b'], 'a') AS res;
```

``` text
┌─res───────┐
│ ['a','b'] │
└───────────┘
```

## arrayResize {#arrayresize}

Изменяет длину массива.

``` sql
arrayResize(array, size[, extender])
```

**Аргументы**

-   `array` — массив.
-   `size` — необходимая длина массива.
    -   Если `size` меньше изначального размера массива, то массив обрезается справа.
    -   Если `size` больше изначального размера массива, массив дополняется справа значениями `extender` или значениями по умолчанию для типа данных элементов массива.
-   `extender` — значение для дополнения массива. Может быть `NULL`.

**Возвращаемое значение:**

Массив длины `size`.

**Примеры вызовов**

``` sql
SELECT arrayResize([1], 3);
```

``` text
┌─arrayResize([1], 3)─┐
│ [1,0,0]             │
└─────────────────────┘
```

``` sql
SELECT arrayResize([1], 3, NULL);
```

``` text
┌─arrayResize([1], 3, NULL)─┐
│ [1,NULL,NULL]             │
└───────────────────────────┘
```

## arraySlice {#arrayslice}

Возвращает срез массива.

``` sql
arraySlice(array, offset[, length])
```

**Аргументы**

-   `array` – массив данных.
-   `offset` – отступ от края массива. Положительное значение - отступ слева, отрицательное значение - отступ справа. Отсчет элементов массива начинается с 1.
-   `length` – длина необходимого среза. Если указать отрицательное значение, то функция вернёт открытый срез `[offset, array_length - length)`. Если не указать значение, то функция вернёт срез `[offset, the_end_of_array]`.

**Пример**

``` sql
SELECT arraySlice([1, 2, NULL, 4, 5], 2, 3) AS res;
```

``` text
┌─res────────┐
│ [2,NULL,4] │
└────────────┘
```

Элементы массива равные `NULL` обрабатываются как обычные значения.

## arraySort(\[func,\] arr, …) {#array_functions-sort}

Возвращает массив `arr`, отсортированный в восходящем порядке. Если задана функция `func`, то порядок сортировки определяется результатом применения этой функции на элементы массива `arr`. Если `func` принимает несколько аргументов, то в функцию `arraySort` нужно передавать несколько массивов, которые будут соответствовать аргументам функции `func`. Подробные примеры рассмотрены в конце описания `arraySort`.

Пример сортировки целочисленных значений:

``` sql
SELECT arraySort([1, 3, 3, 0])
```

``` text
┌─arraySort([1, 3, 3, 0])─┐
│ [0,1,3,3]               │
└─────────────────────────┘
```

Пример сортировки строковых значений:

``` sql
SELECT arraySort(['hello', 'world', '!'])
```

``` text
┌─arraySort(['hello', 'world', '!'])─┐
│ ['!','hello','world']              │
└────────────────────────────────────┘
```

Значения `NULL`, `NaN` и `Inf` сортируются по следующему принципу:

``` sql
SELECT arraySort([1, nan, 2, NULL, 3, nan, -4, NULL, inf, -inf]);
```

``` text
┌─arraySort([1, nan, 2, NULL, 3, nan, -4, NULL, inf, -inf])─┐
│ [-inf,-4,1,2,3,inf,nan,nan,NULL,NULL]                     │
└───────────────────────────────────────────────────────────┘
```

-   Значения `-Inf` идут в начале массива.
-   Значения `NULL` идут в конце массива.
-   Значения `NaN` идут перед `NULL`.
-   Значения `Inf` идут перед `NaN`.

Функция `arraySort` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей можно передать лямбда-функцию. В этом случае порядок сортировки определяется результатом применения лямбда-функции на элементы массива.

Рассмотрим пример:

``` sql
SELECT arraySort((x) -> -x, [1, 2, 3]) as res;
```

``` text
┌─res─────┐
│ [3,2,1] │
└─────────┘
```

Для каждого элемента исходного массива лямбда-функция возвращает ключ сортировки, то есть \[1 –\> -1, 2 –\> -2, 3 –\> -3\]. Так как `arraySort` сортирует элементы в порядке возрастания ключей, результат будет \[3, 2, 1\]. Как можно заметить, функция `x –> -x` устанавливает [обратный порядок сортировки](#array_functions-reverse-sort).

Лямбда-функция может принимать несколько аргументов. В этом случае, в функцию `arraySort` нужно передавать несколько массивов, которые будут соответствовать аргументам лямбда-функции (массивы должны быть одинаковой длины). Следует иметь в виду, что результат будет содержать элементы только из первого массива; элементы из всех последующих массивов будут задавать ключи сортировки. Например:

``` sql
SELECT arraySort((x, y) -> y, ['hello', 'world'], [2, 1]) as res;
```

``` text
┌─res────────────────┐
│ ['world', 'hello'] │
└────────────────────┘
```

Элементы, указанные во втором массиве (\[2,1\]), определяют ключ сортировки для элементов из исходного массива (\[‘hello’, ‘world’\]), то есть \[‘hello’ –\> 2, ‘world’ –\> 1\]. Так как лямбда-функция не использует `x`, элементы исходного массива не влияют на порядок сортировки. Таким образом, ‘hello’ будет вторым элементом в отсортированном массиве, а ‘world’ — первым.

Ниже приведены другие примеры.

``` sql
SELECT arraySort((x, y) -> y, [0, 1, 2], ['c', 'b', 'a']) as res;
```

``` text
┌─res─────┐
│ [2,1,0] │
└─────────┘
```

``` sql
SELECT arraySort((x, y) -> -y, [0, 1, 2], [1, 2, 3]) as res;
```

``` text
┌─res─────┐
│ [2,1,0] │
└─────────┘
```

!!! note "Примечание"
    Для улучшения эффективности сортировки применяется [преобразование Шварца](https://ru.wikipedia.org/wiki/%D0%9F%D1%80%D0%B5%D0%BE%D0%B1%D1%80%D0%B0%D0%B7%D0%BE%D0%B2%D0%B0%D0%BD%D0%B8%D0%B5_%D0%A8%D0%B2%D0%B0%D1%80%D1%86%D0%B0).

## arrayReverseSort(\[func,\] arr, …) {#array_functions-reverse-sort}

Возвращает массив `arr`, отсортированный в нисходящем порядке. Если указана функция `func`, то массив `arr` сначала сортируется в порядке, который определяется функцией `func`, а затем отсортированный массив переворачивается. Если функция `func` принимает несколько аргументов, то в функцию `arrayReverseSort` необходимо передавать несколько массивов, которые будут соответствовать аргументам функции `func`. Подробные примеры рассмотрены в конце описания функции `arrayReverseSort`.

Пример сортировки целочисленных значений:

``` sql
SELECT arrayReverseSort([1, 3, 3, 0]);
```

``` text
┌─arrayReverseSort([1, 3, 3, 0])─┐
│ [3,3,1,0]                      │
└────────────────────────────────┘
```

Пример сортировки строковых значений:

``` sql
SELECT arrayReverseSort(['hello', 'world', '!']);
```

``` text
┌─arrayReverseSort(['hello', 'world', '!'])─┐
│ ['world','hello','!']                     │
└───────────────────────────────────────────┘
```

Значения `NULL`, `NaN` и `Inf` сортируются в следующем порядке:

``` sql
SELECT arrayReverseSort([1, nan, 2, NULL, 3, nan, -4, NULL, inf, -inf]) as res;
```

``` text
┌─res───────────────────────────────────┐
│ [inf,3,2,1,-4,-inf,nan,nan,NULL,NULL] │
└───────────────────────────────────────┘
```

-   Значения `Inf` идут в начале массива.
-   Значения `NULL` идут в конце массива.
-   Значения `NaN` идут перед `NULL`.
-   Значения `-Inf` идут перед `NaN`.

Функция `arrayReverseSort` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей можно передать лямбда-функцию. Например:

``` sql
SELECT arrayReverseSort((x) -> -x, [1, 2, 3]) as res;
```

``` text
┌─res─────┐
│ [1,2,3] │
└─────────┘
```

В этом примере, порядок сортировки устанавливается следующим образом:

1.  Сначала исходный массив (\[1, 2, 3\]) сортируется в том порядке, который определяется лямбда-функцией. Результатом будет массив \[3, 2, 1\].
2.  Массив, который был получен на предыдущем шаге, переворачивается. То есть, получается массив \[1, 2, 3\].

Лямбда-функция может принимать на вход несколько аргументов. В этом случае, в функцию `arrayReverseSort` нужно передавать несколько массивов, которые будут соответствовать аргументам лямбда-функции (массивы должны быть одинаковой длины). Следует иметь в виду, что результат будет содержать элементы только из первого массива; элементы из всех последующих массивов будут определять ключи сортировки. Например:

``` sql
SELECT arrayReverseSort((x, y) -> y, ['hello', 'world'], [2, 1]) as res;
```

``` text
┌─res───────────────┐
│ ['hello','world'] │
└───────────────────┘
```

В этом примере, массив сортируется следующим образом:

1.  Сначала массив сортируется в том порядке, который определяется лямбда-функцией. Элементы, указанные во втором массиве (\[2,1\]), определяют ключи сортировки соответствующих элементов из исходного массива (\[‘hello’, ‘world’\]). То есть, будет массив \[‘world’, ‘hello’\].
2.  Массив, который был отсортирован на предыдущем шаге, переворачивается. Получается массив \[‘hello’, ‘world’\].

Ниже приведены ещё примеры.

``` sql
SELECT arrayReverseSort((x, y) -> y, [0, 1, 2], ['c', 'b', 'a']) as res;
```

``` text
┌─res─────┐
│ [0,1,2] │
└─────────┘
```

``` sql
SELECT arrayReverseSort((x, y) -> -y, [4, 3, 5], [1, 2, 3]) AS res;
```

``` text
┌─res─────┐
│ [4,3,5] │
└─────────┘
```

## arrayUniq(arr, …) {#array-functions-arrayuniq}

Если передан один аргумент, считает количество разных элементов в массиве.
Если передано несколько аргументов, считает количество разных кортежей из элементов на соответствующих позициях в нескольких массивах.

Если необходимо получить список уникальных элементов массива, можно воспользоваться arrayReduce(‘groupUniqArray’, arr).

## arrayJoin(arr) {#array-functions-arrayjoin}

Особенная функция. Смотрите раздел [«Функция arrayJoin»](array-join.md#functions_arrayjoin).

## arrayDifference {#arraydifference}

Вычисляет разность между соседними элементами массива. Возвращает массив, где первым элементом будет 0, вторым – разность `a[1] - a[0]` и т. д. Тип элементов результирующего массива определяется правилами вывода типов при вычитании (напр. `UInt8` - `UInt8` = `Int16`).

**Синтаксис**

``` sql
arrayDifference(array)
```

**Аргументы**

-   `array` – [массив](https://clickhouse.com/docs/ru/data_types/array/).

**Возвращаемое значение**

Возвращает массив разностей между соседними элементами.

**Пример**

Запрос:

``` sql
SELECT arrayDifference([1, 2, 3, 4]);
```

Результат:

``` text
┌─arrayDifference([1, 2, 3, 4])─┐
│ [0,1,1,1]                     │
└───────────────────────────────┘
```

Пример переполнения из-за результирующего типа `Int64`:

Запрос:

``` sql
SELECT arrayDifference([0, 10000000000000000000]);
```

Результат:

``` text
┌─arrayDifference([0, 10000000000000000000])─┐
│ [0,-8446744073709551616]                   │
└────────────────────────────────────────────┘
```

## arrayDistinct {#arraydistinct}

Принимает массив, возвращает массив, содержащий уникальные элементы.

**Синтаксис**

``` sql
arrayDistinct(array)
```

**Аргументы**

-   `array` – [массив](https://clickhouse.com/docs/ru/data_types/array/).

**Возвращаемое значение**

Возвращает массив, содержащий только уникальные элементы исходного массива.

**Пример**

Запрос:

``` sql
SELECT arrayDistinct([1, 2, 2, 3, 1]);
```

Ответ:

``` text
┌─arrayDistinct([1, 2, 2, 3, 1])─┐
│ [1,2,3]                        │
└────────────────────────────────┘
```

## arrayEnumerateDense(arr) {#array_functions-arrayenumeratedense}

Возвращает массив того же размера, что и исходный массив, с индексами исходного массива, указывающими, где каждый элемент впервые появляется в исходном массиве.

Пример:

``` sql
SELECT arrayEnumerateDense([10, 20, 10, 30])
```

``` text
┌─arrayEnumerateDense([10, 20, 10, 30])─┐
│ [1,2,1,3]                             │
└───────────────────────────────────────┘
```

## arrayIntersect(arr) {#array-functions-arrayintersect}

Принимает несколько массивов, возвращает массив с элементами, присутствующими во всех исходных массивах.

Пример:

``` sql
SELECT
    arrayIntersect([1, 2], [1, 3], [2, 3]) AS no_intersect,
    arrayIntersect([1, 2], [1, 3], [1, 4]) AS intersect
```

``` text
┌─no_intersect─┬─intersect─┐
│ []           │ [1]       │
└──────────────┴───────────┘
```

## arrayReduce {#arrayreduce}

Применяет агрегатную функцию к элементам массива и возвращает ее результат. Имя агрегирующей функции передается как строка в одинарных кавычках `'max'`, `'sum'`. При использовании параметрических агрегатных функций, параметр указывается после имени функции в круглых скобках `'uniqUpTo(6)'`.

**Синтаксис**

```sql
arrayReduce(agg_func, arr1, arr2, ..., arrN)
```

**Аргументы**

-   `agg_func` — Имя агрегатной функции, которая должна быть константой [string](../../sql-reference/data-types/string.md).
-   `arr` — Любое количество столбцов типа [array](../../sql-reference/data-types/array.md) в качестве параметров агрегатной функции.

**Возвращаемое значение**

**Пример**

Запрос:

```sql
SELECT arrayReduce('max', [1, 2, 3]);
```

Результат:

```text
┌─arrayReduce('max', [1, 2, 3])─┐
│                             3 │
└───────────────────────────────┘
```

Если агрегатная функция имеет несколько аргументов, то эту функцию можно применять к нескольким массивам одинакового размера.

**Пример**

Запрос:

```sql
SELECT arrayReduce('maxIf', [3, 5], [1, 0]);
```

Результат:

```text
┌─arrayReduce('maxIf', [3, 5], [1, 0])─┐
│                                    3 │
└──────────────────────────────────────┘
```

Пример с параметрической агрегатной функцией:

Запрос:

```sql
SELECT arrayReduce('uniqUpTo(3)', [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
```

Результат:

```text
┌─arrayReduce('uniqUpTo(3)', [1, 2, 3, 4, 5, 6, 7, 8, 9, 10])─┐
│                                                           4 │
└─────────────────────────────────────────────────────────────┘
```

## arrayReduceInRanges {#arrayreduceinranges}

Применяет агрегатную функцию к элементам массива в заданных диапазонах и возвращает массив, содержащий результат, соответствующий каждому диапазону. Функция вернет тот же результат, что и несколько `arrayReduce(agg_func, arraySlice(arr1, index, length), ...)`.

**Синтаксис**

```sql
arrayReduceInRanges(agg_func, ranges, arr1, arr2, ..., arrN)
```

**Аргументы**

-   `agg_func` — имя агрегатной функции, которая должна быть [строковой](../../sql-reference/data-types/string.md) константой.
-   `ranges` — диапазоны для агрегирования, которые должны быть [массивом](../../sql-reference/data-types/array.md) of [кортежей](../../sql-reference/data-types/tuple.md) содержащих индекс и длину каждого диапазона.
-   `arr` — любое количество столбцов типа [Array](../../sql-reference/data-types/array.md) в качестве параметров агрегатной функции.

**Возвращаемое значение**

-   Массив, содержащий результаты агрегатной функции для указанных диапазонов.

Тип: [Array](../../sql-reference/data-types/array.md).

**Пример**

Запрос:

```sql
SELECT arrayReduceInRanges(
    'sum',
    [(1, 5), (2, 3), (3, 4), (4, 4)],
    [1000000, 200000, 30000, 4000, 500, 60, 7]
) AS res
```

Результат:

```text
┌─res─────────────────────────┐
│ [1234500,234000,34560,4567] │
└─────────────────────────────┘
```

## arrayReverse(arr) {#arrayreverse}

Возвращает массив того же размера, что и исходный массив, содержащий элементы в обратном порядке.

Пример:

``` sql
SELECT arrayReverse([1, 2, 3])
```

``` text
┌─arrayReverse([1, 2, 3])─┐
│ [3,2,1]                 │
└─────────────────────────┘
```

## reverse(arr) {#array-functions-reverse}

Синоним для [«arrayReverse»](#arrayreverse)

## arrayFlatten {#arrayflatten}

Преобразует массив массивов в плоский массив.

Функция:

-   Оперирует с массивами любой вложенности.
-   Не изменяет массив, если он уже плоский.

Результирующий массив содержит все элементы исходных массивов.

**Синтаксис**

``` sql
flatten(array_of_arrays)
```

Синоним: `flatten`.

**Аргументы**

-   `array_of_arrays` — [массив](../../sql-reference/functions/array-functions.md) массивов. Например, `[[1,2,3], [4,5]]`.

**Примеры**

``` sql
SELECT flatten([[[1]], [[2], [3]]]);
```

``` text
┌─flatten(array(array([1]), array([2], [3])))─┐
│ [1,2,3]                                     │
└─────────────────────────────────────────────┘
```

## arrayCompact {#arraycompact}

Удаляет последовательно повторяющиеся элементы из массива. Порядок результирующих значений определяется порядком в исходном массиве.

**Синтаксис**

``` sql
arrayCompact(arr)
```

**Аргументы**

`arr` — [массив](../../sql-reference/functions/array-functions.md) для обхода.

**Возвращаемое значение**

Массив без последовательных дубликатов.

Тип: `Array`.

**Пример**

Запрос:

``` sql
SELECT arrayCompact([1, 1, nan, nan, 2, 3, 3, 3]);
```

Результат:

``` text
┌─arrayCompact([1, 1, nan, nan, 2, 3, 3, 3])─┐
│ [1,nan,nan,2,3]                            │
└────────────────────────────────────────────┘
```

## arrayZip {#arrayzip}

Объединяет несколько массивов в один. Результирующий массив содержит соответственные элементы исходных массивов, сгруппированные в кортежи в указанном порядке аргументов.

**Синтаксис**

``` sql
arrayZip(arr1, arr2, ..., arrN)
```

**Аргументы**

-   `arrN` — [массив](../data-types/array.md).

Функция принимает любое количество массивов, которые могут быть различных типов. Все массивы должны иметь одинаковую длину.

**Возвращаемое значение**

-   Массив с элементами исходных массивов, сгруппированными в [кортежи](../data-types/tuple.md). Типы данных в кортежах соответствуют типам данных входных массивов и следуют в том же порядке, в котором переданы массивы.

Тип: [Массив](../data-types/array.md).

**Пример**

Запрос:

``` sql
SELECT arrayZip(['a', 'b', 'c'], [5, 2, 1]);
```

Результат:

``` text
┌─arrayZip(['a', 'b', 'c'], [5, 2, 1])─┐
│ [('a',5),('b',2),('c',1)]            │
└──────────────────────────────────────┘
```

## arrayMap(func, arr1, …) {#array-map}

Возвращает массив, полученный на основе результатов применения функции `func` к каждому элементу массива `arr`.

Примеры:

``` sql
SELECT arrayMap(x -> (x + 2), [1, 2, 3]) as res;
```

``` text
┌─res─────┐
│ [3,4,5] │
└─────────┘
```

Следующий пример показывает, как создать кортежи из элементов разных массивов:

``` sql
SELECT arrayMap((x, y) -> (x, y), [1, 2, 3], [4, 5, 6]) AS res;
```

``` text
┌─res─────────────────┐
│ [(1,4),(2,5),(3,6)] │
└─────────────────────┘
```

Функция `arrayMap` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arrayFilter(func, arr1, …) {#array-filter}

Возвращает массив, содержащий только те элементы массива `arr1`, для которых функция `func` возвращает не 0.

Примеры:

``` sql
SELECT arrayFilter(x -> x LIKE '%World%', ['Hello', 'abc World']) AS res
```

``` text
┌─res───────────┐
│ ['abc World'] │
└───────────────┘
```

``` sql
SELECT
    arrayFilter(
        (i, x) -> x LIKE '%World%',
        arrayEnumerate(arr),
        ['Hello', 'abc World'] AS arr)
    AS res
```

``` text
┌─res─┐
│ [2] │
└─────┘
```

Функция `arrayFilter` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arrayFill(func, arr1, …) {#array-fill}

Перебирает `arr1` от первого элемента к последнему и заменяет `arr1[i]` на `arr1[i - 1]`, если `func` вернула 0. Первый элемент `arr1` остаётся неизменным.

Примеры:

``` sql
SELECT arrayFill(x -> not isNull(x), [1, null, 3, 11, 12, null, null, 5, 6, 14, null, null]) AS res
```

``` text
┌─res──────────────────────────────┐
│ [1,1,3,11,12,12,12,5,6,14,14,14] │
└──────────────────────────────────┘
```

Функция `arrayFill` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arrayReverseFill(func, arr1, …) {#array-reverse-fill}

Перебирает `arr1` от последнего элемента к первому и заменяет `arr1[i]` на `arr1[i + 1]`, если `func` вернула 0. Последний элемент `arr1` остаётся неизменным.

Примеры:

``` sql
SELECT arrayReverseFill(x -> not isNull(x), [1, null, 3, 11, 12, null, null, 5, 6, 14, null, null]) AS res
```

``` text
┌─res────────────────────────────────┐
│ [1,3,3,11,12,5,5,5,6,14,NULL,NULL] │
└────────────────────────────────────┘
```

Функция `arrayReverseFill` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arraySplit(func, arr1, …) {#array-split}

Разделяет массив `arr1` на несколько. Если `func` возвращает не 0, то массив разделяется, а элемент помещается в левую часть. Массив не разбивается по первому элементу.

Примеры:

``` sql
SELECT arraySplit((x, y) -> y, [1, 2, 3, 4, 5], [1, 0, 0, 1, 0]) AS res
```

``` text
┌─res─────────────┐
│ [[1,2,3],[4,5]] │
└─────────────────┘
```

Функция `arraySplit` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arrayReverseSplit(func, arr1, …) {#array-reverse-split}

Разделяет массив `arr1` на несколько. Если `func` возвращает не 0, то массив разделяется, а элемент помещается в правую часть. Массив не разбивается по последнему элементу.

Примеры:

``` sql
SELECT arrayReverseSplit((x, y) -> y, [1, 2, 3, 4, 5], [1, 0, 0, 1, 0]) AS res
```

``` text
┌─res───────────────┐
│ [[1],[2,3,4],[5]] │
└───────────────────┘
```

Функция `arrayReverseSplit` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arrayExists(\[func,\] arr1, …) {#arrayexistsfunc-arr1}

Возвращает 1, если существует хотя бы один элемент массива `arr`, для которого функция func возвращает не 0. Иначе возвращает 0.

Функция `arrayExists` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) - в качестве первого аргумента ей можно передать лямбда-функцию.

## arrayAll(\[func,\] arr1, …) {#arrayallfunc-arr1}

Возвращает 1, если для всех элементов массива `arr`, функция `func` возвращает не 0. Иначе возвращает 0.

Функция `arrayAll` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) - в качестве первого аргумента ей можно передать лямбда-функцию.

## arrayFirst(func, arr1, …) {#array-first}

Возвращает первый элемент массива `arr1`, для которого функция func возвращает не 0.

Функция `arrayFirst` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arrayFirstIndex(func, arr1, …) {#array-first-index}

Возвращает индекс первого элемента массива `arr1`, для которого функция func возвращает не 0.

Функция `arrayFirstIndex` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей нужно передать лямбда-функцию, и этот аргумент не может быть опущен.

## arrayMin {#array-min}

Возвращает значение минимального элемента в исходном массиве.

Если передана функция `func`, возвращается минимум из элементов массива, преобразованных этой функцией.

Функция `arrayMin` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей можно передать лямбда-функцию.

**Синтаксис**

```sql
arrayMin([func,] arr)
```

**Аргументы**

-   `func` — функция. [Expression](../../sql-reference/data-types/special-data-types/expression.md).
-   `arr` — массив. [Array](../../sql-reference/data-types/array.md).

**Возвращаемое значение**

-   Минимальное значение функции (или минимальный элемент массива).

Тип: если передана `func`, соответствует типу ее возвращаемого значения, иначе соответствует типу элементов массива.

**Примеры**

Запрос:

```sql
SELECT arrayMin([1, 2, 4]) AS res;
```

Результат:

```text
┌─res─┐
│   1 │
└─────┘
```

Запрос:

```sql
SELECT arrayMin(x -> (-x), [1, 2, 4]) AS res;
```

Результат:

```text
┌─res─┐
│  -4 │
└─────┘
```

## arrayMax {#array-max}

Возвращает значение максимального элемента в исходном массиве.

Если передана функция `func`, возвращается максимум из элементов массива, преобразованных этой функцией.

Функция `arrayMax` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей можно передать лямбда-функцию.

**Синтаксис**

```sql
arrayMax([func,] arr)
```

**Аргументы**

-   `func` — функция. [Expression](../../sql-reference/data-types/special-data-types/expression.md).
-   `arr` — массив. [Array](../../sql-reference/data-types/array.md).

**Возвращаемое значение**

-   Максимальное значение функции (или максимальный элемент массива).

Тип: если передана `func`, соответствует типу ее возвращаемого значения, иначе соответствует типу элементов массива.

**Примеры**

Запрос:

```sql
SELECT arrayMax([1, 2, 4]) AS res;
```

Результат:

```text
┌─res─┐
│   4 │
└─────┘
```

Запрос:

```sql
SELECT arrayMax(x -> (-x), [1, 2, 4]) AS res;
```

Результат:

```text
┌─res─┐
│  -1 │
└─────┘
```

## arraySum {#array-sum}

Возвращает сумму элементов в исходном массиве.

Если передана функция `func`, возвращается сумма элементов массива, преобразованных этой функцией.

Функция `arraySum` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей можно передать лямбда-функцию.

**Синтаксис**

```sql
arraySum([func,] arr)
```

**Аргументы**

-   `func` — функция. [Expression](../../sql-reference/data-types/special-data-types/expression.md).
-   `arr` — массив. [Array](../../sql-reference/data-types/array.md).

**Возвращаемое значение**

-   Сумма значений функции (или сумма элементов массива).

Тип: для Decimal чисел в исходном массиве (если функция `func` была передана, то для чисел, преобразованных ею) — [Decimal128](../../sql-reference/data-types/decimal.md), для чисел с плавающей точкой — [Float64](../../sql-reference/data-types/float.md), для беззнаковых целых чисел — [UInt64](../../sql-reference/data-types/int-uint.md), для целых чисел со знаком — [Int64](../../sql-reference/data-types/int-uint.md).

**Примеры**

Запрос:

```sql
SELECT arraySum([2, 3]) AS res;
```

Результат:

```text
┌─res─┐
│   5 │
└─────┘
```

Запрос:

```sql
SELECT arraySum(x -> x*x, [2, 3]) AS res;
```

Результат:

```text
┌─res─┐
│  13 │
└─────┘
```

## arrayAvg {#array-avg}

Возвращает среднее значение элементов в исходном массиве.

Если передана функция `func`, возвращается среднее значение элементов массива, преобразованных этой функцией.

Функция `arrayAvg` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) — в качестве первого аргумента ей можно передать лямбда-функцию.

**Синтаксис**

```sql
arrayAvg([func,] arr)
```

**Аргументы**

-   `func` — функция. [Expression](../../sql-reference/data-types/special-data-types/expression.md).
-   `arr` — массив. [Array](../../sql-reference/data-types/array.md).

**Возвращаемое значение**

-   Среднее значение функции (или среднее значение элементов массива).

Тип: [Float64](../../sql-reference/data-types/float.md).

**Примеры**

Запрос:

```sql
SELECT arrayAvg([1, 2, 4]) AS res;
```

Результат:

```text
┌────────────────res─┐
│ 2.3333333333333335 │
└────────────────────┘
```

Запрос:

```sql
SELECT arrayAvg(x -> (x * x), [2, 4]) AS res;
```

Результат:

```text
┌─res─┐
│  10 │
└─────┘
```

**Синтаксис**

``` sql
arraySum(arr)
```

**Возвращаемое значение**

-   Число.

Тип: [Int](../../sql-reference/data-types/int-uint.md) или [Float](../../sql-reference/data-types/float.md).

**Аргументы**

-   `arr` — [массив](../../sql-reference/data-types/array.md).

**Примеры**

Запрос:

```sql
SELECT arraySum([2,3]) AS res;
```

Результат:

``` text
┌─res─┐
│   5 │
└─────┘
```

Запрос:

``` sql
SELECT arraySum(x -> x*x, [2, 3]) AS res;
```

Результат:

``` text
┌─res─┐
│  13 │
└─────┘
```

## arrayCumSum(\[func,\] arr1, …) {#arraycumsumfunc-arr1}

Возвращает массив из частичных сумм элементов исходного массива (сумма с накоплением). Если указана функция `func`, то значения элементов массива преобразуются этой функцией перед суммированием.

Функция `arrayCumSum` является [функцией высшего порядка](../../sql-reference/functions/index.md#higher-order-functions) - в качестве первого аргумента ей можно передать лямбда-функцию.

Пример:

``` sql
SELECT arrayCumSum([1, 1, 1, 1]) AS res
```

``` text
┌─res──────────┐
│ [1, 2, 3, 4] │
└──────────────┘
```

## arrayAUC {#arrayauc}

Вычисляет площадь под кривой.

**Синтаксис**

``` sql
arrayAUC(arr_scores, arr_labels)
```

**Аргументы**

- `arr_scores` — оценка, которую дает модель предсказания.
- `arr_labels` — ярлыки выборок, обычно 1 для содержательных выборок и 0 для бессодержательных выборок.

**Возвращаемое значение**

Значение площади под кривой.

Тип данных: `Float64`.

**Пример**

Запрос:

``` sql
SELECT arrayAUC([0.1, 0.4, 0.35, 0.8], [0, 0, 1, 1]);
```

Результат:

``` text
┌─arrayAUC([0.1, 0.4, 0.35, 0.8], [0, 0, 1, 1])─┐
│                                          0.75 │
└────────────────────────────────────────---──┘
```

## arrayProduct {#arrayproduct}

Возвращает произведение элементов [массива](../../sql-reference/data-types/array.md).

**Синтаксис**

``` sql
arrayProduct(arr)
```

**Аргументы**

-   `arr` — [массив](../../sql-reference/data-types/array.md) числовых значений.

**Возвращаемое значение**

-   Произведение элементов массива.

Тип: [Float64](../../sql-reference/data-types/float.md).

**Примеры**

Запрос:

``` sql
SELECT arrayProduct([1,2,3,4,5,6]) as res;
```

Результат:

``` text
┌─res───┐
│ 720   │
└───────┘
```

Запрос:

``` sql
SELECT arrayProduct([toDecimal64(1,8), toDecimal64(2,8), toDecimal64(3,8)]) as res, toTypeName(res);
```

Возвращаемое значение всегда имеет тип [Float64](../../sql-reference/data-types/float.md). Результат:

``` text
┌─res─┬─toTypeName(arrayProduct(array(toDecimal64(1, 8), toDecimal64(2, 8), toDecimal64(3, 8))))─┐
│ 6   │ Float64                                                                                  │
└─────┴──────────────────────────────────────────────────────────────────────────────────────────┘
```
