---
toc_priority: 20
toc_title: Набор данных о воздушном движении OpenSky Network 2020
---

# Набор данных о воздушном движении OpenSky Network 2020 {#opensky}

"Данные в этом наборе получены и отфильтрованы из полного набора данных OpenSky, чтобы проиллюстрировать развитие воздушного движения во время пандемии COVID-19. Набор включает в себя все рейсы, которые видели более 2500 участников сети с 1 января 2019 года. Дополнительные данные будут периодически включаться в набор данных до окончания пандемии COVID-19".

Источник: https://zenodo.org/record/5092942#.YRBCyTpRXYd

Martin Strohmeier, Xavier Olive, Jannis Lübbe, Matthias Schäfer, and Vincent Lenders
"Crowdsourced air traffic data from the OpenSky Network 2019–2020"
Earth System Science Data 13(2), 2021
https://doi.org/10.5194/essd-13-357-2021

## Загрузите набор данных {#download-dataset}

Выполните команду:

```bash
wget -O- https://zenodo.org/record/5092942 | grep -oP 'https://zenodo.org/record/5092942/files/flightlist_\d+_\d+\.csv\.gz' | xargs wget
```

Загрузка займет около 2 минут при хорошем подключении к интернету. Будет загружено 30 файлов общим размером 4,3 ГБ.

## Создайте таблицу {#create-table}

```sql
CREATE TABLE opensky
(
    callsign String,
    number String,
    icao24 String,
    registration String,
    typecode String,
    origin String,
    destination String,
    firstseen DateTime,
    lastseen DateTime,
    day DateTime,
    latitude_1 Float64,
    longitude_1 Float64,
    altitude_1 Float64,
    latitude_2 Float64,
    longitude_2 Float64,
    altitude_2 Float64
) ENGINE = MergeTree ORDER BY (origin, destination, callsign);
```

## Импортируйте данные в ClickHouse {#import-data}

Загрузите данные в ClickHouse параллельными потоками:

```bash
ls -1 flightlist_*.csv.gz | xargs -P100 -I{} bash -c 'gzip -c -d "{}" | clickhouse-client --date_time_input_format best_effort --query "INSERT INTO opensky FORMAT CSVWithNames"'
```

- Список файлов передаётся (`ls -1 flightlist_*.csv.gz`) в `xargs` для параллельной обработки.
- `xargs -P100` указывает на возможность использования до 100 параллельных обработчиков, но поскольку у нас всего 30 файлов, то количество обработчиков будет всего 30.
- Для каждого файла `xargs` будет запускать скрипт с `bash -c`. Сценарий имеет подстановку в виде `{}`, а команда `xargs` заменяет имя файла на указанные в подстановке символы (мы указали это для `xargs` с помощью `-I{}`).
- Скрипт распакует файл (`gzip -c -d "{}"`) в стандартный вывод (параметр `-c`) и перенаправит его в `clickhouse-client`.
- Чтобы распознать формат ISO-8601 со смещениями часовых поясов в полях типа [DateTime](../../sql-reference/data-types/datetime.md), указывается параметр парсера [--date_time_input_format best_effort](../../operations/settings/settings.md#settings-date_time_input_format).

В итоге: клиент clickhouse добавит данные в таблицу `opensky`. Входные данные импортируются в формате [CSVWithNames](../../interfaces/formats.md#csvwithnames).


Загрузка параллельными потоками займёт около 24 секунд.

Также вы можете использовать вариант последовательной загрузки:

```bash
for file in flightlist_*.csv.gz; do gzip -c -d "$file" | clickhouse-client --date_time_input_format best_effort --query "INSERT INTO opensky FORMAT CSVWithNames"; done
```

## Проверьте импортированные данные {#validate-data}

Запрос:

```sql
SELECT count() FROM opensky;
```

Результат:

```text
┌──count()─┐
│ 66010819 │
└──────────┘
```

Убедитесь, что размер набора данных в ClickHouse составляет всего 2,66 GiB.

Запрос:

```sql
SELECT formatReadableSize(total_bytes) FROM system.tables WHERE name = 'opensky';
```

Результат:

```text
┌─formatReadableSize(total_bytes)─┐
│ 2.66 GiB                        │
└─────────────────────────────────┘
```

## Примеры {#run-queries}

Общее пройденное расстояние составляет 68 миллиардов километров.

Запрос:

```sql
SELECT formatReadableQuantity(sum(geoDistance(longitude_1, latitude_1, longitude_2, latitude_2)) / 1000) FROM opensky;
```

Результат:

```text
┌─formatReadableQuantity(divide(sum(geoDistance(longitude_1, latitude_1, longitude_2, latitude_2)), 1000))─┐
│ 68.72 billion                                                                                            │
└──────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

Средняя дальность полета составляет около 1000 км.

Запрос:

```sql
SELECT avg(geoDistance(longitude_1, latitude_1, longitude_2, latitude_2)) FROM opensky;
```

Результат:

```text
┌─avg(geoDistance(longitude_1, latitude_1, longitude_2, latitude_2))─┐
│                                                 1041090.6465708319 │
└────────────────────────────────────────────────────────────────────┘
```

### Наиболее загруженные аэропорты в указанных координатах и среднее пройденное расстояние {#busy-airports-average-distance}

Запрос:

```sql
SELECT
    origin,
    count(),
    round(avg(geoDistance(longitude_1, latitude_1, longitude_2, latitude_2))) AS distance,
    bar(distance, 0, 10000000, 100) AS bar
FROM opensky
WHERE origin != ''
GROUP BY origin
ORDER BY count() DESC
LIMIT 100;
```

Результат:

```text
     ┌─origin─┬─count()─┬─distance─┬─bar────────────────────────────────────┐
  1. │ KORD   │  745007 │  1546108 │ ███████████████▍                       │
  2. │ KDFW   │  696702 │  1358721 │ █████████████▌                         │
  3. │ KATL   │  667286 │  1169661 │ ███████████▋                           │
  4. │ KDEN   │  582709 │  1287742 │ ████████████▊                          │
  5. │ KLAX   │  581952 │  2628393 │ ██████████████████████████▎            │
  6. │ KLAS   │  447789 │  1336967 │ █████████████▎                         │
  7. │ KPHX   │  428558 │  1345635 │ █████████████▍                         │
  8. │ KSEA   │  412592 │  1757317 │ █████████████████▌                     │
  9. │ KCLT   │  404612 │   880355 │ ████████▋                              │
 10. │ VIDP   │  363074 │  1445052 │ ██████████████▍                        │
 11. │ EDDF   │  362643 │  2263960 │ ██████████████████████▋                │
 12. │ KSFO   │  361869 │  2445732 │ ████████████████████████▍              │
 13. │ KJFK   │  349232 │  2996550 │ █████████████████████████████▊         │
 14. │ KMSP   │  346010 │  1287328 │ ████████████▋                          │
 15. │ LFPG   │  344748 │  2206203 │ ██████████████████████                 │
 16. │ EGLL   │  341370 │  3216593 │ ████████████████████████████████▏      │
 17. │ EHAM   │  340272 │  2116425 │ █████████████████████▏                 │
 18. │ KEWR   │  337696 │  1826545 │ ██████████████████▎                    │
 19. │ KPHL   │  320762 │  1291761 │ ████████████▊                          │
 20. │ OMDB   │  308855 │  2855706 │ ████████████████████████████▌          │
 21. │ UUEE   │  307098 │  1555122 │ ███████████████▌                       │
 22. │ KBOS   │  304416 │  1621675 │ ████████████████▏                      │
 23. │ LEMD   │  291787 │  1695097 │ ████████████████▊                      │
 24. │ YSSY   │  272979 │  1875298 │ ██████████████████▋                    │
 25. │ KMIA   │  265121 │  1923542 │ ███████████████████▏                   │
 26. │ ZGSZ   │  263497 │   745086 │ ███████▍                               │
 27. │ EDDM   │  256691 │  1361453 │ █████████████▌                         │
 28. │ WMKK   │  254264 │  1626688 │ ████████████████▎                      │
 29. │ CYYZ   │  251192 │  2175026 │ █████████████████████▋                 │
 30. │ KLGA   │  248699 │  1106935 │ ███████████                            │
 31. │ VHHH   │  248473 │  3457658 │ ██████████████████████████████████▌    │
 32. │ RJTT   │  243477 │  1272744 │ ████████████▋                          │
 33. │ KBWI   │  241440 │  1187060 │ ███████████▋                           │
 34. │ KIAD   │  239558 │  1683485 │ ████████████████▋                      │
 35. │ KIAH   │  234202 │  1538335 │ ███████████████▍                       │
 36. │ KFLL   │  223447 │  1464410 │ ██████████████▋                        │
 37. │ KDAL   │  212055 │  1082339 │ ██████████▋                            │
 38. │ KDCA   │  207883 │  1013359 │ ██████████▏                            │
 39. │ LIRF   │  207047 │  1427965 │ ██████████████▎                        │
 40. │ PANC   │  206007 │  2525359 │ █████████████████████████▎             │
 41. │ LTFJ   │  205415 │   860470 │ ████████▌                              │
 42. │ KDTW   │  204020 │  1106716 │ ███████████                            │
 43. │ VABB   │  201679 │  1300865 │ █████████████                          │
 44. │ OTHH   │  200797 │  3759544 │ █████████████████████████████████████▌ │
 45. │ KMDW   │  200796 │  1232551 │ ████████████▎                          │
 46. │ KSAN   │  198003 │  1495195 │ ██████████████▊                        │
 47. │ KPDX   │  197760 │  1269230 │ ████████████▋                          │
 48. │ SBGR   │  197624 │  2041697 │ ████████████████████▍                  │
 49. │ VOBL   │  189011 │  1040180 │ ██████████▍                            │
 50. │ LEBL   │  188956 │  1283190 │ ████████████▋                          │
 51. │ YBBN   │  188011 │  1253405 │ ████████████▌                          │
 52. │ LSZH   │  187934 │  1572029 │ ███████████████▋                       │
 53. │ YMML   │  187643 │  1870076 │ ██████████████████▋                    │
 54. │ RCTP   │  184466 │  2773976 │ ███████████████████████████▋           │
 55. │ KSNA   │  180045 │   778484 │ ███████▋                               │
 56. │ EGKK   │  176420 │  1694770 │ ████████████████▊                      │
 57. │ LOWW   │  176191 │  1274833 │ ████████████▋                          │
 58. │ UUDD   │  176099 │  1368226 │ █████████████▋                         │
 59. │ RKSI   │  173466 │  3079026 │ ██████████████████████████████▋        │
 60. │ EKCH   │  172128 │  1229895 │ ████████████▎                          │
 61. │ KOAK   │  171119 │  1114447 │ ███████████▏                           │
 62. │ RPLL   │  170122 │  1440735 │ ██████████████▍                        │
 63. │ KRDU   │  167001 │   830521 │ ████████▎                              │
 64. │ KAUS   │  164524 │  1256198 │ ████████████▌                          │
 65. │ KBNA   │  163242 │  1022726 │ ██████████▏                            │
 66. │ KSDF   │  162655 │  1380867 │ █████████████▋                         │
 67. │ ENGM   │  160732 │   910108 │ █████████                              │
 68. │ LIMC   │  160696 │  1564620 │ ███████████████▋                       │
 69. │ KSJC   │  159278 │  1081125 │ ██████████▋                            │
 70. │ KSTL   │  157984 │  1026699 │ ██████████▎                            │
 71. │ UUWW   │  156811 │  1261155 │ ████████████▌                          │
 72. │ KIND   │  153929 │   987944 │ █████████▊                             │
 73. │ ESSA   │  153390 │  1203439 │ ████████████                           │
 74. │ KMCO   │  153351 │  1508657 │ ███████████████                        │
 75. │ KDVT   │  152895 │    74048 │ ▋                                      │
 76. │ VTBS   │  152645 │  2255591 │ ██████████████████████▌                │
 77. │ CYVR   │  149574 │  2027413 │ ████████████████████▎                  │
 78. │ EIDW   │  148723 │  1503985 │ ███████████████                        │
 79. │ LFPO   │  143277 │  1152964 │ ███████████▌                           │
 80. │ EGSS   │  140830 │  1348183 │ █████████████▍                         │
 81. │ KAPA   │  140776 │   420441 │ ████▏                                  │
 82. │ KHOU   │  138985 │  1068806 │ ██████████▋                            │
 83. │ KTPA   │  138033 │  1338223 │ █████████████▍                         │
 84. │ KFFZ   │  137333 │    55397 │ ▌                                      │
 85. │ NZAA   │  136092 │  1581264 │ ███████████████▋                       │
 86. │ YPPH   │  133916 │  1271550 │ ████████████▋                          │
 87. │ RJBB   │  133522 │  1805623 │ ██████████████████                     │
 88. │ EDDL   │  133018 │  1265919 │ ████████████▋                          │
 89. │ ULLI   │  130501 │  1197108 │ ███████████▊                           │
 90. │ KIWA   │  127195 │   250876 │ ██▌                                    │
 91. │ KTEB   │  126969 │  1189414 │ ███████████▊                           │
 92. │ VOMM   │  125616 │  1127757 │ ███████████▎                           │
 93. │ LSGG   │  123998 │  1049101 │ ██████████▍                            │
 94. │ LPPT   │  122733 │  1779187 │ █████████████████▋                     │
 95. │ WSSS   │  120493 │  3264122 │ ████████████████████████████████▋      │
 96. │ EBBR   │  118539 │  1579939 │ ███████████████▋                       │
 97. │ VTBD   │  118107 │   661627 │ ██████▌                                │
 98. │ KVNY   │  116326 │   692960 │ ██████▊                                │
 99. │ EDDT   │  115122 │   941740 │ █████████▍                             │
100. │ EFHK   │  114860 │  1629143 │ ████████████████▎                      │
     └────────┴─────────┴──────────┴────────────────────────────────────────┘
```

### Номера рейсов из трех крупных аэропортов Москвы, еженедельно {#flights-from-moscow}

Запрос:

```sql
SELECT
    toMonday(day) AS k,
    count() AS c,
    bar(c, 0, 10000, 100) AS bar
FROM opensky
WHERE origin IN ('UUEE', 'UUDD', 'UUWW')
GROUP BY k
ORDER BY k ASC;
```

Результат:

```text
     ┌──────────k─┬────c─┬─bar──────────────────────────────────────────────────────────────────────────┐
  1. │ 2018-12-31 │ 5248 │ ████████████████████████████████████████████████████▍                        │
  2. │ 2019-01-07 │ 6302 │ ███████████████████████████████████████████████████████████████              │
  3. │ 2019-01-14 │ 5701 │ █████████████████████████████████████████████████████████                    │
  4. │ 2019-01-21 │ 5638 │ ████████████████████████████████████████████████████████▍                    │
  5. │ 2019-01-28 │ 5731 │ █████████████████████████████████████████████████████████▎                   │
  6. │ 2019-02-04 │ 5683 │ ████████████████████████████████████████████████████████▋                    │
  7. │ 2019-02-11 │ 5759 │ █████████████████████████████████████████████████████████▌                   │
  8. │ 2019-02-18 │ 5736 │ █████████████████████████████████████████████████████████▎                   │
  9. │ 2019-02-25 │ 5873 │ ██████████████████████████████████████████████████████████▋                  │
 10. │ 2019-03-04 │ 5965 │ ███████████████████████████████████████████████████████████▋                 │
 11. │ 2019-03-11 │ 5900 │ ███████████████████████████████████████████████████████████                  │
 12. │ 2019-03-18 │ 5823 │ ██████████████████████████████████████████████████████████▏                  │
 13. │ 2019-03-25 │ 5899 │ ██████████████████████████████████████████████████████████▊                  │
 14. │ 2019-04-01 │ 6043 │ ████████████████████████████████████████████████████████████▍                │
 15. │ 2019-04-08 │ 6098 │ ████████████████████████████████████████████████████████████▊                │
 16. │ 2019-04-15 │ 6196 │ █████████████████████████████████████████████████████████████▊               │
 17. │ 2019-04-22 │ 6486 │ ████████████████████████████████████████████████████████████████▋            │
 18. │ 2019-04-29 │ 6682 │ ██████████████████████████████████████████████████████████████████▋          │
 19. │ 2019-05-06 │ 6739 │ ███████████████████████████████████████████████████████████████████▍         │
 20. │ 2019-05-13 │ 6600 │ ██████████████████████████████████████████████████████████████████           │
 21. │ 2019-05-20 │ 6575 │ █████████████████████████████████████████████████████████████████▋           │
 22. │ 2019-05-27 │ 6786 │ ███████████████████████████████████████████████████████████████████▋         │
 23. │ 2019-06-03 │ 6872 │ ████████████████████████████████████████████████████████████████████▋        │
 24. │ 2019-06-10 │ 7045 │ ██████████████████████████████████████████████████████████████████████▍      │
 25. │ 2019-06-17 │ 7045 │ ██████████████████████████████████████████████████████████████████████▍      │
 26. │ 2019-06-24 │ 6852 │ ████████████████████████████████████████████████████████████████████▌        │
 27. │ 2019-07-01 │ 7248 │ ████████████████████████████████████████████████████████████████████████▍    │
 28. │ 2019-07-08 │ 7284 │ ████████████████████████████████████████████████████████████████████████▋    │
 29. │ 2019-07-15 │ 7142 │ ███████████████████████████████████████████████████████████████████████▍     │
 30. │ 2019-07-22 │ 7108 │ ███████████████████████████████████████████████████████████████████████      │
 31. │ 2019-07-29 │ 7251 │ ████████████████████████████████████████████████████████████████████████▌    │
 32. │ 2019-08-05 │ 7403 │ ██████████████████████████████████████████████████████████████████████████   │
 33. │ 2019-08-12 │ 7457 │ ██████████████████████████████████████████████████████████████████████████▌  │
 34. │ 2019-08-19 │ 7502 │ ███████████████████████████████████████████████████████████████████████████  │
 35. │ 2019-08-26 │ 7540 │ ███████████████████████████████████████████████████████████████████████████▍ │
 36. │ 2019-09-02 │ 7237 │ ████████████████████████████████████████████████████████████████████████▎    │
 37. │ 2019-09-09 │ 7328 │ █████████████████████████████████████████████████████████████████████████▎   │
 38. │ 2019-09-16 │ 5566 │ ███████████████████████████████████████████████████████▋                     │
 39. │ 2019-09-23 │ 7049 │ ██████████████████████████████████████████████████████████████████████▍      │
 40. │ 2019-09-30 │ 6880 │ ████████████████████████████████████████████████████████████████████▋        │
 41. │ 2019-10-07 │ 6518 │ █████████████████████████████████████████████████████████████████▏           │
 42. │ 2019-10-14 │ 6688 │ ██████████████████████████████████████████████████████████████████▊          │
 43. │ 2019-10-21 │ 6667 │ ██████████████████████████████████████████████████████████████████▋          │
 44. │ 2019-10-28 │ 6303 │ ███████████████████████████████████████████████████████████████              │
 45. │ 2019-11-04 │ 6298 │ ██████████████████████████████████████████████████████████████▊              │
 46. │ 2019-11-11 │ 6137 │ █████████████████████████████████████████████████████████████▎               │
 47. │ 2019-11-18 │ 6051 │ ████████████████████████████████████████████████████████████▌                │
 48. │ 2019-11-25 │ 5820 │ ██████████████████████████████████████████████████████████▏                  │
 49. │ 2019-12-02 │ 5942 │ ███████████████████████████████████████████████████████████▍                 │
 50. │ 2019-12-09 │ 4891 │ ████████████████████████████████████████████████▊                            │
 51. │ 2019-12-16 │ 5682 │ ████████████████████████████████████████████████████████▋                    │
 52. │ 2019-12-23 │ 6111 │ █████████████████████████████████████████████████████████████                │
 53. │ 2019-12-30 │ 5870 │ ██████████████████████████████████████████████████████████▋                  │
 54. │ 2020-01-06 │ 5953 │ ███████████████████████████████████████████████████████████▌                 │
 55. │ 2020-01-13 │ 5698 │ ████████████████████████████████████████████████████████▊                    │
 56. │ 2020-01-20 │ 5339 │ █████████████████████████████████████████████████████▍                       │
 57. │ 2020-01-27 │ 5566 │ ███████████████████████████████████████████████████████▋                     │
 58. │ 2020-02-03 │ 5801 │ ██████████████████████████████████████████████████████████                   │
 59. │ 2020-02-10 │ 5692 │ ████████████████████████████████████████████████████████▊                    │
 60. │ 2020-02-17 │ 5912 │ ███████████████████████████████████████████████████████████                  │
 61. │ 2020-02-24 │ 6031 │ ████████████████████████████████████████████████████████████▎                │
 62. │ 2020-03-02 │ 6105 │ █████████████████████████████████████████████████████████████                │
 63. │ 2020-03-09 │ 5823 │ ██████████████████████████████████████████████████████████▏                  │
 64. │ 2020-03-16 │ 4659 │ ██████████████████████████████████████████████▌                              │
 65. │ 2020-03-23 │ 3720 │ █████████████████████████████████████▏                                       │
 66. │ 2020-03-30 │ 1720 │ █████████████████▏                                                           │
 67. │ 2020-04-06 │  849 │ ████████▍                                                                    │
 68. │ 2020-04-13 │  710 │ ███████                                                                      │
 69. │ 2020-04-20 │  725 │ ███████▏                                                                     │
 70. │ 2020-04-27 │  920 │ █████████▏                                                                   │
 71. │ 2020-05-04 │  859 │ ████████▌                                                                    │
 72. │ 2020-05-11 │ 1047 │ ██████████▍                                                                  │
 73. │ 2020-05-18 │ 1135 │ ███████████▎                                                                 │
 74. │ 2020-05-25 │ 1266 │ ████████████▋                                                                │
 75. │ 2020-06-01 │ 1793 │ █████████████████▊                                                           │
 76. │ 2020-06-08 │ 1979 │ ███████████████████▋                                                         │
 77. │ 2020-06-15 │ 2297 │ ██████████████████████▊                                                      │
 78. │ 2020-06-22 │ 2788 │ ███████████████████████████▊                                                 │
 79. │ 2020-06-29 │ 3389 │ █████████████████████████████████▊                                           │
 80. │ 2020-07-06 │ 3545 │ ███████████████████████████████████▍                                         │
 81. │ 2020-07-13 │ 3569 │ ███████████████████████████████████▋                                         │
 82. │ 2020-07-20 │ 3784 │ █████████████████████████████████████▋                                       │
 83. │ 2020-07-27 │ 3960 │ ███████████████████████████████████████▌                                     │
 84. │ 2020-08-03 │ 4323 │ ███████████████████████████████████████████▏                                 │
 85. │ 2020-08-10 │ 4581 │ █████████████████████████████████████████████▋                               │
 86. │ 2020-08-17 │ 4791 │ ███████████████████████████████████████████████▊                             │
 87. │ 2020-08-24 │ 4928 │ █████████████████████████████████████████████████▎                           │
 88. │ 2020-08-31 │ 4687 │ ██████████████████████████████████████████████▋                              │
 89. │ 2020-09-07 │ 4643 │ ██████████████████████████████████████████████▍                              │
 90. │ 2020-09-14 │ 4594 │ █████████████████████████████████████████████▊                               │
 91. │ 2020-09-21 │ 4478 │ ████████████████████████████████████████████▋                                │
 92. │ 2020-09-28 │ 4382 │ ███████████████████████████████████████████▋                                 │
 93. │ 2020-10-05 │ 4261 │ ██████████████████████████████████████████▌                                  │
 94. │ 2020-10-12 │ 4243 │ ██████████████████████████████████████████▍                                  │
 95. │ 2020-10-19 │ 3941 │ ███████████████████████████████████████▍                                     │
 96. │ 2020-10-26 │ 3616 │ ████████████████████████████████████▏                                        │
 97. │ 2020-11-02 │ 3586 │ ███████████████████████████████████▋                                         │
 98. │ 2020-11-09 │ 3403 │ ██████████████████████████████████                                           │
 99. │ 2020-11-16 │ 3336 │ █████████████████████████████████▎                                           │
100. │ 2020-11-23 │ 3230 │ ████████████████████████████████▎                                            │
101. │ 2020-11-30 │ 3183 │ ███████████████████████████████▋                                             │
102. │ 2020-12-07 │ 3285 │ ████████████████████████████████▋                                            │
103. │ 2020-12-14 │ 3367 │ █████████████████████████████████▋                                           │
104. │ 2020-12-21 │ 3748 │ █████████████████████████████████████▍                                       │
105. │ 2020-12-28 │ 3986 │ ███████████████████████████████████████▋                                     │
106. │ 2021-01-04 │ 3906 │ ███████████████████████████████████████                                      │
107. │ 2021-01-11 │ 3425 │ ██████████████████████████████████▎                                          │
108. │ 2021-01-18 │ 3144 │ ███████████████████████████████▍                                             │
109. │ 2021-01-25 │ 3115 │ ███████████████████████████████▏                                             │
110. │ 2021-02-01 │ 3285 │ ████████████████████████████████▋                                            │
111. │ 2021-02-08 │ 3321 │ █████████████████████████████████▏                                           │
112. │ 2021-02-15 │ 3475 │ ██████████████████████████████████▋                                          │
113. │ 2021-02-22 │ 3549 │ ███████████████████████████████████▍                                         │
114. │ 2021-03-01 │ 3755 │ █████████████████████████████████████▌                                       │
115. │ 2021-03-08 │ 3080 │ ██████████████████████████████▋                                              │
116. │ 2021-03-15 │ 3789 │ █████████████████████████████████████▊                                       │
117. │ 2021-03-22 │ 3804 │ ██████████████████████████████████████                                       │
118. │ 2021-03-29 │ 4238 │ ██████████████████████████████████████████▍                                  │
119. │ 2021-04-05 │ 4307 │ ███████████████████████████████████████████                                  │
120. │ 2021-04-12 │ 4225 │ ██████████████████████████████████████████▎                                  │
121. │ 2021-04-19 │ 4391 │ ███████████████████████████████████████████▊                                 │
122. │ 2021-04-26 │ 4868 │ ████████████████████████████████████████████████▋                            │
123. │ 2021-05-03 │ 4977 │ █████████████████████████████████████████████████▋                           │
124. │ 2021-05-10 │ 5164 │ ███████████████████████████████████████████████████▋                         │
125. │ 2021-05-17 │ 4986 │ █████████████████████████████████████████████████▋                           │
126. │ 2021-05-24 │ 5024 │ ██████████████████████████████████████████████████▏                          │
127. │ 2021-05-31 │ 4824 │ ████████████████████████████████████████████████▏                            │
128. │ 2021-06-07 │ 5652 │ ████████████████████████████████████████████████████████▌                    │
129. │ 2021-06-14 │ 5613 │ ████████████████████████████████████████████████████████▏                    │
130. │ 2021-06-21 │ 6061 │ ████████████████████████████████████████████████████████████▌                │
131. │ 2021-06-28 │ 2554 │ █████████████████████████▌                                                   │
     └────────────┴──────┴──────────────────────────────────────────────────────────────────────────────┘
```

### Online Playground {#playground}

Вы можете протестировать другие запросы к этому набору данным с помощью интерактивного ресурса [Online Playground](https://gh-api.clickhouse.tech/play?user=play). Например, [вот так](https://gh-api.clickhouse.tech/play?user=play#U0VMRUNUCiAgICBvcmlnaW4sCiAgICBjb3VudCgpLAogICAgcm91bmQoYXZnKGdlb0Rpc3RhbmNlKGxvbmdpdHVkZV8xLCBsYXRpdHVkZV8xLCBsb25naXR1ZGVfMiwgbGF0aXR1ZGVfMikpKSBBUyBkaXN0YW5jZSwKICAgIGJhcihkaXN0YW5jZSwgMCwgMTAwMDAwMDAsIDEwMCkgQVMgYmFyCkZST00gb3BlbnNreQpXSEVSRSBvcmlnaW4gIT0gJycKR1JPVVAgQlkgb3JpZ2luCk9SREVSIEJZIGNvdW50KCkgREVTQwpMSU1JVCAxMDA=). Однако обратите внимание, что здесь нельзя создавать временные таблицы.

