---
toc_priority: 4
toc_title: S3
---

# Движок таблиц S3 {#table-engine-s3}

Этот движок обеспечивает интеграцию с экосистемой [Amazon S3](https://aws.amazon.com/s3/). Он похож на движок [HDFS](../../../engines/table-engines/special/file.md#table_engines-hdfs), но обеспечивает специфические для S3 возможности.

## Создание таблицы {#creating-a-table}

``` sql
CREATE TABLE s3_engine_table (name String, value UInt32)
    ENGINE = S3(path, [aws_access_key_id, aws_secret_access_key,] format, [compression])
    [SETTINGS ...]
```

**Параметры движка**

-   `path` — URL-адрес бакета с указанием пути к файлу. Поддерживает следующие подстановочные знаки в режиме "только чтение": `*`, `?`, `{abc,def}` и `{N..M}` где `N`, `M` — числа, `'abc'`, `'def'` — строки. Подробнее смотри [ниже](#wildcards-in-path).
-   `format` — [формат](../../../interfaces/formats.md#formats) файла.
-   `aws_access_key_id`, `aws_secret_access_key` - данные пользователя учетной записи [AWS](https://aws.amazon.com/ru/). Вы можете использовать их для аутентификации ваших запросов. Необязательный параметр. Если параметры учетной записи не указаны, то используются данные из конфигурационного файла. Смотрите подробнее [Использование сервиса S3 для хранения данных](../mergetree-family/mergetree.md#table_engine-mergetree-s3).
-   `compression` — тип сжатия. Возможные значения: `none`, `gzip/gz`, `brotli/br`, `xz/LZMA`, `zstd/zst`. Необязательный параметр. Если не указано, то тип сжатия определяется автоматически по расширению файла.

**Пример**

``` sql
CREATE TABLE s3_engine_table (name String, value UInt32) 
    ENGINE=S3('https://storage.yandexcloud.net/my-test-bucket-768/test-data.csv.gz', 'CSV', 'gzip')
    SETTINGS input_format_with_names_use_header = 0;

INSERT INTO s3_engine_table VALUES ('one', 1), ('two', 2), ('three', 3);

SELECT * FROM s3_engine_table LIMIT 2;
```

``` text
┌─name─┬─value─┐
│ one  │     1 │
│ two  │     2 │
└──────┴───────┘
```

## Виртуальные столбцы {#virtual-columns}

-   `_path` — путь к файлу.
-   `_file` — имя файла.

Подробнее про виртуальные столбцы можно прочитать [здесь](../../../engines/table-engines/index.md#table_engines-virtual_columns).

## Детали реализации {#implementation-details}

-   Чтение и запись могут быть параллельными.
-   Поддерживается репликация без копирования данных ([zero-copy](../../../operations/storing-data.md#zero-copy)).
-   Не поддерживаются:
    -   запросы `ALTER` и `SELECT...SAMPLE`,
    -   индексы.

## Символы подстановки  {#wildcards-in-path}

Аргумент `path` может указывать на несколько файлов, используя символы подстановки. Для обработки файл должен существовать и соответствовать всему шаблону пути. Список файлов определяется во время выполнения запроса `SELECT` (не в момент выполнения запроса `CREATE`).

-   `*` — заменяет любое количество любых символов, кроме `/`, включая пустую строку.
-   `?` — заменяет любые одиночные символы.
-   `{some_string, another_string, yet_another_one}` — заменяет любые строки `'some_string', 'another_string', 'yet_another_one'`.
-   `{N..M}` — заменяет любое число от N до M, включая обе границы. N и M могут иметь ведущие нули, например `000..078`.

Конструкции с `{}` аналогичны функции [remote](../../../sql-reference/table-functions/remote.md).

!!! warning "Примечание"
    Если список файлов содержит диапазоны чисел с ведущими нулями, используйте конструкцию с фигурными скобками для каждой цифры отдельно или используйте `?`.

**Пример подстановки 1**

Таблица содержит данные из файлов с именами `file-000.csv`, `file-001.csv`, … , `file-999.csv`:

``` sql
CREATE TABLE big_table (name String, value UInt32)
    ENGINE = S3('https://storage.yandexcloud.net/my-bucket/my_folder/file-{000..999}.csv', 'CSV');
```

**Пример подстановки 2**

Предположим, есть несколько файлов в формате CSV со следующими URL-адресами в S3:

-   'https://storage.yandexcloud.net/my-bucket/some_folder/some_file_1.csv'
-   'https://storage.yandexcloud.net/my-bucket/some_folder/some_file_2.csv'
-   'https://storage.yandexcloud.net/my-bucket/some_folder/some_file_3.csv'
-   'https://storage.yandexcloud.net/my-bucket/another_folder/some_file_1.csv'
-   'https://storage.yandexcloud.net/my-bucket/another_folder/some_file_2.csv'
-   'https://storage.yandexcloud.net/my-bucket/another_folder/some_file_3.csv'

Существует несколько способов создать таблицу, включающую в себя все шесть файлов:

1. Задайте диапазон для суффиксов в названии файла: 

``` sql
CREATE TABLE table_with_range (name String, value UInt32)
    ENGINE = S3('https://storage.yandexcloud.net/my-bucket/{some,another}_folder/some_file_{1..3}', 'CSV');
```

2. Таблица содержит все файлы с префиксом `some_file_` (в каталогах не должно быть других файлов с таким префиксом):

``` sql
CREATE TABLE table_with_question_mark (name String, value UInt32)
    ENGINE = S3('https://storage.yandexcloud.net/my-bucket/{some,another}_folder/some_file_?', 'CSV');
```

3. Таблица содержит все файлы в обоих каталогах (в каталогах не должно быть других файлов, соответствующих формату и схеме, описанным в запросе):

``` sql
CREATE TABLE table_with_asterisk (name String, value UInt32)
    ENGINE = S3('https://storage.yandexcloud.net/my-bucket/{some,another}_folder/*', 'CSV');
```

## Настройки движка S3 {#s3-settings}

Перед выполнением запроса или в конфигурационном файле могут быть установлены следующие настройки:

-   `s3_max_single_part_upload_size` — максимальный размер объекта для загрузки с использованием однокомпонентной загрузки в S3. Значение по умолчанию — `64 Mб`.
-   `s3_min_upload_part_size` — минимальный размер объекта для загрузки при многокомпонентной загрузке в [S3 Multipart upload](https://docs.aws.amazon.com/AmazonS3/latest/dev/uploadobjusingmpu.html). Значение по умолчанию — `512 Mб`.
-   `s3_max_redirects` — максимальное количество разрешенных переадресаций S3. Значение по умолчанию — `10`.
-   `s3_single_read_retries` — максимальное количество попыток запроса при единичном чтении. Значение по умолчанию — `4`.

Соображение безопасности: если злонамеренный пользователь попробует указать произвольные URL-адреса S3, параметр `s3_max_redirects` должен быть установлен в ноль, чтобы избежать атак [SSRF](https://en.wikipedia.org/wiki/Server-side_request_forgery). Как альтернатива, в конфигурации сервера должен быть указан `remote_host_filter`.

## Настройки точки приема запроса {#endpoint-settings}

Для точки приема запроса (которая соответствует точному префиксу URL-адреса) в конфигурационном файле могут быть заданы следующие настройки:

Обязательная настройка:
-   `endpoint` — указывает префикс точки приема запроса.

Необязательные настройки:
-   `access_key_id` и `secret_access_key` — указывают учетные данные для использования с данной точкой приема запроса.
-   `use_environment_credentials` — если `true`, S3-клиент будет пытаться получить учетные данные из переменных среды и метаданных [Amazon EC2](https://ru.wikipedia.org/wiki/Amazon_EC2) для данной точки приема запроса. Значение по умолчанию — `false`.
-   `use_insecure_imds_request` — признак использования менее безопасного соединения при выполнении запроса к IMDS при получении учётных данных из метаданных Amazon EC2. Значение по умолчанию — `false`.
-   `region` — название региона S3.
-   `header` — добавляет указанный HTTP-заголовок к запросу на заданную точку приема запроса. Может быть определен несколько раз.
-   `server_side_encryption_customer_key_base64` — устанавливает необходимые заголовки для доступа к объектам S3 с шифрованием SSE-C.
-   `single_read_retries` — Максимальное количество попыток запроса при единичном чтении. Значение по умолчанию — `4`.

**Пример**

``` xml
<s3>
    <endpoint-name>
        <endpoint>https://storage.yandexcloud.net/my-test-bucket-768/</endpoint>
        <!-- <access_key_id>ACCESS_KEY_ID</access_key_id> -->
        <!-- <secret_access_key>SECRET_ACCESS_KEY</secret_access_key> -->
        <!-- <region>us-west-1</region> -->
        <!-- <use_environment_credentials>false</use_environment_credentials> -->
		<!-- <use_insecure_imds_request>false</use_insecure_imds_request> -->
        <!-- <header>Authorization: Bearer SOME-TOKEN</header> -->
        <!-- <server_side_encryption_customer_key_base64>BASE64-ENCODED-KEY</server_side_encryption_customer_key_base64> -->
        <!-- <single_read_retries>4</single_read_retries> -->
    </endpoint-name>
</s3>
```

**Смотрите также**

-  [Табличная функция s3](../../../sql-reference/table-functions/s3.md)
